/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diadynelement.h"

//static void handle_move_x (DiaObject *obj, gfloat x);
//static void handle_move_y (DiaObject *obj, gfloat y);

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaDynElement *from, DiaDynElement *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *p,
			   DiaHandle *h);

static DiaObjectOps element_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_dyn_element_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_dyn_element_get_type ()
{
  static DiaObjectType element_type =
  {
    "DiaDynElement",
    sizeof (DiaDynElement),
    (DiaInitFunc) dia_dyn_element_init,
    &element_ops,
    dia_base_element_get_type /* parent */
  };

  return &element_type;
}

void
dia_dyn_element_init (DiaDynElement *element)
{
  //gint i;
  //DiaObject *obj;
  //DiaHandle *h;
  
  //obj = DIA_OBJECT (element);
  
  /* call parent */
  //dia_base_element_init (DIA_BASE_ELEMENT (element));
  /* save parent options */
  //if (!parent_ops)
  //parent_ops = obj->ops;
  
  /* overwrite old function declarations */
  //obj->object_type = &element_type;
  //obj->ops = &element_ops;
}

void
dia_dyn_element_create_cp_data (DiaDynElement *elem,
				DiaConnectionPoint *cp)
{
  DiaBaseElementCP *becp;
  DiaObject *obj = DIA_OBJECT (elem);
  //DiaBaseElementDirection dir;
  
  g_return_if_fail (obj != NULL);
  g_return_if_fail (cp != NULL);
  
  if (cp->data)
    g_free (cp->data);
  
  becp = g_new (DiaBaseElementCP, 1);

  if (DIA_BASE_ELEMENT (elem)->size.top == cp->pos.y)
    {
      becp->direction = DIA_BASE_ELEMENT_TL;
      becp->factor = 
	(cp->pos.x -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x) /
	(DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.x -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x);
    }
  else if (DIA_BASE_ELEMENT (elem)->size.right == cp->pos.x)
    {
      becp->direction = DIA_BASE_ELEMENT_TR;
      becp->factor = 
	(cp->pos.y -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y) /
	(DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.y -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y);
    }
  else if (DIA_BASE_ELEMENT (elem)->size.bottom == cp->pos.y)
    {
      becp->direction = DIA_BASE_ELEMENT_BR;
      becp->factor = 
	(cp->pos.x -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x) /
	(DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.x -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x);
    }
  else if (DIA_BASE_ELEMENT (elem)->size.left == cp->pos.x)
    {
      becp->direction = DIA_BASE_ELEMENT_BL;
      becp->factor = 
	(cp->pos.y -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y) /
	(DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.y -
	 DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y);
    }
  else
    {
      g_error ("dia_dyn_element_create_cp_data: "
	       "The CP is not located on the SIZE rectangle");
    }
  
  cp->data = becp;
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaDynElement *from, DiaDynElement *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return TRUE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  return parent_ops->calc_bounding_box (obj);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  return parent_ops->event (obj, event, layer);
}

/* calculate the distance from a point POS to an element and return the
 * distance and set P_ON_LINE to the point of connection.
 */
static gfloat
cp_distance_real (DiaObject *obj, Point *pos,
		  Point *p_on_line)
{
  Point tmp_p_on_line;
  gfloat dist;
  gfloat tmp_dist;
  DiaBaseElementDirection dir;
  
  dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos,
			      &DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos,
			      0.01, pos, p_on_line);
  dir = DIA_BASE_ELEMENT_TL;
  
  tmp_dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos,
				  0.01, pos, &tmp_p_on_line);
  if (tmp_dist < dist)
    {
      dist = tmp_dist;
      *p_on_line = tmp_p_on_line;
      dir = DIA_BASE_ELEMENT_BR;
    }
  
  tmp_dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos,
				  0.01, pos, &tmp_p_on_line);
  if (tmp_dist < dist)
    {
      dist = tmp_dist;
      *p_on_line = tmp_p_on_line;
      dir = DIA_BASE_ELEMENT_BL;
    }
  
  tmp_dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos,
				  0.01, pos, &tmp_p_on_line);
  if (tmp_dist < dist)
    {
      dist = tmp_dist;
      *p_on_line = tmp_p_on_line;
      dir = DIA_BASE_ELEMENT_TR;
    }
  
  return dist;
}


#ifdef 0
/* Find a point on the border of the object and create a connectionpoint
 * there.
 */
static gpointer
cp_distance_inside (DiaObject *obj, DiaHandle *handle, Point *pos,
		    DiaConnectionPoint **cp, Point *cp_pos)
{
  Point start1, end1, start2, end2, intersect;
  gint i;
  DiaObject *hobj;
  gboolean found = FALSE;
  
  g_error ("calc_connection_point_on_element: This method should be updated first!");
  /* FIXME: Add support for DiaBaseElementCP. */
  if (!point_in_rectangle (&obj->bounding_box, pos))
    return NULL;
   
  hobj = handle->object;
  
  for (i = 0; i < hobj->handles->len; i++)
    {
      if (g_ptr_array_index (hobj->handles, i) == handle)
	{
	  start1 = handle->pos;
	  if (i < hobj->handles->len - 1)
	    {
	      end1 = DIA_OBJECT_GET_HANDLE (hobj, i+1)->pos;
	    }
	  else
	    if (i > 0)
	      {
		end1 = DIA_OBJECT_GET_HANDLE (hobj, i-1)->pos;
	      }
	    else
	      {
		end1 = start1;
	      }
	  found = TRUE;
	  break;
	}
    }
  
  /* A start handle and an end handle are defined, now we can calculate
   * the position on the box where the line should connect to the box.
   */
  if (found)
    {      
      Point diff;
      
      diff = *pos;
      //point_add (&diff, &handle->object->position);
      point_sub (&diff, &handle->pos); /* diff is now the difference between
					  the object position and the handle
					  position. */
      point_add (&start1, &diff);
      point_add (&end1, &diff);
      /* positions are now relative to obj! */
      
      /* find the point on the box where the line should connect */
      start2.x = DIA_BASE_ELEMENT (obj)->size.left;
      start2.y = DIA_BASE_ELEMENT (obj)->size.top;
      end2.x = DIA_BASE_ELEMENT (obj)->size.right;
      end2.y = DIA_BASE_ELEMENT (obj)->size.top;
      if (line_line_intersection (&start1, &end1, &start2, &end2, &intersect))
	return dia_object_add_connection_point (obj, intersect.x, intersect.y);
      
      start2.x = DIA_BASE_ELEMENT (obj)->size.right;
      start2.y = DIA_BASE_ELEMENT (obj)->size.top;
      end2.x = DIA_BASE_ELEMENT (obj)->size.right;
      end2.y = DIA_BASE_ELEMENT (obj)->size.bottom;
      if (line_line_intersection (&start1, &end1, &start2, &end2, &intersect))
	return dia_object_add_connection_point (obj, intersect.x, intersect.y);
	
      start2.x = DIA_BASE_ELEMENT (obj)->size.left;
      start2.y = DIA_BASE_ELEMENT (obj)->size.bottom;
      end2.x = DIA_BASE_ELEMENT (obj)->size.right;
      end2.y = DIA_BASE_ELEMENT (obj)->size.bottom;
      if (line_line_intersection (&start1, &end1, &start2, &end2, &intersect))
	return dia_object_add_connection_point (obj, intersect.x, intersect.y);

      start2.x = DIA_BASE_ELEMENT (obj)->size.left;
      start2.y = DIA_BASE_ELEMENT (obj)->size.top;
      end2.x = DIA_BASE_ELEMENT (obj)->size.left;
      end2.y = DIA_BASE_ELEMENT (obj)->size.bottom;
      if (line_line_intersection (&start1, &end1, &start2, &end2, &intersect))
	return dia_object_add_connection_point (obj, intersect.x, intersect.y);
    }
  else
    g_warning ("Could not find handle in owners handle collection!!!");
  
  return NULL;
}
#endif /* 0 */

/* Gives a (newly created) connection point to connect a handle to */
static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  gfloat dist1, dist2;
  Point  pos2;
  
  dist1 = parent_ops->cp_distance (obj, pos, con_pos, cp);
  
  dist2 = cp_distance_real (obj, pos, &pos2);
    
  if ((dist1 < DIA_CONNECTION_POINT_TRESHOLD) || (dist1 < dist2))
    {
      return dist1;
    }
  else
    {
      *con_pos = pos2;
      *cp = NULL;
      return dist2;
    }
}

static DiaConnectionPoint*
cp_connect (DiaObject *object, DiaHandle *h, Point *pos)
{
  DiaConnectionPoint *cp;
  
  cp = parent_ops->cp_connect (object, h, pos);
  if (cp)
    return cp;
  
  if (pos)
    {
      cp = dia_object_add_connection_point (object, pos->x, pos->y);
      dia_dyn_element_create_cp_data (DIA_DYN_ELEMENT (object), cp);
    }
  else
    {
      Point p_on_line;
      
      dia_object_cp_distance (object, &h->pos, &p_on_line, &cp);
      if (!cp)
	cp = dia_object_add_connection_point (object, p_on_line.x,
					      p_on_line.y);
      dia_dyn_element_create_cp_data (DIA_DYN_ELEMENT (object), cp);
    }

  if (!dia_handle_connect (h, cp))
    {
      g_warning ("DiaDynElement::cp_connect: Connection could not be established");
      dia_connection_point_free (cp);
      cp = NULL;
    }
  
  return cp;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			     DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);

  if (cp->connected == NULL)
    {
      if (cp->data)
	g_free (cp->data);
      dia_connection_point_free (cp);
    }
}



