/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diadynline.h"
#include "math.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaDynLine* from, DiaDynLine *to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps line_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_dyn_line_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_dyn_line_get_type ()
{
  static DiaObjectType line_type =
  {
    "DiaDynLine",
    sizeof (DiaDynLine),
    (DiaInitFunc) dia_dyn_line_init,
    &line_ops,
    dia_base_line_get_type
  };
  return &line_type;
}

void
dia_dyn_line_init (DiaDynLine *dyn_line)
{
  //DiaObject *object = DIA_OBJECT (dyn_line);
  
  /* call parent */
  //dia_base_line_init (DIA_BASE_LINE (dyn_line));
  /* save parent options */
  //if (!parent_ops)
  //parent_ops = object->ops; 
 
  /* overwrite old function declarations */
  //object->object_type = &line_type;
  //object->ops = &line_ops;
}

DiaObject*
dia_dyn_line_new (Point *pos)
{
  DiaObject *new_line;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_line = dia_object_create (dia_dyn_line_get_type ());
  //g_new (DiaDynLine, 1);
  
  //dia_dyn_line_init (new_line);
  dia_object_move (new_line, pos->x, pos->y);
  
  return new_line;
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaDynLine *from, DiaDynLine *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  return parent_ops->calc_bounding_box (obj);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  return parent_ops->event (obj, event, layer);
}

/* in:
 *  obj:
 *  pos: position of the handle that should be connected
 * out:
 *  cp:  used to call distance's parent
 *  p_on_line: will contain the point on the line that should be the position
 *  of a (new) connectionpoint
 *  index: number of the handle that starts the line segment P_ON_LINE
 *  is on.
 */
static gfloat
cp_distance_real (DiaObject *obj, Point *pos, DiaConnectionPoint **cp,
		  Point *p_on_line, gint *index)
{
  //  DiaConnectionPoint *cp = NULL;
  Point p;
  gfloat min_dist;
  gfloat dist;
  gint i;
  
  min_dist = parent_ops->cp_distance (obj, pos, p_on_line, cp);

  if (min_dist < DIA_CONNECTION_POINT_TRESHOLD)
    return min_dist;
  
  for (i = 0; i < obj->handles->len - 1; i++)
    {
      dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, i)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, i+1)->pos,
				  DIA_BASE_LINE (obj)->line_width,
				  pos, &p);
      if (dist < min_dist)
	{
	  min_dist = dist;
	  *p_on_line = p;
	  *cp = NULL;
	  *index = i;
	}
    }
  return min_dist;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos,  Point *con_pos,
	     DiaConnectionPoint **cp)
{
  //  gfloat dist1, dist2;
  //Point  pos2;
  //DiaConnectionPoint *cp2
  gint i;
  
  //dist1 = parent_ops->cp_distance (obj, pos, con_pos, cp);
  //dist2 = cp_distance_real (obj, pos, &cp2, &pos2, &i);

  return cp_distance_real (obj, pos, cp, con_pos, &i);

/*   if (dist1 < dist2) */
/*     { */
/*       return dist1; */
/*     } */
/*   else */
/*     { */
/*       *con_pos = pos2; */
/*       *cp = cp2; */
/*       return dist2; */
/*     } */
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  DiaConnectionPoint *cp = NULL;
  Point p;
  gint index;
  DiaBaseLineCP *blcp;
  
  //cp = parent_ops->cp_connect (obj, h, pos);
  
  //if (cp)
  //return cp;
  
  cp_distance_real (obj, &h->pos, &cp, &p, &index);
  
  if (index == -1)
    return NULL;
   
  if (cp)
    {
      dia_handle_connect (h, cp);
      return cp;
    }
  
  cp = dia_object_add_connection_point (obj, p.x, p.y);
    
  if (!dia_handle_connect (h, cp))
    {
      dia_connection_point_free (cp);
      cp = NULL;
    }
  else
    {
      blcp = g_new (DiaBaseLineCP, 1);
      
      blcp->handle = DIA_OBJECT_GET_HANDLE (obj, index);
      blcp->factor =
	dia_base_line_cp_calc_factor (&blcp->handle->pos,
				      &DIA_OBJECT_GET_HANDLE (obj,
							      index + 1)->pos,
				      &cp->pos);
      cp->data = blcp;
    }
  
  return cp;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			     DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
  
  if (cp->connected == NULL)
    {
      g_free (cp->data);
      dia_connection_point_free (cp);
    }
}





