/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diaelement.h"
#include "diagraph.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaElement *from, DiaElement *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps element_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

#define parent_ops (dia_element_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_element_get_type ()
{
  static DiaObjectType element_type =
  {
    "DiaElement",
    sizeof (DiaElement),
    (DiaInitFunc) dia_element_init,
    &element_ops,
    dia_dyn_element_get_type
  };
  return &element_type;
}

void
dia_element_init (DiaElement *element)
{
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaElement *from, DiaElement *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return TRUE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  return parent_ops->calc_bounding_box (obj);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result;
  
  switch (event->type)
    {
    default:
      result = parent_ops->event (obj, event, layer);
    }
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  DiaConnectionPoint *cp = NULL;

  if (dia_object_check_type (h->object, dia_graph_get_type ()))
    {
      if (pos)
	{
	  //return parent_ops->cp_connect (obj, h, pos);
	  /* Do it self: this avoids the possibility that more than one
	   * handle is connected to a connection point. */
	  cp = dia_object_add_connection_point (obj, pos->x, pos->y);
	  dia_dyn_element_create_cp_data (DIA_DYN_ELEMENT (obj), cp);
	  if (!dia_handle_connect (h, cp))
	    dia_connection_point_free (cp);
	}
      else
	g_error ("DiaElement::cp_connect: a position should be given "
		 "to create a connection point!");
      
      //return parent_ops->cp_connect (obj, h, pos);
    }
  
    return cp;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}



