/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gdk/gdkkeysyms.h>
#include "diaentry.h"
#include "math.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void copy (DiaEntry* from, DiaEntry **to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static DiaConnectionPoint* connection_point_request (DiaObject *obj,
						     DiaHandle *handle,
						     Point *pos);
static void connection_point_disconnect (DiaObject *obj,
					 DiaConnectionPoint *p,
					 DiaHandle *h);

DiaColor dia_entry_selection_bg_color = { 0.0, 0.0, 1.0 };
DiaColor dia_entry_selection_fg_color = { 1.0, 1.0, 1.0 };

static DiaObjectType entry_type =
{
  "DiaEntry",
  dia_base_text_get_type
};

static DiaObjectOps entry_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  connection_point_request,
  connection_point_disconnect
};

static DiaObjectOps *parent_ops = NULL;

DiaObjectType*
dia_entry_get_type ()
{
  return &entry_type;
}

void
dia_entry_init (DiaEntry *entry)
{
  DiaObject *object = DIA_OBJECT (entry);
  
  /* call parent */
  dia_base_text_init (DIA_BASE_TEXT (entry));
  /* save parent options */
  if (!parent_ops)
    {
      parent_ops = object->ops; 
    }
  
  /* overwrite old function declarations */
  object->object_type = &entry_type;
  object->ops = &entry_ops;

  /* add our own declarations: */
  //entry->has_selection = FALSE;
  entry->has_selection = TRUE;
  entry->selection_start = 0;
}

DiaObject*
dia_entry_new (Point *pos)
{
  DiaEntry *new_entry;
  
  g_return_if_fail (pos != NULL);
  
  new_entry = g_new (DiaEntry, 1);
  
  dia_entry_init (new_entry);
  DIA_OBJECT (new_entry)->position = *pos;
  
  return DIA_OBJECT (new_entry);
}

void
calc_bounding_box (DiaObject *obj)
{
  parent_ops->calc_bounding_box (obj);
}

void
dia_entry_draw_selection (DiaEntry* entry, DiaRenderer* renderer,
			  gfloat text_bottom)
{
  DiaObject *obj;
  DiaBaseText *text;
  Point p1, p2;
  
  obj = DIA_OBJECT (entry);
  text = DIA_BASE_TEXT (entry);
  
  if (!text->editable || !renderer->is_interactive)
    return;
  
  p1.x = renderer->interactive_ops->get_text_width (renderer,
						    text->text->str,
						    text->cursor_pos)
    + obj->bounding_box.left;
  
  p1.y = obj->bounding_box.top;
  p2.y = obj->bounding_box.bottom;

  /* draw a selection rectangle if some text is selected: */
  if (entry->has_selection)
    {
      gfloat tmp;
      gchar *s;
      gint len;
      Point pos;
      
      p2.x = 
	renderer->interactive_ops->get_text_width (renderer,
						   text->text->str,
						   entry->selection_start)
	+ obj->bounding_box.left;
      tmp = p1.x;
      p1.x = MIN (p1.x, p2.x);
      p2.x = MAX (p2.x, tmp);
      renderer->ops->fill_rect (renderer, &p1, &p2,
				&dia_entry_selection_bg_color);
      /* Print the text in another color: */
      pos.x = p1.x;
      pos.y = text_bottom;
      
      len = ABS (text->cursor_pos - entry->selection_start);
      s = g_memdup (&text->text->str[MIN (text->cursor_pos,
					  entry->selection_start)],
		    len + 1);
      s[len] = '\0';
      
      if (s[0] != '\0')
	renderer->ops->draw_string (renderer, s, &pos,
				    &dia_entry_selection_fg_color);
      g_free (s);
      //p1.x = tmp;
    }
}

/* "events" */
static void
destroy (DiaObject *object)
{
  //DiaBaseText *text = DIA_BASE_TEXT (object);
  
  //g_string_free (DIA_BASE_TEXT (object)->text, TRUE);
  /* text->font is a pointer to a font entry that is static! */

  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  
  parent_ops->draw (obj, renderer);
  
  /* draw cursor if text has focus */
  if (DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_FOCUSED))
    {
      dia_entry_draw_selection (DIA_ENTRY (obj), renderer,
				obj->bounding_box.bottom
				- DIA_BASE_TEXT (obj)->descent);
      dia_base_text_draw_cursor (DIA_BASE_TEXT (obj), renderer,
				 obj->bounding_box.bottom);
    }
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
copy (DiaEntry *from, DiaEntry **to)
{
  g_return_if_fail (from != NULL);
  g_return_if_fail (to != NULL);
  g_return_if_fail (dia_object_check_type (DIA_OBJECT (from),
					   dia_entry_get_type ()));

  if (*to == NULL)
    {
      *to = DIA_ENTRY (dia_entry_new (&DIA_OBJECT (from)->position));
    }
  
  parent_ops->copy (DIA_OBJECT (from), (DiaObject**) to);

  (*to)->has_selection = from->has_selection;
  (*to)->selection_start = from->selection_start;
}

static gint
is_empty (DiaObject* obj)
{
  return parent_ops->is_empty (obj);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  DiaBaseText *text;
  DiaEntry *entry;
  
  g_return_if_fail (obj != NULL);
  g_return_if_fail (event != NULL);
  g_return_if_fail (layer != NULL);
  //g_return_if_fail (dia_object_check_type (obj, dia_dyn_line_get_type ()));
  
  text = DIA_BASE_TEXT (obj);
  entry = DIA_ENTRY (obj);
  
  if (!text->editable)
    return FALSE; /* Fast way... */
  
  switch (event->type)
    {
    case DIA_BUTTON_PRESS:
      if ((event->button.button == 1)
	  && !event->button.handle)
	{
	  Point p;
	  p = event->button.pos;
	  point_sub (&p, &obj->position);
	  /* is pointer inside or outside the bounding box (grab or ungrab)*/
	  if (point_in_rectangle (&obj->bounding_box, &p))
	    {
	      if (!entry->has_selection &&
		  DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_FOCUSED) &&
		  (event->button.modifier & DIA_SHIFT_MASK))
		{
		  entry->selection_start = text->cursor_pos;
		  entry->has_selection = TRUE;		  
		}
	      else
		if (!(event->key.modifier & DIA_SHIFT_MASK))
		  entry->has_selection = FALSE;
	      result = parent_ops->event (obj, event, layer);	      
	    }
	  else
	    {
	      if (entry->has_selection &&
		  !(event->button.modifier & DIA_SHIFT_MASK))
		entry->has_selection = FALSE;
	      result = parent_ops->event (obj, event, layer);	      
	    }
	}
      break;
    case DIA_MOTION:
      //g_message ("motioning!");
      if (DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_FOCUSED) &&
	  (event->motion.modifier & DIA_BUTTON1_MASK))
	{
	  if (!entry->has_selection)
	    {
	      //g_message ("setting selection!");
	      entry->selection_start = text->cursor_pos;
	      entry->has_selection = TRUE;
	    }
	  text->cursor = event->motion.pos.x - obj->position.x;
	  text->cursor_update = TRUE;
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  //g_message ("updating selection!(%x, %x)", obj->flags, text->has_selection);
	  parent_ops->event (obj, event, layer);
	  result = TRUE;
	}
      else
	result = parent_ops->event (obj, event, layer);
      break;
    case DIA_KEY_PRESS:
      switch (event->key.keyval)
	{
	case GDK_Left:
	case GDK_KP_Left:
	  if (text->cursor_pos > 0)
	    {
	      if (!entry->has_selection &&
		  (event->key.modifier & DIA_SHIFT_MASK))
		{
		  entry->selection_start = text->cursor_pos;
		  entry->has_selection = TRUE;		  
		}
	      else
		if (!(event->key.modifier & DIA_SHIFT_MASK))
		  entry->has_selection = FALSE;
	      result = parent_ops->event (obj, event, layer);
	    }
	  break;
	case GDK_Right:
	case GDK_KP_Right:
	  if (text->cursor_pos < text->text->len)
	    {
	      if (!entry->has_selection &&
		  (event->key.modifier & DIA_SHIFT_MASK))
		{
		  entry->selection_start = text->cursor_pos;
		  entry->has_selection = TRUE;		  
		}
	      else
		if (!(event->key.modifier & DIA_SHIFT_MASK))
		  entry->has_selection = FALSE;
	      result = parent_ops->event (obj, event, layer);
	    }
	  break;
	case GDK_BackSpace:
	case GDK_Delete:
	  if (entry->has_selection)
	    {
	      gint len = ABS (text->cursor_pos - entry->selection_start);
	      text->cursor_pos = MIN (text->cursor_pos,
				      entry->selection_start);
	      text->text = g_string_erase (text->text, text->cursor_pos,
					   len);
	      entry->has_selection = FALSE;
	      obj->update_box = obj->bounding_box;
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW|
				      DIA_REQUEST_EXTENTS);     
	      obj->ops->calc_bounding_box (obj);
	      result = TRUE;
	    }
	  else
	    result = parent_ops->event (obj, event, layer);
	  break;
	default:
	  /* only add "normal" characters */
	  if ((event->key.length > 0) && (event->key.string[0] > 31))
	    {
	      /* overwrite selection by a character: */
	      if (entry->has_selection)
		{
		  gint len = ABS (text->cursor_pos - entry->selection_start);
		  text->cursor_pos = MIN (text->cursor_pos,
					  entry->selection_start);
		  text->text = g_string_erase (text->text, text->cursor_pos,
					       len);
		  
		  entry->has_selection = FALSE;
		}
	    }
	  result = parent_ops->event (obj, event, layer);
	}
      result = TRUE;
      break;
    default:
      result = parent_ops->event (obj, event, layer);
      break;
    }

  if (!result)
    return parent_ops->event (obj, event, layer);
  else
    return result;
}

/* Gives a (newly created) connection point to connect a handle to */
static DiaConnectionPoint*
connection_point_request (DiaObject *obj, DiaHandle *handle, Point *pos)
{
  return parent_ops->request_connection_point (obj, handle, pos);
}

static void
connection_point_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			     DiaHandle *h)
{
  parent_ops->disconnect_connection_point (obj, cp, h);
}





