/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* diaevent.h
 * ----------
 * Event handler routiunes for the canvas objects.
 * These look a lot like the types used by GTK.
 *
 */

#ifndef __DIA_EVENT_H__
#define __DIA_EVENT_H__

#include <glib.h>
#include <diacanvas/diadefs.h>
#include <diacanvas/geometry.h>

DIA_OPEN

typedef struct _DiaEventAny DiaEventAny;
typedef struct _DiaEventMotion DiaEventMotion;
typedef struct _DiaEventButton DiaEventButton;
typedef struct _DiaEventKey DiaEventKey;
typedef struct _DiaEventCrossing DiaEventCrossing;
typedef struct _DiaEventHandleMotion DiaEventHandleMotion;
typedef struct _DiaEventUpdateCP DiaEventUpdateCP;

typedef union _DiaEvent DiaEvent;

#include <diacanvas/diaobject.h>

typedef enum
{
  DIA_NONE,
  DIA_MOTION,         /* A pointer move is performed */
  DIA_BUTTON_PRESS,   /* A button is pressed */
  DIA_2BUTTON_PRESS,  /* double click event */
  DIA_3BUTTON_PRESS,  /* triple click event */
  DIA_BUTTON_RELEASE, /* The mouse button is released */
  DIA_PLACE,          /* This event is fired as soon as the object is connected
		       * to the canvas. It uses the "button" type. */
  DIA_KEY_PRESS,      /* key press */
  DIA_KEY_RELEASE,
  DIA_HANDLE_MOTION,  /* A explicit move on a handle. For example if a
		       * connection point is moved: this causes the handle
		       * to move too. */
  DIA_UPDATE_CP       /* A request to update the position of a connection
		       * point. */
} DiaEventType;

#define DIA_EVENT_IS_BUTTON(ev) ((ev->type == DIA_BUTTON_PRESS) \
				 || (ev->type == DIA_2BUTTON_PRESS) \
				 || (ev->type == DIA_3BUTTON_PRESS) \
				 || (ev->type == DIA_BUTTON_RELEASE) \
				 || (ev->type == DIA_PLACE))
/* Modifiers for the event-types.
 * (For now this is a duplicate of the GDK event mask enumeration.)
 */
typedef enum
{
  DIA_SHIFT_MASK    = 1 << 0,
  DIA_LOCK_MASK	    = 1 << 1,
  DIA_CONTROL_MASK  = 1 << 2,
  DIA_MOD1_MASK	    = 1 << 3,
  DIA_MOD2_MASK	    = 1 << 4,
  DIA_MOD3_MASK	    = 1 << 5,
  DIA_MOD4_MASK	    = 1 << 6,
  DIA_MOD5_MASK	    = 1 << 7,
  DIA_BUTTON1_MASK  = 1 << 8,
  DIA_BUTTON2_MASK  = 1 << 9,
  DIA_BUTTON3_MASK  = 1 << 10,
  DIA_BUTTON4_MASK  = 1 << 11,
  DIA_BUTTON5_MASK  = 1 << 12,
  DIA_RELEASE_MASK  = 1 << 13,
  DIA_MODIFIER_MASK = 0x3fff
} DiaModifierType;

struct _DiaEventAny
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
};

/* used by: DIA_MOTION. */
struct _DiaEventMotion
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
  Point pos;
  Point snap; /* point if "snap to grid" is on (use this for move, use POS for
	       * connection points. */
  gdouble pressure;
};

/* used by: DIA_BUTTON_PRESS,
 *          DIA_2BUTTON_PRESS,
 *          DIA_3BUTTON_PRESS,
 *          DIA_BUTTON_RELEASE and
 *          DIA_PLACE.
 */
struct _DiaEventButton
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
  Point pos;
  Point snap; /* point if "snap to grid" is on (use this for move, use POS for
	       * connection points. */
  gdouble pressure; /* will be initialized by dia_event_convert_gdk_event */
  guint button;
  DiaHandle *handle; /* if a button press action occures and the pointer is
		      * on a handle, this pointer will point to that handle. */
};

/* used by: DIA_KEY_PRESS and DIA_KEY_RELEASE. */
struct _DiaEventKey
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
  guint keyval;
  gint length;
  gchar *string;
};

/* used by: DIA_HANDLE_MOTION. */
struct _DiaEventHandleMotion
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
  Point pos; /* position of ConnectionPoint on canvas (=absolute) */
  DiaHandle *handle; /* pointer to a DiaHandle. This is the handle that has
		      to be moved. */
};

/* used by: DIA_UPDATE_CP. */
struct _DiaEventUpdateCP
{
  DiaEventType type;
  guint32 time;
  DiaModifierType modifier;
  DiaConnectionPoint *cp; /* A pointer to the Connection point that has to
			   * be updated. */
};

union _DiaEvent
{
  DiaEventType         type;
  DiaEventAny          any;
  DiaEventMotion       motion;
  DiaEventButton       button;
  DiaEventKey          key;
  DiaEventHandleMotion handle_motion;
  DiaEventUpdateCP     update_cp;
};

DIA_CLOSE

#endif /* __DIA_EVENT_H__ */








