/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diagraph.h"
#include "diaelement.h"
#include "math.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaGraph* from, DiaGraph *to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps graph_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

#define parent_ops (dia_graph_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_graph_get_type ()
{
  static DiaObjectType graph_type =
  {
    "DiaGraph",
    sizeof (DiaGraph),
    (DiaInitFunc) dia_graph_init,
    &graph_ops,
    dia_multi_line_get_type
  };
  return &graph_type;
}

void
dia_graph_init (DiaGraph *graph)
{
  graph->centric = TRUE;
 
  DIA_OBJECT_GET_HANDLE (graph, 0)->keep_on_move = TRUE;
  DIA_OBJECT_GET_HANDLE (graph, 1)->keep_on_move = TRUE;
}

DiaObject*
dia_graph_new (Point **pos, DiaElement *elem1, DiaElement *elem2)
{
  DiaObject *new_graph;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_graph = dia_object_create (dia_graph_get_type ());
  
  //dia_object_move (new_graph, pos->x, pos->y);
  /* FIXME: implement */

  return new_graph;
}

void
dia_graph_center_handle (DiaGraph *graph, DiaHandle *handle, DiaElement *elem)
{
  Point pos;

  if (graph->centric)
    {
      pos.x = (DIA_OBJECT (elem)->bounding_box.left
	       + DIA_OBJECT (elem)->bounding_box.right) / 2.0;
      pos.y = (DIA_OBJECT (elem)->bounding_box.top
	       + DIA_OBJECT (elem)->bounding_box.bottom) / 2.0;
      handle->pos = pos;
    }
}

gboolean
dia_graph_connect_element (DiaGraph *graph, DiaHandle *handle,
			   DiaElement *elem)
{
  g_return_val_if_fail (graph != NULL, FALSE);
  g_return_val_if_fail (handle != NULL, FALSE);
  g_return_val_if_fail (handle->object == DIA_OBJECT (graph), FALSE);
  
  dia_graph_center_handle (graph, handle, elem);
  
  return dia_object_cp_connect (DIA_OBJECT (elem), handle, NULL) ? TRUE : FALSE;
}

DiaHandle*
dia_graph_get_next_handle (DiaGraph *graph, DiaHandle *handle)
{
  g_return_val_if_fail (graph != NULL, NULL);
  g_return_val_if_fail (handle != NULL, NULL);
  g_return_val_if_fail (DIA_OBJECT (graph)->handles->len > 1, NULL);
  
  if (DIA_OBJECT_GET_HANDLE (graph, 0) == handle)
    return DIA_OBJECT_GET_HANDLE (graph, 1);
  else if (DIA_OBJECT_GET_HANDLE (graph, DIA_OBJECT (graph)->handles->len - 1)
	   == handle)
    return DIA_OBJECT_GET_HANDLE (graph, DIA_OBJECT (graph)->handles->len - 2);
  else
    return NULL;
}


/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaGraph *from, DiaGraph *to)
{
  to->centric = from->centric;
  
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  return parent_ops->calc_bounding_box (obj);
}

DiaElement*
dia_graph_find_dia_element (DiaGraph *graph, DiaLayer *layer, Point *pos)
{
  return DIA_ELEMENT (dia_layer_find_closest_object_by_type (layer, pos,
							     DIA_CONNECTION_POINT_TRESHOLD, dia_element_get_type ()));
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  DiaElement *elem;
  
  switch (event->type)
    {
      /* Check if the object is placed inside an element, else request a
	 destroy. */
    case DIA_PLACE:
      ///g_message ("DIA_PLACE signal");
      elem = dia_graph_find_dia_element (DIA_GRAPH (obj), layer,
					 &event->button.pos);
      if (elem)
	{
	  //g_message ("placing graph");
	  
/* 	  if (DIA_GRAPH (obj)->centric) */
/* 	    { */
/* 	      pos.x = (DIA_OBJECT (elem)->bounding_box.left */
/* 		       + DIA_OBJECT (elem)->bounding_box.right) / 2.0; */
/* 	      pos.y = (DIA_OBJECT (elem)->bounding_box.top */
/* 		       + DIA_OBJECT (elem)->bounding_box.bottom) / 2.0; */
/* 	      DIA_OBJECT_GET_HANDLE (obj, 0)->pos = pos; */
/* 	    } */
	  DIA_OBJECT_HANDLE_FIRST (obj)->could_be_connected = TRUE;
	  DIA_GRAPH (obj)->stored_style = DIA_BASE_LINE (obj)->line_style;
	  DIA_GRAPH (obj)->elem = elem;
	  DIA_BASE_LINE (obj)->line_style = DIA_LINE_STYLE_DOTTED;
	  DIA_OBJECT_SET_FLAGS (obj, DIA_GRAPH_IS_NEW);
	  parent_ops->event (obj, event, layer);
	}
      else
	{
	  //g_message ("wrong placement of graph", obj->flags);
	  dia_object_add_update (obj, &obj->bounding_box);
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_DESTROY
				  | DIA_REQUEST_REDRAW
				  | DIA_REQUEST_UNGRAB);
	}
      result = TRUE;
      break;
    case DIA_MOTION:
      result = parent_ops->event (obj, event, layer);

      if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_MOVE_HANDLE))
	{
	  if (DIA_BASE_LINE (obj)->handle_moving
	      == DIA_OBJECT_HANDLE_FIRST (obj)
	      || DIA_BASE_LINE (obj)->handle_moving
	      == DIA_OBJECT_HANDLE_LAST (obj))
	    {
	      if (dia_graph_find_dia_element (DIA_GRAPH (obj), layer,
					  &event->button.pos))
		{
		  DIA_BASE_LINE (obj)->handle_moving->could_be_connected = TRUE;
		}
	      else
		{
		  DIA_BASE_LINE (obj)->handle_moving->could_be_connected = FALSE;
		}
	    }
	  if (DIA_OBJECT_HANDLE_FIRST (obj)->connected_to)
	    dia_graph_center_handle (DIA_GRAPH (obj),
				     DIA_OBJECT_GET_HANDLE (obj, 0),
				     DIA_ELEMENT (DIA_OBJECT_GET_HANDLE (obj, 0)->connected_to->object));
	  if (DIA_OBJECT_HANDLE_LAST (obj)->connected_to)
	    dia_graph_center_handle (DIA_GRAPH (obj),
				     DIA_OBJECT_HANDLE_LAST (obj),
				     DIA_ELEMENT (DIA_OBJECT_HANDLE_LAST (obj)->connected_to->object));
	    	  
	  dia_object_calc_bounding_box_update (obj);
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				  | DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_UPDATE_HANDLES);
	  
	}
      break;
    case DIA_BUTTON_RELEASE:
      switch (event->button.button)
	{
	case 1:
	  result = TRUE;
	  if (DIA_OBJECT_IS_SET (obj, DIA_GRAPH_IS_NEW))
	    {
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	      
	      elem = dia_graph_find_dia_element (DIA_GRAPH (obj), layer,
						 &event->button.pos);
	      if (elem && (elem != DIA_GRAPH (obj)->elem))
		{
		  DIA_BASE_LINE (obj)->line_style = DIA_GRAPH (obj)->stored_style;
		  /* center the handle: */
		  dia_graph_center_handle (DIA_GRAPH (obj),
					   DIA_OBJECT_GET_HANDLE (obj, 0),
					   DIA_GRAPH (obj)->elem);
		  dia_graph_center_handle (DIA_GRAPH (obj),
					   DIA_BASE_LINE (obj)->handle_moving,
					   elem);
		  /* connect to elements: */
		  dia_object_cp_connect (DIA_OBJECT (DIA_GRAPH (obj)->elem),
					 DIA_OBJECT_GET_HANDLE (obj, 0),
					 NULL);
		  dia_object_cp_connect (DIA_OBJECT (elem),
					 DIA_BASE_LINE (obj)->handle_moving,
					 NULL);

		  dia_object_calc_bounding_box_update (obj);
		}
	      else
		{
		  dia_object_add_update (obj, &obj->bounding_box);
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_DESTROY);
		}
	      DIA_OBJECT_UNSET_FLAGS (obj, DIA_GRAPH_IS_NEW);
	    } /* (DIA_OBJECT_IS_SET (obj, DIA_GRAPH_IS_NEW)) */
	  //else if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_MOVE))
	  //{
	      /* destroy graph if one end is unconnected */
	  //  if (!DIA_OBJECT_HANDLE_FIRST (obj)->connected_to
	  //  || !DIA_OBJECT_HANDLE_LAST (obj)->connected_to)
	  //{
	  //  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_DESTROY);
	  //}
	  //}
	  else if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_MOVE_HANDLE)
		   && (DIA_BASE_LINE (obj)->handle_moving
		       == DIA_OBJECT_HANDLE_FIRST (obj)
		       || DIA_BASE_LINE (obj)->handle_moving
		       == DIA_OBJECT_HANDLE_LAST (obj)))
	    {
	      elem = dia_graph_find_dia_element (DIA_GRAPH (obj), layer,
						 &event->button.pos);
	      g_message ("DiaGraph: DIA_BASE_LINE_MOVE_HANDLE");
	      
	      if (elem)
		{
		  g_message ("Connecting element!");
		  
		  dia_graph_connect_element (DIA_GRAPH (obj),
					     DIA_BASE_LINE (obj)->handle_moving,
					     elem);
		  dia_object_calc_bounding_box_update (obj);
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
					  | DIA_REQUEST_EXTENTS);
		}
	      else
		{
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_DESTROY);
		}
	    }
	  result |= parent_ops->event (obj, event, layer);
	  break;
	default:
	  result = parent_ops->event (obj, event, layer);
	  break;
	} /* button number switch */
      break;
    case DIA_HANDLE_MOTION:
      /* get the object the graph is connected to: */
      if (event->handle_motion.handle->connected_to)
	{
	  DiaElement *elem;
	  elem = DIA_ELEMENT (event->handle_motion.handle->connected_to->object);
	  if (DIA_ELEMENT (DIA_OBJECT_HANDLE_FIRST (obj)->connected_to))
	      dia_graph_center_handle (DIA_GRAPH (obj),
				       DIA_OBJECT_HANDLE_FIRST (obj),
				       DIA_ELEMENT (DIA_OBJECT_HANDLE_FIRST (obj)->connected_to->object));

	  if (DIA_ELEMENT (DIA_OBJECT_HANDLE_LAST (obj)->connected_to))
	      dia_graph_center_handle (DIA_GRAPH (obj),
				       DIA_OBJECT_HANDLE_LAST (obj),
				       DIA_ELEMENT (DIA_OBJECT_HANDLE_LAST (obj)->connected_to->object));
	    	  
	  dia_object_calc_bounding_box_update (obj);
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				  | DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_UPDATE_HANDLES);
	  //DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	}
      break;
    default:
      result = parent_ops->event (obj, event, layer);
      break;
    }
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  if (dia_object_check_type (h->object, dia_graph_get_type ())) 
    return parent_ops->cp_connect (obj, h, pos);
  else
    return NULL;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}





