/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <math.h>

#include "diagrid.h"

void
dia_grid_init (DiaGrid *grid)
{
  grid->visible = TRUE;
  grid->snap = FALSE;
  grid->width_x = 1.0;
  grid->width_y = 1.0;
  grid->point_grid = TRUE;
  grid->interval_x = 1;
  grid->interval_y = 1;
  grid->color.red = 0.6;
  grid->color.green = 0.6;
  grid->color.blue = 0.6;
  
  grid->data = NULL;
}

void
dia_grid_draw (DiaDisplay *ddisp, Rectangle *update_box)
{
  DiaGrid *grid;
  DiaRenderer *renderer;
  gfloat offset_x, offset_y;
  gint x, y;
  GdkGC *gc;
  GdkColor gdkcolor;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);
  g_return_if_fail (update_box != NULL);
  
  grid = &ddisp->grid;
  gc = ddisp->renderer->render_gc;

  dia_color_convert (&grid->color, &gdkcolor);
  gdk_gc_set_foreground (gc, &gdkcolor);

  /* return if grid becomes to small or no grid is visible
   * (the background becomes to grey). FAST ROUTE!*/
  if (!grid->visible
      || (dia_display_transform_length (ddisp, grid->width_x * grid->interval_x) < 3)
      || (dia_display_transform_length (ddisp, grid->width_y * grid->interval_y) < 3))
    return;
  
  renderer = (DiaRenderer*)ddisp->renderer;
  
  /* some init stuff */
  renderer->ops->set_linewidth (renderer, 0.0);
  renderer->ops->set_linestyle (renderer, DIA_LINE_STYLE_SOLID);
  renderer->ops->set_origin (renderer, 0.0, 0.0);

  /* find next point where grid has to be drawn */
  offset_x = floor (update_box->left / grid->width_x) * grid->width_x;
  offset_y = floor (update_box->top / grid->width_y) * grid->width_y;
    
  if (grid->point_grid)
    {
      gfloat offset_y_orig = offset_y;
    
      while (offset_x <= update_box->right)
	{
	  offset_y = offset_y_orig;
	      
	  while (offset_y <= update_box->bottom)
	    {
	      if ((((gint)ceil (offset_x / grid->width_x) % grid->interval_x) == 0)
		  || (((gint)ceil (offset_y / grid->width_y) % grid->interval_y) == 0))
		{
		  dia_display_transform_coords (ddisp, offset_x, offset_y,
						&x, &y);
		  
		  gdk_draw_point (ddisp->renderer->pixmap, gc, x, y);
		}
	      offset_y += grid->width_y;
	    }
	  offset_x += grid->width_x;
	}
    } /* if (grid->draw_as_points) */
  else
    {
      gint width, height;
            
      dia_display_transform_coords (ddisp, update_box->right,
				    update_box->bottom,
				    &width, &height);
  
      while (offset_x <= update_box->right)
	{
	  if (((gint)ROUND (offset_x / grid->width_x) % grid->interval_x) == 0)
	    {
	      dia_display_transform_coords (ddisp, offset_x,
					    update_box->top, &x, &y);
	      gdk_draw_line (ddisp->renderer->pixmap, gc, x, y, x, height + 1);
	    }
	  offset_x += grid->width_x;
	}
      
      while (offset_y <= update_box->bottom)
	{
	  if (((gint)ROUND (offset_y / grid->width_y) % grid->interval_y) == 0)
	    {
	      dia_display_transform_coords (ddisp, update_box->left,
					    offset_y, &x, &y);

	      gdk_draw_line (ddisp->renderer->pixmap, gc, x, y, width + 1, y);
	    }
	  offset_y += grid->width_y;
	}
    } /* else */
}

gboolean
dia_grid_snap (DiaGrid *grid, gfloat *x, gfloat *y)
{
  g_return_val_if_fail (grid != NULL, FALSE);
  
  if (grid->snap)
    {
      *x = ROUND((*x) / grid->width_x) * grid->width_x;
      *y = ROUND((*y) / grid->width_y) * grid->width_y;
      return TRUE;
    }
  return FALSE;
}








