/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diagroup.h"

static void destroy (DiaObject *object);
static void draw (DiaObject *obj, DiaRenderer *renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject *obj, Point *point);
static void copy (DiaGroup *from, DiaGroup **to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static DiaConnectionPoint* connection_point_request (DiaObject *obj,
						     DiaHandle *handle,
						     Point *pos);
static void connection_point_disconnect (DiaObject *obj,
					 DiaConnectionPoint *p,
					 DiaHandle *h);

static DiaObjectType group_type =
{
  "DiaGroup",
  dia_object_get_type /* parent */
};

static DiaObjectOps group_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,  
  distance,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  connection_point_request,
  connection_point_disconnect
};

static DiaObjectOps *parent_ops = NULL;

DiaObjectType*
dia_group_get_type ()
{
  return &group_type;
}

void
dia_group_init (DiaGroup *group)
{
  /* call parent */
  dia_object_init (DIA_OBJECT (group)); /* explicit! */
  /* save parent options */
  if (!parent_ops)
    parent_ops = group->object.ops;
  
  /* overwrite old function declarations */
  group->object.object_type = &group_type;
  group->object.ops = &group_ops;

  /* initialize other stuff (attributes, etc.) */
  group->objects = NULL;
}

DiaObject*
dia_group_new ()
{
  DiaGroup *new_group;

  new_group = g_new (DiaGroup, 1);
  
  dia_group_init (new_group);

  return DIA_OBJECT (new_group);
}

void
dia_group_add_object (DiaGroup *group, DiaObject *object)
{
  g_return_if_fail (group != NULL);
  g_return_if_fail (object != NULL);
  
  if (!g_list_find (group->objects, object))
    group->objects = g_list_append (group->objects, object);
}

gboolean
dia_group_remove_object (DiaGroup *group, DiaObject *object)
{
  g_return_if_fail (group != NULL);
  g_return_if_fail (object != NULL);
  
  if (!g_list_find (group->objects, object))
    {
      group->objects = g_list_remove (group->objects, object);
      return TRUE;
    }
  else
    return FALSE;
}

gboolean
dia_group_has_object (DiaGroup *group, DiaObject *object)
{
  g_return_if_fail (group != NULL);
  g_return_if_fail (object != NULL);
  
  return g_list_find (group->objects, object) ? TRUE : FALSE;
}

void
dia_group_add_object_list (DiaGroup *group, GList *list)
{
  g_return_if_fail (group != NULL);
  g_return_if_fail (list != NULL);
  
  while (list)
    {
      dia_group_add_object (group, DIA_OBJECT (list->data));
      list = g_list_next (list);
    }
}

void
dia_group_remove_object_list (DiaGroup *group, GList *list)
{
  g_return_if_fail (group != NULL);
  g_return_if_fail (list != NULL);
  
  while (list)
    {
      dia_group_remove_object (group, DIA_OBJECT (list->data));
      list = g_list_next (list);
    }
}

/* "events" */
static void
destroy (DiaObject *object)
{
  DiaGroup *group = DIA_GROUP (object);
  GList *l;
  
  l = group->objects;
  while (l)
    {
      DIA_OBJECT (l->data)->ops->destroy (DIA_OBJECT (l->data));
      l = g_list_next (l);
    }
  g_list_free (group->objects);

  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  GList *l;
  
  g_return_if_fail (obj != NULL);
  g_return_if_fail (renderer != NULL);
  
  l = DIA_GROUP (obj)->objects;
  while (l)
    {
      DIA_OBJECT (l->data)->ops->draw (DIA_OBJECT (l->data), renderer);
      l = g_list_next (l);
    }
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  gfloat dist, mindist;
  GList *l;
  
  g_return_if_fail (obj != NULL);
  g_return_if_fail (point != NULL);

  mindist = G_MAXFLOAT;
  l = DIA_GROUP (obj)->objects;
  while (l)
    {
      dist = DIA_OBJECT (l->data)->ops->distance (DIA_OBJECT (l->data), point);
      if (dist < mindist)
	mindist = dist;
      l = g_list_next (l);
    }
  return mindist;
}

static void
copy (DiaGroup *from, DiaGroup **to)
{
  DiaObject *new_item;
  
  GList *l;
  
  g_return_if_fail (from != NULL);
  g_return_if_fail (to != NULL);
  
  if (*to == NULL)
    {
      *to = DIA_GROUP (dia_group_new ());
    }
  
  parent_ops->copy (DIA_OBJECT (from), (DiaObject**) to);
  
  l = from->objects;
  while (l)
    {
      new_item = NULL;
      DIA_OBJECT (l->data)->ops->copy (DIA_OBJECT (l->data), &new_item);
      (*to)->objects = g_list_append ((*to)->objects, new_item);
      g_list_next (l);
    }
}

static gint
is_empty (DiaObject* obj)
{
  return DIA_GROUP (obj)->objects ? FALSE : TRUE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  /* FIXME: implement */
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  GList *l;
  gboolean result = FALSE;
  
  obj->request = DIA_REQUEST_NOTHING;
  
  /* No handle movements, please... */
  if (DIA_EVENT_IS_BUTTON (event))
    event->button.handle = NULL;
  
  l = DIA_GROUP (obj)->objects;
  while (l)
    {
      result |= DIA_OBJECT (l->data)->ops->event (DIA_OBJECT (l->data),
						  event, layer);
      obj->request |= DIA_OBJECT (l->data)->request;
      l = g_list_next (l);
    }
  return result;
}

static DiaConnectionPoint*
connection_point_request (DiaObject *obj, DiaHandle *handle, Point *pos)
{
  GList *l;
  DiaConnectionPoint *cp = NULL;
  Point abs_point, p;
  
  abs_point = *pos;
  point_add (&abs_point, &obj->position);
  
  l = DIA_GROUP (obj)->objects;
  while (l && !cp)
    {
      p = abs_point;
      point_sub (&p, &DIA_OBJECT (l->data)->position);
      
      cp = DIA_OBJECT (l->data)->ops->request_connection_point (DIA_OBJECT (l->data), handle, &p);
      l = g_list_next (l);
    }
  
  return cp;
}

static void
connection_point_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			     DiaHandle *h)
{
  if (g_list_find (DIA_GROUP (obj)->objects, cp->object))
    cp->object->ops->disconnect_connection_point (cp->object, cp, h);
  else
    {
      g_warning ("DiaGroup::connection_point_disconnect: connection point");
      g_warning ("   doesn't belong to an object in this group!");
    }
}



