/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "dialayer.h"

/* Render engine for objects: */
static void
normal_renderer (DiaObject *obj, DiaRenderer *renderer, gpointer data)
{
  obj->ops->draw (obj, renderer);
}

DiaLayer*
dia_layer_new (gchar *name)
{
  DiaLayer *new_layer;
  
  new_layer = g_new (DiaLayer, 1);
  
  if (name)
    new_layer->name = g_strdup (name);
  else
    new_layer->name = NULL;
  
  new_layer->extents.left = -1000.0; 
  new_layer->extents.right = 1000.0; 
  new_layer->extents.top = -1000.0; 
  new_layer->extents.bottom = 1000.0; 
 
  new_layer->objects = NULL;
  new_layer->visible = TRUE;
  
  return new_layer;
}

void
dia_layer_destroy (DiaLayer *layer)
{
  GList *list;
  
  if (layer->name)
    g_free (layer->name);
  
  list = layer->objects;
  while (list)
    {
      dia_object_destroy ((DiaObject*) list->data);
    }
  g_list_free (layer->objects);
}

void
dia_layer_render (DiaLayer *layer, DiaRenderer *renderer,
		  DiaObjectRenderer obj_renderer /* Can be NULL */,
		  Rectangle *update_box, gpointer data)
{
  GList *list;
  DiaObject *obj;
  
  g_return_if_fail (layer != NULL);
  g_return_if_fail (renderer != NULL);
  g_return_if_fail (update_box != NULL);
  
  if (obj_renderer == NULL)
    obj_renderer = normal_renderer;
  
  /* Draw all objects, but only if they intersect with the update-rectangle: */
  list = layer->objects;
  while (list != NULL)
    {
      obj = (DiaObject *) list->data;
      
      if (rectangle_intersects (update_box, &obj->bounding_box))
	{
	  (*obj_renderer) (obj, renderer, data);
	}
      
      list = g_list_next(list);
    }
}

void
dia_layer_add_object (DiaLayer *layer, DiaObject *obj)
{
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj != NULL);
  
  layer->objects = g_list_append (layer->objects, (gpointer) obj);
}

void
dia_layer_add_objects (DiaLayer *layer, GList *obj_list)
{
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj_list != NULL);
  
  layer->objects = g_list_concat (layer->objects, obj_list);
}

void
dia_layer_add_object_first (DiaLayer *layer, DiaObject *obj)
{
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj != NULL);
  
  layer->objects = g_list_prepend (layer->objects, (gpointer) obj);
}

void
dia_layer_add_objects_first (DiaLayer *layer, GList *obj_list)
{
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj_list != NULL);
  
  layer->objects = g_list_concat (obj_list, layer->objects);
}

void
dia_layer_remove_object (DiaLayer *layer, DiaObject *obj)
{
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj != NULL);
  
  layer->objects = g_list_remove (layer->objects, obj);
}

GList*
dia_layer_find_objects_in_rectangle (DiaLayer *layer, 
				     Rectangle *rect)
{
  GList *list;
  GList *selected_list;
  DiaObject *obj;
  
  selected_list = NULL;
  list = layer->objects;
  while (list != NULL)
    {
      obj = (DiaObject *)list->data;
    
      if (rectangle_in_rectangle(rect, &obj->bounding_box))
	{
	  selected_list = g_list_prepend (selected_list, obj);
	}
    
      list = g_list_next(list);
    }

  return selected_list;
}

DiaObject*
dia_layer_find_closest_object (DiaLayer *layer, Point *pos,
			       gfloat maxdist)
{
  return dia_layer_find_closest_object_by_type (layer, pos, maxdist, NULL);
}

DiaObject*
dia_layer_find_closest_object_by_type (DiaLayer *layer, Point *pos,
				       gfloat maxdist, DiaObjectType *type)
{
  GList *l;
  DiaObject *closest;
  DiaObject *obj;
  gfloat dist;
  gfloat mindist;
  
  closest = NULL;
  mindist = maxdist;
  
  l = layer->objects;
  while (l)
    {
      obj = DIA_OBJECT (l->data);

      /* Calculating with relative values */
      dist = dia_object_distance (obj, pos);
      if ((dist <= mindist)
	  && ((type == NULL)
	      || dia_object_check_type (DIA_OBJECT (l->data), type)))
	{
	  closest = obj;
	  mindist = dist;
	}
      l = g_list_next(l);
    }
  return closest;  
}

gfloat
dia_layer_find_closest_cp (DiaLayer *layer, DiaHandle *handle,
			   DiaObject **closest_obj, Point *con_pos,
			   DiaConnectionPoint **closest_cp)
{
  GList *l;
  DiaObject *obj;
  gfloat mindist = G_MAXFLOAT; /* A real big value */
  gfloat dist;
  Point pos;
  DiaConnectionPoint *cp;
    
  g_return_val_if_fail (layer != NULL, mindist);
  g_return_val_if_fail (handle != NULL, mindist);
  g_return_val_if_fail (closest_obj != NULL, mindist);
  g_return_val_if_fail (con_pos != NULL, mindist);
  g_return_val_if_fail (closest_cp != NULL, mindist);
  g_return_val_if_fail (handle->object, mindist);
  
  *closest_cp = NULL;
  *closest_obj = NULL;
  
  l = layer->objects;
  while (l)
    {
      obj = DIA_OBJECT (l->data);
      if (handle->object != obj)
	{
     	  dist = dia_object_cp_distance (obj, &handle->pos, &pos, &cp);
	  /* Rather snap to a 
	   */
	
	  if ((!*closest_cp && cp && (dist < DIA_CONNECTION_POINT_TRESHOLD))
	      || (!cp && (dist < mindist)
		  && (!*closest_cp
		      || (mindist >= DIA_CONNECTION_POINT_TRESHOLD))))
	    {
	      *closest_obj = obj;
	      *con_pos = pos;
	      *closest_cp = cp;
	      mindist = dist;
	    }
	}
      l = g_list_next(l);
    }

  return mindist;
}

/* find the handle closest to POS */
gfloat
dia_layer_find_closest_handle (DiaLayer *layer, Point *pos,
			       DiaHandle **handle)
{
  g_return_val_if_fail (layer != NULL, G_MAXFLOAT);
  g_return_val_if_fail (pos != NULL, G_MAXFLOAT);
  g_return_val_if_fail (handle != NULL, G_MAXFLOAT);

  return dia_find_closest_handle_from_list (layer->objects, pos, handle);
}

gint
dia_layer_update_extents (DiaLayer *layer)
{
  GList *l;
  Rectangle new_extents;
  gint changed;
  
  l = layer->objects;
  if (l)
    {
      new_extents = DIA_OBJECT (l->data)->bounding_box;
      l = g_list_next(l);
  
      while (l)
	{
	  rectangle_union(&new_extents, &DIA_OBJECT (l->data)->bounding_box);
	  l = g_list_next(l);
	}
    }
  else
    {
      new_extents.top = 0.0;
      new_extents.bottom = 0.0;
      new_extents.left = 0.0;
      new_extents.right = 0.0;
    }
  
  changed = ( (new_extents.left != layer->extents.left) ||
	      (new_extents.right != layer->extents.right) ||
	      (new_extents.top != layer->extents.top) ||
	      (new_extents.bottom != layer->extents.bottom) );

  layer->extents = new_extents;

  return changed;
}

void
dia_layer_replace_object_with_list (DiaLayer *layer, DiaObject *obj,
				    GList *insert_list)
{
  GList *list;

  list = g_list_find(layer->objects, obj);

  g_return_if_fail (list != NULL);
  
  if (list->prev == NULL)
    {
      layer->objects = insert_list;
    }
  else
    {
      list->prev->next = insert_list;
      insert_list->prev = list->prev;
    }

  if (list->next != NULL)
    {
      GList *last;
      last = g_list_last(insert_list);
      last->next = list->next;
      list->next->prev = last;
    }
  
  /* remove object from memory: */
  ((DiaObject*)list->data)->ops->destroy ((DiaObject*)list->data);
  g_list_free_1(list);
}

























































































