/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diamultiline.h"
#include "math.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaMultiLine* from, DiaMultiLine *to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps line_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

#define parent_ops (dia_multi_line_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_multi_line_get_type ()
{
  static DiaObjectType line_type =
  {
    "DiaMultiLine",
    sizeof (DiaMultiLine),
    (DiaInitFunc) dia_multi_line_init,
    &line_ops,
    dia_dyn_line_get_type
  };
  return &line_type;
}

void
dia_multi_line_init (DiaMultiLine *line)
{
  ((DiaBaseLine*)line)->line_caps = DIA_LINE_CAPS_ROUND;
  ((DiaBaseLine*)line)->line_join = DIA_LINE_JOIN_ROUND;
}

DiaObject*
dia_multi_line_new (Point **pos, gint num_points)
{
  DiaObject *new_line;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_line = dia_object_create (dia_multi_line_get_type ());
  
  //dia_object_move (new_line, pos->x, pos->y);
  /* FIXME: implement */

  return new_line;
}

/* Point* dia_multi_line_point_on_line (...)
 *
 * Figure out if a point is on the line and, if so, returns the handle
 * that starts the line segment. POINT is adjusted so it lays on the line.
 */
DiaHandle*
dia_multi_line_point_on_line (DiaMultiLine *line, Point *point)
{
  gint i;
  DiaHandle *h = NULL;
  Point p_on_line;
  Point tmp;
  gfloat min_dist = (gfloat) DIA_CONNECTION_POINT_TRESHOLD;
  gfloat dist;
  
  g_return_val_if_fail (line != NULL, NULL);
  g_return_val_if_fail (point != NULL, NULL);

  for (i=0; i < DIA_OBJECT (line)->handles->len-1; i++)
    {
      dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (line, i)->pos,
				  &DIA_OBJECT_GET_HANDLE (line, i+1)->pos,
				  DIA_BASE_LINE (line)->line_width,
				  point, &tmp);
      if (dist < min_dist)
	{
	  min_dist = dist;
	  p_on_line = tmp;
	  h = DIA_OBJECT_GET_HANDLE (line, i);
	}
    }

  if (h)
    *point = p_on_line;

  return h;
}

/* dia_multi_line_add_line_segment
 *
 * Add a handle to the line at POINT, this will cause a new segment to
 * be created. PREV_HANDLE is the first handle of the segment that is about
 * to be split.
 * The CP's will also be updated.
 * NOTE: The point has to be placed between the first and the last point, the
 *       line can not be extended. 
 */
DiaHandle*
dia_multi_line_add_line_segment (DiaMultiLine *line, Point *point,
				 DiaHandle *prev_handle)
{
  DiaObject *obj;
  gint index;
  gint i;
  gfloat factor;
  DiaBaseLineCP *blcp;
  DiaHandle *new_handle;
  
  g_return_val_if_fail (line != NULL, NULL);
  g_return_val_if_fail (point != NULL, NULL);
  g_return_val_if_fail (prev_handle != NULL, NULL);
  
  obj = DIA_OBJECT (line);
    
  index = obj->handles->len - 1;
  
  /* find the index where the point has to be inserted: */
  while ((index > 0) && 
	 (DIA_OBJECT_GET_HANDLE (obj, index - 1) != prev_handle))
    index--;
    
  /* First find out our own factor (the CP's factor should be bigger) */
  factor =
    dia_base_line_cp_calc_factor (&DIA_OBJECT_GET_HANDLE (obj, index - 1)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, index)->pos,
				  point);
  
  /* add the new handle: */
  new_handle = dia_object_insert_handle (obj, point->x, point->y, index);

  new_handle->is_connectable = FALSE;
  
  /* Now reassign the DiaBaseLineCP's for the CP's that are now located on
   * the new segment: */
  for (i = 0; i < obj->connections->len; i++)
    {
      blcp = (DiaBaseLineCP*) DIA_OBJECT_GET_CP (line, i)->data;
      
      if (blcp->handle == prev_handle)
	{
	  if (blcp->factor > factor)
	    {
	      blcp->handle = new_handle;
	      blcp->factor = 
		dia_base_line_cp_calc_factor (point,
					      &DIA_OBJECT_GET_HANDLE (obj,
								      index + 1)->pos,
					      &DIA_OBJECT_GET_CP (line, i)->pos);
	    }
	  else
	    {
	      blcp->factor = 
		dia_base_line_cp_calc_factor (&prev_handle->pos,
					      point,
					      &DIA_OBJECT_GET_CP (line, i)->pos);
	      
	    }
	}
    }
  
  return new_handle;
}

/* Remove handles that lay on one line. If handles were removed the BB
 * is recalculated and the REDRAW request is set.
 */
gboolean
dia_multi_line_update_handles (DiaMultiLine *line)
{
  DiaObject *obj;
  gint i;
  gfloat a1, a2;
  Point *p1, *p2, *p3;
  gboolean updated = FALSE;
  
  g_return_val_if_fail (line != NULL, FALSE);
  
  obj = DIA_OBJECT (line);
  
  if (obj->handles->len < 3)
    return FALSE;
  
  i = 0;
  while (i < obj->handles->len - 2)
    {      
      p1 = &DIA_OBJECT_GET_HANDLE (obj, i)->pos; // previous handle
      p2 = &DIA_OBJECT_GET_HANDLE (obj, i + 1)->pos; // handle to check
      p3 = &DIA_OBJECT_GET_HANDLE (obj, i + 2)->pos; // next handle

      /* We use the dy/dx method to see if these points are in one line */
      if (ABS (p2->x - p1->x) <= DIA_MULTI_LINE_SIGMA)
	{
	  if (ABS (p3->x - p2->x) <= DIA_MULTI_LINE_SIGMA)
	    {
	      dia_base_line_remove_point (DIA_BASE_LINE (obj), i + 1);
	      updated = TRUE;
	    }
	  else
	    i++;
	}
      else if ((p3->x - p2->x) == 0.0)
	{
	  i++;
	}
      else /* line is not vertical */
	{
	  a1 = atan2 (p2->y - p1->y, p2->x - p1->x);
	  a2 = atan2 (p3->y - p2->y, p3->x - p2->x);
	  //g_message ("a1 = %f; a2 = %f", a1, a2);
	  
	  if (ABS (a1 - a2) < DIA_MULTI_LINE_SIGMA)
	    {
	      dia_base_line_remove_point (DIA_BASE_LINE (obj), i + 1);
	      updated = TRUE;
	    }
	  else
	    i++;
	}
    }

  if (updated)
    {
      dia_object_add_update (obj, &obj->bounding_box);
      dia_object_calc_bounding_box (obj);
      /* BB can only become smaller. */
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
    }
  return updated;
}


/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaMultiLine *from, DiaMultiLine *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  return parent_ops->calc_bounding_box (obj);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  
  switch (event->type)
    {
      /* Check if the object is placed inside an element, else request a
	 destroy. */
    case DIA_BUTTON_PRESS:
      switch (event->button.button)
	{
	case 1:
	  DIA_MULTI_LINE (obj)->handle_on_line =
	    dia_multi_line_point_on_line (DIA_MULTI_LINE (obj),
					  &event->button.pos);
	  if (!event->button.handle && DIA_MULTI_LINE (obj)->handle_on_line
	      && !(event->button.modifier & DIA_SHIFT_MASK))
	    {
	      //g_message ("Add a handle if a motion event occurs. %x");
	      DIA_OBJECT_SET_FLAGS (obj, DIA_MULTI_LINE_MOUSE_ON_LINE);	    
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_SELECT
				      | DIA_REQUEST_FOCUS
				      | DIA_REQUEST_GRAB
				      | DIA_REQUEST_REDRAW);
	    }
	  else
	    {
	      //g_message ("calling parent for line motion");
	      result = parent_ops->event (obj, event, layer);
	    }
	  result = TRUE;
	  break;
	}
      break;
    case DIA_MOTION:
      /* create a new handle if the mouse is pulling on a line. */
      if (DIA_OBJECT_IS_SET (obj, DIA_MULTI_LINE_MOUSE_ON_LINE))
	{
	  //g_message ("Add handle");
	  	  
	  DIA_BASE_LINE (obj)->handle_moving = 
	    dia_multi_line_add_line_segment (DIA_MULTI_LINE (obj),
					     &event->motion.pos,
					     DIA_MULTI_LINE (obj)->handle_on_line);
	  /* We are now moving a handle (lets fool the tool!) */
	  event->button.handle = DIA_BASE_LINE (obj)->handle_moving;
	  
	  DIA_OBJECT_UNSET_FLAGS (obj, DIA_MULTI_LINE_MOUSE_ON_LINE);
	  DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_LINE_MOVE_HANDLE);    
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  result = TRUE;
	}
      result |= parent_ops->event (obj, event, layer);
      break;
    case DIA_BUTTON_RELEASE:
      switch (event->button.button)
	{
	case 1:
	  if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_MOVE_HANDLE))
	    {
	      if (dia_multi_line_update_handles (DIA_MULTI_LINE (obj)))
		DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_UPDATE_CONNECTIONS);
	      result = TRUE;
	    }
	  /* unset: maybe the mouse hasn't moved... */
	  DIA_OBJECT_UNSET_FLAGS (obj, DIA_MULTI_LINE_MOUSE_ON_LINE);

	  result |= parent_ops->event (obj, event, layer);
	} /* switch (event->button.button) */
      break;
    default:
      result = parent_ops->event (obj, event, layer);
      break;
    }
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  return parent_ops->cp_connect (obj, h, pos);
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}





