/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "diaobject.h"
#include "dialayer.h"

/* use this function to return a NULL. This makes the dia_object_check_cast
 * function much easier.
 */
DiaColor dia_handle_colors[] =
{ /* red  green  blue */
  { 0.0, 0.0, 0.4 }, /* 0: selected, unmovable */
  { 0.0, 0.8, 0.0 }, /* 1: selected, movable */
  { 1.0, 0.5, 0.5 }, /* 2: selected, movable, can be connected */
  { 1.0, 0.0, 0.0 }, /* 3: selected, movable, connected */
  { 0.0, 0.0, 1.0 }, /* 4: focused, unmovable */
  { 0.0, 1.0, 0.0 }, /* 5: focused, movable */
  { 1.0, 0.5, 0.5 }, /* 6: focused, movable, can be connected */
  { 1.0, 0.0, 0.2 }, /* 7: focused, movable, connected */
};

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaObject* from, DiaObject *to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject* obj) { }
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps object_ops = 
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

DiaObjectType*
dia_object_get_type ()
{
  static DiaObjectType object_type =
  {
    "DiaObject",
    sizeof (DiaObject),
    dia_object_init,
    &object_ops,
    NULL /* Only DiaObject has parent NULL!!! */
  };
  
  return &object_type;
}

gboolean
dia_object_check_type (DiaObject *obj, DiaObjectType *typ)
{
  DiaObjectType *t;
  
  g_return_val_if_fail (obj != NULL, FALSE);
  g_return_val_if_fail (typ != NULL, FALSE);
  
  t = obj->object_type;

  while (t)
    {
      /* check if types are identical using their pointer values. */
      if (t == typ)
	return TRUE;
       else
	 if (t->parent_type)
	   t = t->parent_type ();
	 else
	   return FALSE;
    }
    
/*   g_warning ("Invalid cast from \"%s\" to \"%s\".", obj->object_type->name, */
/* 	     typ->name); */
  return FALSE;
}

gboolean
dia_object_check_type_name (DiaObject *obj, gchar* name)
{
  DiaObjectType *t;
  
  g_return_val_if_fail (obj != NULL, FALSE);
  g_return_val_if_fail (name != NULL, FALSE);
  
  t = obj->object_type;
  while (t)
    {
      /* check if types are identical, e.g. have the same name */
      if (!strcmp (t->name, name))
      	return TRUE;
      else
	t = t->parent_type ();
    }
  
/*   g_warning ("Invalid cast from \"%s\" to \"%s\".", obj->object_type->name, */
/* 	     name); */
  return FALSE;
}

void
dia_object_init (DiaObject *object)
{
  object->flags = 0;
  
  object->handles = g_ptr_array_new ();

  object->connections = g_ptr_array_new ();

  object->bounding_box.top = 0.0;
  object->bounding_box.bottom = 0.0;
  object->bounding_box.left = 0.0;
  object->bounding_box.right = 0.0;

  object->request = DIA_REQUEST_NOTHING;
  
  object->update_box.top = 0.0;
  object->update_box.bottom = 0.0;
  object->update_box.left = 0.0;
  object->update_box.right = 0.0;

  object->snap_pos.x = 0.0;
  object->snap_pos.y = 0.0;
  
  object->data = NULL;
}

static void
recursive_init (DiaObject *obj, DiaObjectType *typ)
{
  if (typ->parent_type)
    recursive_init (obj, typ->parent_type ());
  
  if (typ->init)
    typ->init (obj);
}

DiaObject*
dia_object_create (DiaObjectType *typ)
{
  DiaObject *new_obj;
  
  g_return_val_if_fail (typ != NULL, NULL);
  
  new_obj = g_malloc (typ->size);
  
  g_assert (new_obj != NULL);
  
  new_obj->object_type = typ;
  new_obj->ops = typ->ops;

  recursive_init (new_obj, typ);
  
  /* Also calculate the bounding box, this avoids every init func calling the
   * bounding-box func. */
  dia_object_calc_bounding_box (new_obj);
  
  return new_obj;
}

void
dia_object_destroy (DiaObject *object)
{
  g_return_if_fail (object != NULL);
  
  object->ops->destroy (object);
}

void
dia_object_draw (DiaObject* obj, DiaRenderer* renderer)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (renderer != NULL);
  
  obj->ops->draw (obj, renderer);
}
void
dia_object_draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (renderer != NULL);
  
  obj->ops->draw_handles (obj, renderer);
}

void
dia_object_draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (renderer != NULL);
  
  obj->ops->draw_cps (obj, renderer);
}

gfloat
dia_object_distance (DiaObject* obj, Point* point)
{
  g_return_val_if_fail (obj != NULL, -1.0);
  g_return_val_if_fail (point != NULL, -1.0);
  
  return obj->ops->distance (obj, point);
}

void
dia_object_move (DiaObject *object, gfloat dx, gfloat dy)
{
  g_return_if_fail (object != NULL);

  object->ops->move (object, dx, dy);
}

void
dia_object_move_handle (DiaObject *object, DiaHandle *handle,
			gfloat dx, gfloat dy)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (handle != NULL);
  g_return_if_fail (handle->object == object);

  object->ops->move_handle (object, handle, dx, dy);
}

void
dia_object_move_handle_abs (DiaObject *object, DiaHandle *handle,
			    gfloat x, gfloat y)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (handle != NULL);
  g_return_if_fail (handle->object == object);

  object->ops->move_handle (object, handle, x - handle->pos.x,
			    y - handle->pos.y);
}

DiaObject*
dia_object_copy (DiaObject* from)
{
  DiaObject *to;
  
  g_return_val_if_fail (from != NULL, NULL);

  to = dia_object_create (from->object_type);
  
  from->ops->copy (from, to);

  return to;
}

gint
dia_object_event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  g_return_val_if_fail (obj != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (layer != NULL, FALSE);

  return obj->ops->event (obj, event, layer);
}

gint
dia_object_is_empty (DiaObject *obj)
{
  g_return_val_if_fail (obj != NULL, FALSE);

  return obj->ops->is_empty (obj);
}

void
dia_object_calc_bounding_box (DiaObject *obj)
{
  g_return_if_fail (obj != NULL);

  return obj->ops->calc_bounding_box (obj);
}
  
gfloat
dia_object_cp_distance (DiaObject *object, Point *pos, Point *con_pos,
			DiaConnectionPoint **cp)
{
  gfloat dist;
  
  g_return_val_if_fail (object != NULL, -1.0);
  g_return_val_if_fail (pos != NULL, -1.0);
  g_return_val_if_fail (con_pos != NULL, -1.0);
  g_return_val_if_fail (cp != NULL, -1.0);
  
  *cp = NULL;
  
  dist = object->ops->cp_distance (object, pos, con_pos, cp);

  /* Test if, when a CP is returned, both positions are the same. */
  if (*cp)
    {
      g_assert ((*cp)->pos.x == con_pos->x);
      g_assert ((*cp)->pos.y == con_pos->y);
    }
  
  return dist;
}

DiaConnectionPoint*
dia_object_cp_connect (DiaObject *object, DiaHandle *h, Point *pos)
{
  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (h != NULL, NULL);
  //g_return_if_fail (pos != NULL); may be NULL!

  return object->ops->cp_connect (object, h, pos);
}

void
dia_object_cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (obj == cp->object);
  g_return_if_fail (h != NULL);
  
  obj->ops->cp_disconnect (obj, cp, h);
}

void
dia_object_add_update (DiaObject *obj, Rectangle *update_box)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (update_box != NULL);

  if ((obj->update_box.left == obj->update_box.right)
      && (obj->update_box.top == obj->update_box.bottom))
    obj->update_box = *update_box;
  else
    rectangle_union (&obj->update_box, update_box);
}

void
dia_object_calc_bounding_box_update (DiaObject *obj)
{
  g_return_if_fail (obj != NULL);

  dia_object_add_update (obj, &obj->bounding_box);
  dia_object_calc_bounding_box (obj);
  dia_object_add_update (obj, &obj->bounding_box);
}

void
dia_object_destroy_object_list (GList *list_to_be_destroyed)
{
  GList *list;
  DiaObject *obj;

  g_return_if_fail (list_to_be_destroyed != NULL);
  
  list = list_to_be_destroyed;
  while (list)
    {
      obj = (DiaObject*) list->data;

      obj->ops->destroy (obj);
    
      list = g_list_next(list);
    }

  g_list_free(list_to_be_destroyed);
}

DiaHandle*
dia_object_add_handle (DiaObject *object, gfloat x, gfloat y)
{
  DiaHandle *new_handle;
  
  g_return_val_if_fail (object != NULL, NULL);
  
  new_handle = (DiaHandle*) g_new (DiaHandle, 1);
  
  dia_handle_init (new_handle);
  
  new_handle->pos.x = x;
  new_handle->pos.y = y;
  new_handle->object = object;
  
  g_ptr_array_add (object->handles, new_handle);
  
  return new_handle;
}

DiaHandle*
dia_object_insert_handle (DiaObject *object, gfloat x, gfloat y, gint index)
{
  DiaHandle *new_handle;
  gint i;
  
  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (index >= 0, NULL);
  g_return_val_if_fail (index < object->handles->len, NULL);
  
  new_handle = (DiaHandle*) g_new (DiaHandle, 1);
  
  dia_handle_init (new_handle);
  
  new_handle->pos.x = x;
  new_handle->pos.y = y;
  new_handle->object = object;
  
  g_ptr_array_add (object->handles,
		   DIA_OBJECT_GET_HANDLE (object, object->handles->len - 1));

  for (i =  object->handles->len - 1; i > index; i--)
    DIA_OBJECT_GET_HANDLE (object, i) = DIA_OBJECT_GET_HANDLE (object, i - 1);

  DIA_OBJECT_GET_HANDLE (object, index) = new_handle;
    
  return new_handle;
}

void
dia_object_remove_handle (DiaObject *object, DiaHandle *handle)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (handle != NULL);
  g_return_if_fail (handle->connected_to == NULL);
  
  g_ptr_array_remove (object->handles, handle);
}

gfloat
dia_object_find_closest_handle (DiaObject *object, Point *pos,
				DiaHandle **handle)
{
  gint i;
  gfloat dist;
  gfloat mindist = G_MAXFLOAT;
  
  //g_message ("requested max_distance = %f", max_distance);
  g_return_val_if_fail (object != NULL, -1.0);
  g_return_val_if_fail (pos != NULL, -1.0);
  g_return_val_if_fail (handle != NULL, -1.0);
  
  for (i = 0; i < object->handles->len; i++)
    {
      /* Note: Uses manhattan metric for speed... */
      dist = distance_point_point_manhattan (pos,
					     &DIA_OBJECT_GET_HANDLE (object, i)->pos);
      if (dist < mindist)
	{
	  *handle = DIA_OBJECT_GET_HANDLE (object, i);
	  mindist = dist;
	}
    }
  return mindist;
}

DiaConnectionPoint*
dia_object_add_connection_point (DiaObject *object, gfloat x, gfloat y)
{
  DiaConnectionPoint *new_cp;
  
  g_return_val_if_fail (object != NULL, NULL);
    
  new_cp = g_new (DiaConnectionPoint, 1);

  new_cp->object = object;
  new_cp->connected = NULL;
  new_cp->pos.x = x;
  new_cp->pos.y = y;
  new_cp->data = NULL;
  
  g_ptr_array_add (object->connections, new_cp);

  return new_cp;
}

gboolean
dia_object_is_connected_to (DiaObject *object, DiaObject *conn_obj)
{
  gint i;
  DiaConnectionPoint *cp;
  
  g_return_val_if_fail (object != NULL, FALSE);
  g_return_val_if_fail (conn_obj != NULL, FALSE);
  
  for (i = 0; i < conn_obj->handles->len; i++)
    {
      cp = DIA_OBJECT_GET_HANDLE (conn_obj, i)->connected_to;
      if (cp && (cp->object == object))
	return TRUE;
      else if (cp && dia_object_is_connected_to (object, cp->object))
        return TRUE;
    }
  return FALSE;
}

void
dia_object_unconnect_all (DiaObject *object)
{
  gint i, j;
  
  g_return_if_fail (object != NULL);

  for (i=0; i < object->handles->len; i++)
    dia_handle_disconnect ((DiaHandle*)
			   g_ptr_array_index (object->handles, i));
  
  j= object->connections->len;
  i = 0;
  while (i < j)
    {
      dia_connection_point_unconnect ((DiaConnectionPoint*) 
				      g_ptr_array_index(object->connections, i));
      if (object->connections->len < j) /* connection point = removed */
	{
	  j = object->connections->len;
	}
      else
	i++;
    }
}

gfloat
dia_find_closest_handle_from_list (GList *list, Point *pos, DiaHandle **handle)
{
  DiaObject *obj;
  DiaHandle *han;
  gfloat mindist, dist;
  gint i;
  
  g_return_val_if_fail (list != NULL, -1.0);
  g_return_val_if_fail (pos != NULL, -1.0);
  g_return_val_if_fail (handle != NULL, -1.0);

  mindist = G_MAXFLOAT; /* Realy big value... */
  
  *handle = NULL;
  
  while (list)
    {
      obj = (DiaObject *) list->data;
      
      for (i = 0; i < obj->handles->len; i++)
	{
	  han = DIA_OBJECT_GET_HANDLE (obj, i);
	  /* Note: Uses manhattan metric for speed... */
	  dist = distance_point_point_manhattan (pos, &han->pos);
	  //g_message ("distance cursor-handle: %f", dist);
	  if (dist < mindist)
	    {
	      mindist = dist;
	      *handle = han;
	    }
	}
      list = g_list_next(list);
    }

  return mindist;
}


/* HANDLE
 */
void
dia_handle_init (DiaHandle *handle)
{
  g_return_if_fail (handle != NULL);
  
  handle->pos.x = 0.0;
  handle->pos.y = 0.0;
  handle->object = NULL;
  handle->keep_on_move = FALSE;
  handle->is_connectable = TRUE;
  handle->is_movable = TRUE;
  handle->could_be_connected = FALSE;
  handle->connected_to = NULL;
  handle->data = NULL;
}

gint
dia_handle_connect (DiaHandle *handle, DiaConnectionPoint *connectionpoint)
{
  g_return_val_if_fail (handle != NULL, FALSE);
  g_return_val_if_fail (connectionpoint != NULL, FALSE);

  handle->could_be_connected = FALSE;
  if (!handle->is_connectable)
    {
      g_warning ("Error? trying to connect a non connectable handle. "
		 "Check this out...");
      return FALSE;
    }
  else if (handle->connected_to)
    {
      g_warning ("Trying to connect a connected handle!!!");
      return FALSE;
    }
  else if (dia_object_is_connected_to (handle->object,
				       connectionpoint->object))
    {
      g_warning ("Objects are already connected... They will not be"
		 " connected to avoid cyclic references!");
      return FALSE;
    }
  else
    {
      handle->connected_to = connectionpoint;
      connectionpoint->connected =
	g_list_prepend (connectionpoint->connected, handle);
      
      handle->pos = connectionpoint->pos;
      return TRUE;
    }
}

void
dia_handle_disconnect (DiaHandle *handle)
{
  DiaObject *obj;
  
  g_return_if_fail (handle != NULL);
  
  if (handle->connected_to != NULL)
    {
      obj = handle->connected_to->object;
      dia_object_cp_disconnect (obj, handle->connected_to, handle);
    }
}

void
dia_handle_free (DiaHandle *handle)
{
  DiaObject *object;
  
  g_return_if_fail (handle != NULL);
  g_return_if_fail (handle->connected_to == NULL);
  g_return_if_fail (handle->object != NULL);
  
/*   dia_handle_unconnect (handle); */
  object = handle->object;
  
  if (g_ptr_array_remove (object->handles, handle))
    g_free (handle);
  else
    g_error ("dia_handle_free: could not find DiaHandle!");
}


/* CONNECTION POINT
 */
void
dia_connection_point_move (DiaConnectionPoint *conpoint,
			   gfloat dx, gfloat dy, DiaRequest *request)
{
  g_return_if_fail (conpoint != NULL);
  
  *request = DIA_REQUEST_NOTHING;
  conpoint->pos.x += dx;
  conpoint->pos.y += dy;
}

void
dia_connection_point_unconnect (DiaConnectionPoint *conpoint)
{
  DiaObject *obj;
  
  g_return_if_fail (conpoint != NULL);
  
  obj = conpoint->object;
  dia_object_cp_disconnect (obj, conpoint, NULL);
}

void
dia_connection_point_free (DiaConnectionPoint *conpoint)
{
  DiaObject *object;
  
  g_return_if_fail (conpoint != NULL);
  g_return_if_fail (conpoint->connected == NULL);
  
  object = conpoint->object;
  
  /*  dia_connection_point_unconnect (conpoint); */
  
  if (g_ptr_array_remove (object->connections, conpoint))
    g_free (conpoint);
  else
    g_error ("dia_connection_point_free: Could not find ConnectionPoint "
	       "to remove!");
}

/* events
 */
static void
destroy (DiaObject *object)
{
  gint i;
  
  g_return_if_fail (object != NULL);
  
  dia_object_unconnect_all (object);

  for (i=0; i < object->connections->len; i++)
    dia_connection_point_free (DIA_OBJECT_GET_CP (object, i));
  g_ptr_array_free (object->connections, FALSE);
  
  for (i=0; i < object->handles->len; i++)
    dia_handle_free (DIA_OBJECT_GET_HANDLE (object, i));
  g_ptr_array_free (object->handles, FALSE);
  
  g_free (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  gint i;
  DiaColor *color;
  DiaHandle *handle;
  
  for (i = 0; i < obj->handles->len; i++)
    {
      handle = (DiaHandle*) g_ptr_array_index (obj->handles, i);
      
      if (DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_FOCUSED))
	{
	  if (handle->connected_to)
	    color = &dia_handle_colors[7]; 
	  else if  (handle->could_be_connected)
	    color = &dia_handle_colors[6];
	  else if (handle->is_movable)
	    color = &dia_handle_colors[5]; 
	  else
	    color = &dia_handle_colors[4]; 
	}
      else
	{
	  if (handle->connected_to)
	    color = &dia_handle_colors[3]; 
	  else if  (handle->could_be_connected)
	    color = &dia_handle_colors[2];
	  else if (handle->is_movable)
	    color = &dia_handle_colors[1]; 
	  else
	    color = &dia_handle_colors[0]; 
	}
      
      renderer->ops->draw_handle (renderer,
				  &handle->pos,
				  color,
				  handle->is_connectable);
    }
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  DiaColor cp_color = {0.0, 0.0, 1.0};
  gint i;
  
  //renderer->ops->set_origin (renderer, obj->position.x, obj->position.y);

  for (i=0; i < obj->connections->len; i++)
    {
      //renderer->ops->set_origin (renderer, obj->position.x, 
      //			 obj->position.y);
      renderer->ops->draw_cp (renderer,
			      &((DiaConnectionPoint*)
				g_ptr_array_index (obj->connections, i))->pos,
			      &cp_color);
    }
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return distance_rectangle_point (&obj->bounding_box, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  gint i;
  
  //g_print ("dia_object_move: dx = %f, dy = %f\n", dx, dy);
  obj->update_box = obj->bounding_box;
  
  obj->bounding_box.left += dx;
  obj->bounding_box.right += dx;
  obj->bounding_box.top += dy;
  obj->bounding_box.bottom += dy;
  
  obj->snap_pos.x += dx;
  obj->snap_pos.y += dy;
  
  for (i = 0; i < obj->handles->len; i++)
    {
      DIA_OBJECT_GET_HANDLE (obj, i)->pos.x += dx;
      DIA_OBJECT_GET_HANDLE (obj, i)->pos.y += dy;
    }
  for (i = 0; i < obj->connections->len; i++)
    {
      DIA_OBJECT_GET_CP (obj, i)->pos.x += dx;
      DIA_OBJECT_GET_CP (obj, i)->pos.y += dy;
    }
 
  rectangle_union (&obj->update_box, &obj->bounding_box);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle,
	     gfloat dx, gfloat dy)
{
  handle->pos.x += dx;
  handle->pos.y += dy;
  
  dia_object_calc_bounding_box_update (obj);
}


static void
copy (DiaObject *from, DiaObject *to)
{
  gint i;
  DiaHandle *handle;
  DiaHandle *new_handle;
  DiaConnectionPoint *cp;
  DiaConnectionPoint *new_cp;

  g_return_if_fail ( to->handles->len == from->handles->len );
  g_return_if_fail ( to->connections->len == from->connections->len );
   
  to->object_type = from->object_type;
  to->ops = from->ops;
  
  /* do not copy state since it depends on the display it is drawn in */
  to->flags = 0;
  to->bounding_box = from->bounding_box;
  to->snap_pos = from->snap_pos;
  to->request = from->request;
  to->update_box = from->update_box;
  to->cursor = from->cursor;
  
  for (i = 0; i < from->handles->len; i++)
    {
      handle = DIA_OBJECT_GET_HANDLE (from, i);
      new_handle = DIA_OBJECT_GET_HANDLE (to, i);
      new_handle = dia_object_add_handle (to, handle->pos.x, handle->pos.y);

      new_handle->keep_on_move = handle->keep_on_move;
      new_handle->is_connectable = handle->is_connectable;
      new_handle->is_movable = handle->is_movable;
      new_handle->could_be_connected = handle->could_be_connected;
      /* do not copy connections to handles and data */
    }
  
  for (i = 0; i < from->connections->len; i++)
    {
      cp = DIA_OBJECT_GET_CP (from, i);
      new_cp = DIA_OBJECT_GET_CP (to, i);
      //dia_object_add_connection_point (to, cp->pos.x, cp->pos.y);
      new_cp->pos = cp->pos;
      /* do not copy connections to handles and data */
    }
  to->data = NULL;
}

static gint
is_empty (DiaObject* obj)
{
  return TRUE;
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  return FALSE;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos,
	     Point *con_pos, DiaConnectionPoint **cp)
{
  gfloat dist;
  gfloat min_dist = G_MAXFLOAT;
  gint i;

  *cp = NULL;

  for (i = 0; i < obj->connections->len; i++)
    {
      dist = distance_point_point (&DIA_OBJECT_GET_CP (obj, i)->pos, pos);
      if (dist < min_dist)
	{
	  *cp = DIA_OBJECT_GET_CP (obj, i);
	  *con_pos = (*cp)->pos;
	  min_dist = dist;
	}
    }

  return min_dist;
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  DiaConnectionPoint *cp;
  Point con_pos;
  
  dia_object_cp_distance (obj, pos, &con_pos, &cp);
  
  if (cp)
    dia_handle_connect (h, cp);

  return cp;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  GList *l;
  //gint i;
  
  if (h && g_list_find (cp->connected, h))
    {
      //i = g_list_length (cp->connected);
      cp->connected = g_list_remove (cp->connected, h);
      h->connected_to = NULL;
    }
  else
    if (h == NULL)
      {
	l = cp->connected;
	while (l)
	  {
	    ((DiaHandle*) l->data)->connected_to = NULL;
	    l = g_list_next (l);
	  }
	g_list_free (cp->connected);
      }
    else
      g_warning ("Could not find handle on connection point while disconnecting!");
}








