/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <time.h>
#include <math.h>
#include <unistd.h>
#include "../config.h"
//#include "intl.h"
#include "diarenderereps.h"

static void begin_render (DiaRendererEps *renderer);
static void end_render (DiaRendererEps *renderer);
static void set_origin (DiaRendererEps *renderer, gfloat x, gfloat y);
static void set_linewidth (DiaRendererEps *renderer, gfloat linewidth);
static void set_linecaps (DiaRendererEps *renderer, DiaLineCaps mode);
static void set_linejoin (DiaRendererEps *renderer, DiaLineJoin mode);
static void set_linestyle (DiaRendererEps *renderer, DiaLineStyle mode);
static void set_dashlength (DiaRendererEps *renderer, gfloat length);
static void set_fillstyle (DiaRendererEps *renderer, DiaFillStyle mode);
static void set_font (DiaRendererEps *renderer, DiaFont *font, gfloat height);
static void draw_line (DiaRendererEps *renderer, 
		      Point *start, Point *end, 
		      DiaColor *line_color);
static void draw_polyline(DiaRendererEps *renderer, 
			  Point *points, gint num_points, 
			  DiaColor *line_color);
static void draw_polygon(DiaRendererEps *renderer, 
			 Point *points, gint num_points, 
			 DiaColor *line_color);
static void fill_polygon(DiaRendererEps *renderer, 
			 Point *points, gint num_points, 
			 DiaColor *line_color);
static void draw_rect(DiaRendererEps *renderer, 
		      Point *ul_corner, Point *lr_corner,
		      DiaColor *color);
static void fill_rect(DiaRendererEps *renderer, 
		      Point *ul_corner, Point *lr_corner,
		      DiaColor *color);
static void draw_arc(DiaRendererEps *renderer, 
		     Point *center,
		     gfloat width, gfloat height,
		     gfloat angle1, gfloat angle2,
		     DiaColor *color);
static void fill_arc(DiaRendererEps *renderer, 
		     Point *center,
		     gfloat width, gfloat height,
		     gfloat angle1, gfloat angle2,
		     DiaColor *color);
static void draw_ellipse(DiaRendererEps *renderer, 
			 Point *center,
			 gfloat width, gfloat height,
			 DiaColor *color);
static void fill_ellipse(DiaRendererEps *renderer, 
			 Point *center,
			 gfloat width, gfloat height,
			 DiaColor *color);
static void draw_bezier(DiaRendererEps *renderer, 
			Point *points,
			int numpoints, /* numpoints = 4+3*n, n=>0 */
			DiaColor *color);
static void fill_bezier(DiaRendererEps *renderer, 
			Point *points, /* Last point must be same as first point */
			int numpoints, /* numpoints = 4+3*n, n=>0 */
			DiaColor *color);
static void draw_string(DiaRendererEps *renderer,
			const char *text,
			Point *pos,
/* 			DiaHAlignment halign, */
/* 			DiaVAlignment valign, */
			DiaColor *color);
static void draw_image(DiaRendererEps *renderer,
		       Point *point,
		       gfloat width, gfloat height,
		       DiaImage *image);
static void draw_handle (DiaRendererEps *renderer, Point *position,
			 DiaColor *color, gboolean is_connectable);
static void draw_cp (DiaRendererEps *renderer, Point *position,
		     DiaColor *color);

static DiaRenderOps EpsRenderOps = {
  (DiaBeginRenderFunc) begin_render,
  (DiaEndRenderFunc) end_render,

  (DiaSetOriginFunc) set_origin,

  (DiaSetLineWidthFunc) set_linewidth,
  (DiaSetLineCapsFunc) set_linecaps,
  (DiaSetLineJoinFunc) set_linejoin,
  (DiaSetLineStyleFunc) set_linestyle,
  (DiaSetDashLengthFunc) set_dashlength,
  (DiaSetFillStyleFunc) set_fillstyle,
  (DiaSetFontFunc) set_font,
  
  (DiaDrawLineFunc) draw_line,
  (DiaDrawPolyLineFunc) draw_polyline,
  
  (DiaDrawPolygonFunc) draw_polygon,
  (DiaFillPolygonFunc) fill_polygon,

  (DiaDrawRectangleFunc) draw_rect,
  (DiaFillRectangleFunc) fill_rect,

  (DiaDrawArcFunc) draw_arc,
  (DiaFillArcFunc) fill_arc,

  (DiaDrawEllipseFunc) draw_ellipse,
  (DiaFillEllipseFunc) fill_ellipse,

  (DiaDrawBezierFunc) draw_bezier,
  (DiaFillBezierFunc) fill_bezier,

  (DiaDrawStringFunc) draw_string,

  (DiaDrawImageFunc) draw_image,

  (DiaDrawHandleFunc) draw_handle,
  (DiaDrawConnectionPointFunc) draw_cp
};

DiaRendererEps *
dia_renderer_eps_new (DiaDiagram *dia, gchar *filename)
{
  return dia_renderer_eps_new_scale (dia, filename, 1.0);
}

DiaRendererEps *
dia_renderer_eps_new_scale (DiaDiagram *dia, gchar *filename, gdouble scale)
{
  DiaRendererEps *renderer;
  FILE *file;
  time_t time_now;
  const gdouble scale_factor = 28.346;
  
  Rectangle *extent;
  gchar *name;
 
  g_return_val_if_fail (dia != NULL, NULL);
  g_return_val_if_fail (filename != NULL, NULL);
  
  scale *= scale_factor;
  
  file = fopen(filename, "w");

  if (file==NULL) {
    g_warning ("Couldn't open: '%s' for writing.\n", filename);
    return NULL;
  }

  renderer = g_new (DiaRendererEps, 1);
  renderer->renderer.ops = &EpsRenderOps;
  renderer->renderer.is_interactive = 0;
  renderer->renderer.interactive_ops = NULL;

  renderer->file = file;

  renderer->dash_length = 1.0;
  renderer->dot_length = 0.2;
  renderer->saved_line_style = DIA_LINE_STYLE_SOLID;
  
  time_now  = time (NULL);
  extent = &dia->extents;
  
  name = getlogin ();
  if (name==NULL)
    name = "a user";
  
  fprintf (file,
	   "%%!PS-Adobe-2.0 EPSF-2.0\n"
	   "%%%%Title: %s\n"
	   "%%%%Creator: %s v%s\n"
	   "%%%%CreationDate: %s"
	   "%%%%For: %s\n"
	   "%%%%Magnification: 1.0000\n"
	   "%%%%Orientation: Portrait\n"
	   "%%%%BoundingBox: 0 0 %d %d\n" 
	   "%%%%Pages: 1\n"
	   "%%%%BeginSetup\n"
	   "%%%%EndSetup\n"
	   "%%%%EndComments\n",
	   dia->name,
	   PACKAGE, VERSION,
	   ctime (&time_now),
	   name,
	   (int) ceil ((extent->right - extent->left)*scale),
	   (int) ceil ((extent->bottom - extent->top)*scale));

  fprintf (file,
	   "/cp {closepath} bind def\n"
	   "/c {curveto} bind def\n"
	   "/f {fill} bind def\n"
	   "/a {arc} bind def\n"
	   "/ef {eofill} bind def\n"
	   "/ex {exch} bind def\n"
	   "/gr {grestore} bind def\n"
	   "/gs {gsave} bind def\n"
	   "/sa {save} bind def\n"
	   "/rs {restore} bind def\n"
	   "/l {lineto} bind def\n"
	   "/m {moveto} bind def\n"
	   "/rm {rmoveto} bind def\n"
	   "/n {newpath} bind def\n"
	   "/s {stroke} bind def\n"
	   "/sh {show} bind def\n"
	   "/slc {setlinecap} bind def\n"
	   "/slj {setlinejoin} bind def\n"
	   "/slw {setlinewidth} bind def\n"
	   "/srgb {setrgbcolor} bind def\n"
	   "/rot {rotate} bind def\n"
	   "/sc {scale} bind def\n"
	   "/sd {setdash} bind def\n"
	   "/ff {findfont} bind def\n"
	   "/sf {setfont} bind def\n"
	   "/scf {scalefont} bind def\n"
	   "/strw {stringwidth pop} bind def\n"
	   "/strh {stringheight pop} bind def\n"
	   "/tr {translate} bind def\n"

	   "\n/ellipsedict 8 dict def\n"
	   "ellipsedict /mtrx matrix put\n"
	   "/ellipse\n"
	   "{ ellipsedict begin\n"
	   "   /endangle exch def\n"
	   "   /startangle exch def\n"
	   "   /yrad exch def\n"
	   "   /xrad exch def\n"
	   "   /y exch def\n"
	   "   /x exch def"
	   "   /savematrix mtrx currentmatrix def\n"
	   "   x y tr xrad yrad sc\n"
	   "   0 0 1 startangle endangle arc\n"
	   "   savematrix setmatrix\n"
	   "   end\n"
	   "} def\n\n"


	   "/colortogray {\n"
	   "/rgbdata exch store\n"
	   "rgbdata length 3 idiv\n"
	   "/npixls exch store\n"
	   "/rgbindx 0 store\n"
	   "0 1 npixls 1 sub {\n"
	   "grays exch\n"
	   "rgbdata rgbindx       get 20 mul\n"
	   "rgbdata rgbindx 1 add get 32 mul\n"
	   "rgbdata rgbindx 2 add get 12 mul\n"
	   "add add 64 idiv\n"
	   "put\n"
	   "/rgbindx rgbindx 3 add store\n"
	   "} for\n"
	   "grays 0 npixls getinterval\n"
	   "} bind def\n"
	  
	   "/mergeprocs {\n"
	   "dup length\n"
	   "3 -1 roll\n"
	   "dup\n"
	   "length\n"
	   "dup\n"
	   "5 1 roll\n"
	   "3 -1 roll\n"
	   "add\n"
	   "array cvx\n"
	   "dup\n"
	   "3 -1 roll\n"
	   "0 exch\n"
	   "putinterval\n"
	   "dup\n"
	   "4 2 roll\n"
	   "putinterval\n"
	   "} bind def\n"
	  
	   "/colorimage {\n"
	   "pop pop\n"
	   "{colortogray} mergeprocs\n"
	   "image\n"
	   "} bind def\n\n"
	   
	   "%f %f scale\n"
	   "%f %f translate\n"
	   "%%%%EndProlog\n\n\n",
	   scale, -scale,
	   -extent->left, -extent->bottom );
  
  return renderer;
}

void
dia_renderer_eps_destroy (DiaRendererEps *renderer)
{  
  g_return_if_fail (renderer != NULL);
  
  fclose (renderer->file);

  g_free (renderer);
}

#define OX (double)((DiaRenderer*)renderer)->origin.x
#define OY (double)((DiaRenderer*)renderer)->origin.y

static void
begin_render (DiaRendererEps *renderer)
{
}

static void
end_render (DiaRendererEps *renderer)
{
  fprintf (renderer->file, "showpage\n");
}

static void
set_origin (DiaRendererEps *renderer, gfloat x, gfloat y)
{
  ((DiaRenderer*)renderer)->origin.x = x;
  ((DiaRenderer*)renderer)->origin.y = y;
}

static void
set_linewidth(DiaRendererEps *renderer, gfloat linewidth)
{  /* 0 == hairline **/

  fprintf(renderer->file, "%f slw\n", (double) linewidth);
}

static void
set_linecaps(DiaRendererEps *renderer, DiaLineCaps mode)
{
  int ps_mode;
  
  switch(mode) {
  case DIA_LINE_CAPS_BUTT:
    ps_mode = 0;
    break;
  case DIA_LINE_CAPS_ROUND:
    ps_mode = 1;
    break;
  case DIA_LINE_CAPS_PROJECTING:
    ps_mode = 2;
    break;
  default:
    ps_mode = 0;
  }

  fprintf(renderer->file, "%d slc\n", ps_mode);
}

static void
set_linejoin(DiaRendererEps *renderer, DiaLineJoin mode)
{
  int ps_mode;
  
  switch(mode) {
  case DIA_LINE_JOIN_MITER:
    ps_mode = 0;
    break;
  case DIA_LINE_JOIN_ROUND:
    ps_mode = 1;
    break;
  case DIA_LINE_JOIN_BEVEL:
    ps_mode = 2;
    break;
  default:
    ps_mode = 0;
  }

  fprintf(renderer->file, "%d slj\n", ps_mode);
}

static void
set_linestyle(DiaRendererEps *renderer, DiaLineStyle mode)
{
  gfloat hole_width;

  renderer->saved_line_style = mode;
  
  switch(mode) {
  case  DIA_LINE_STYLE_SOLID:
    fprintf(renderer->file, "[] 0 sd\n");
    break;
  case DIA_LINE_STYLE_DASHED:
    fprintf(renderer->file, "[%f] 0 sd\n", renderer->dash_length);
    break;
  case DIA_LINE_STYLE_DASH_DOT:
    hole_width = (renderer->dash_length - renderer->dot_length) / 2.0;
    fprintf(renderer->file, "[%f %f %f %f] 0 sd\n",
	    renderer->dash_length,
	    hole_width,
	    renderer->dot_length,
	    hole_width );
    break;
  case DIA_LINE_STYLE_DASH_DOT_DOT:
    hole_width = (renderer->dash_length - 2.0*renderer->dot_length) / 3.0;
    fprintf(renderer->file, "[%f %f %f %f %f %f] 0 sd\n",
	    renderer->dash_length,
	    hole_width,
	    renderer->dot_length,
	    hole_width,
	    renderer->dot_length,
	    hole_width );
    break;
  case  DIA_LINE_STYLE_DOTTED:
    fprintf(renderer->file, "[%f] 0 sd\n", renderer->dot_length);
    break;
  }
}

static void
set_dashlength(DiaRendererEps *renderer, gfloat length)
{  /* dot = 20% of len */
  
  renderer->dash_length = length;
  renderer->dot_length = length*0.2;
  
  set_linestyle(renderer, renderer->saved_line_style);
}

static void
set_fillstyle(DiaRendererEps *renderer, DiaFillStyle mode)
{
  switch(mode) {
  case DIA_FILL_STYLE_SOLID:
    break;
  default:
    g_warning ("eps_renderer: Unsupported fill mode specified!\n");
  }
}

static void
set_font(DiaRendererEps *renderer, DiaFont *font, gfloat height)
{
  fprintf(renderer->file, "/%s ff %f scf sf\n",
	  dia_font_get_psfontname(font), (double)height);
}

static void
draw_line(DiaRendererEps *renderer, 
	  Point *start, Point *end, 
	  DiaColor *line_color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) line_color->red, (double) line_color->green, (double) line_color->blue);

  fprintf(renderer->file, "n %f %f m %f %f l s\n",
	  OX +(double) start->x, OY + (double)start->y,
	  OX + (double)end->x, OY + (double)end->y);
}

static void
draw_polyline(DiaRendererEps *renderer, 
	      Point *points, gint num_points, 
	      DiaColor *line_color)
{
  int i;
  
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) line_color->red, (double) line_color->green, (double) line_color->blue);
  
  fprintf(renderer->file, "n %f %f m ",
	  (double)points[0].x + OX, (double)points[0].y + OY);

  for (i=1;i<num_points;i++) {
    fprintf(renderer->file, "%f %f l ",
	  (double)points[i].x + OX, (double)points[i].y + OY);
  }

  fprintf(renderer->file, "s\n");
}

static void
draw_polygon(DiaRendererEps *renderer, 
	      Point *points, int num_points, 
	      DiaColor *line_color)
{
  int i;
  
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) line_color->red, (double) line_color->green, (double) line_color->blue);
  
  fprintf(renderer->file, "n %f %f m ",
	  (double)points[0].x + OX, (double)points[0].y + OY);

  for (i=1;i<num_points;i++) {
    fprintf(renderer->file, "%f %f l ",
	  (double)points[i].x + OX, (double)points[i].y + OY);
  }

  fprintf(renderer->file, "cp s\n");
}

static void
fill_polygon(DiaRendererEps *renderer, 
	      Point *points, int num_points, 
	      DiaColor *line_color)
{
  int i;
  
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) line_color->red, (double) line_color->green, (double) line_color->blue);
  
  fprintf(renderer->file, "n %f %f m ",
	  (double)points[0].x + OX, (double)points[0].y + OY);

  for (i=1;i<num_points;i++) {
    fprintf(renderer->file, "%f %f l ",
	  (double)points[i].x + OX, (double)points[i].y + OY);
  }

  fprintf(renderer->file, "f\n");
}

static void
draw_rect(DiaRendererEps *renderer, 
	  Point *ul_corner, Point *lr_corner,
	  DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);
  
  fprintf(renderer->file, "n %f %f m %f %f l %f %f l %f %f l cp s\n",
	  (double) ul_corner->x + OX, (double) ul_corner->y + OY,
	  (double) ul_corner->x + OX, (double) lr_corner->y + OY,
	  (double) lr_corner->x + OX, (double) lr_corner->y + OY,
	  (double) lr_corner->x + OX, (double) ul_corner->y + OY);

}

static void
fill_rect(DiaRendererEps *renderer, 
	  Point *ul_corner, Point *lr_corner,
	  DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f m %f %f l %f %f l %f %f l f\n",
	  (double) ul_corner->x + OX, (double) ul_corner->y + OY,
	  (double) ul_corner->x + OX, (double) lr_corner->y + OY,
	  (double) lr_corner->x + OX, (double) lr_corner->y + OY,
	  (double) lr_corner->x + OX, (double) ul_corner->y + OY );
}

static void
draw_arc(DiaRendererEps *renderer, 
	 Point *center,
	 gfloat width, gfloat height,
	 gfloat angle1, gfloat angle2,
	 DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f %f %f %f %f ellipse s\n",
	  (double) center->x + OX, (double) center->y + OY,
	  (double) width/2.0, (double) height/2.0,
	  (double) 360.0 - angle2, (double) 360.0 - angle1 );
}

static void
fill_arc(DiaRendererEps *renderer, 
	 Point *center,
	 gfloat width, gfloat height,
	 gfloat angle1, gfloat angle2,
	 DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f m %f %f %f %f %f %f ellipse f\n",
	  (double) center->x + OX, (double) center->y + OY,
	  (double) center->x + OX, (double) center->y + OY,
	  (double) width/2.0, (double) height/2.0,
	  (double) 360.0 - angle2, (double) 360.0 - angle1 );
}

static void
draw_ellipse(DiaRendererEps *renderer, 
	     Point *center,
	     gfloat width, gfloat height,
	     DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f %f %f 0 360 ellipse cp s\n",
	  (double) center->x + OX, (double) center->y + OY,
	  (double) width/2.0, (double) height/2.0 );
}

static void
fill_ellipse(DiaRendererEps *renderer, 
	     Point *center,
	     gfloat width, gfloat height,
	     DiaColor *color)
{
  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f %f %f 0 360 ellipse f\n",
	  (double) center->x + OX, (double) center->y + OY,
	  (double) width/2.0, (double) height/2.0 );
}

static void
draw_bezier(DiaRendererEps *renderer, 
	    Point *points,
	    int numpoints, /* numpoints = 4+3*n, n=>0 */
	    DiaColor *color)
{
  int i;

  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f m",
	  (double) points[0].x + OX, (double) points[0].y + OY);
  
  i = 1;
  while (i<=numpoints-3) {
  fprintf(renderer->file, " %f %f %f %f %f %f c",
	  (double) points[i].x + OX, (double) points[i].y + OY,
	  (double) points[i+1].x + OX, (double) points[i+1].y + OY,
	  (double) points[i+2].x + OX, (double) points[i+2].y + OY);
    i += 3;
  }

  fprintf(renderer->file, " s\n");
}

static void
fill_bezier(DiaRendererEps *renderer, 
	    Point *points, /* Last point must be same as first point */
	    int numpoints, /* numpoints = 4+3*n, n=>0 */
	    DiaColor *color)
{
  int i;

  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);

  fprintf(renderer->file, "n %f %f m",
	  (double) points[0].x + OX, (double) points[0].y + OY);
  
  i = 1;
  while (i<=numpoints-3) {
  fprintf(renderer->file, " %f %f %f %f %f %f c",
	  (double) points[i].x + OX, (double) points[i].y + OX,
	  (double) points[i+1].x + OX, (double) points[i+1].y + OX,
	  (double) points[i+2].x + OX, (double) points[i+2].y + OX );
    i += 3;
  }

  fprintf(renderer->file, " f\n");
}

static void
draw_string(DiaRendererEps *renderer,
	    const char *text,
	    Point *pos,
/* 	    DiaHAlignment halign, */
/* 	    DiaVAlignment valign, */
	    DiaColor *color)
{
  char *buffer;
  const char *str;
  int len;

  /* TODO: Use latin-1 encoding */

  fprintf(renderer->file, "%f %f %f srgb\n",
	  (double) color->red, (double) color->green, (double) color->blue);


  /* Escape all '(' and ')':  */
  buffer = g_malloc(2*strlen(text)+1);
  *buffer = 0;
  str = text;
  while (*str != 0) {
    len = strcspn(str,"()\\");
    strncat(buffer, str, len);
    str += len;
    if (*str != 0) {
      strcat(buffer,"\\");
      strncat(buffer, str, 1);
      str++;
    }
  }
  fprintf(renderer->file, "(%s) ", buffer);
  
  fprintf(renderer->file, "%f %f m", pos->x + OX, pos->y + OY); 

  /* Stack now contains the string */
/*   switch (halign) */
/*     { */
/*     case DIA_ALIGN_LEFT: */
/*       fprintf(renderer->file, "%f ", pos->x + OX); */
/*       break; */
/*     case DIA_ALIGN_CENTER: */
/*       fprintf(renderer->file, "dup strw 2 div %f ex sub ", pos->x + OX); */
/*       break; */
/*     case DIA_ALIGN_RIGHT: */
/*       fprintf(renderer->file, "dup strw %f ex sub ", pos->x + OX); */
/*       break; */
/*     } */

  /* Stack now contains the string and the X position */
/*   switch (valign) */
/*     { */
/*     case DIA_ALIGN_BOTTOM: */
/*       fprintf(renderer->file, "%f m", pos->y + OY); */
/*       break; */
/*     case DIA_ALIGN_MIDDLE: */
/*       fprintf(renderer->file, "%s strh 2 div %f ex sub m", buffer, pos->y + OY); */
/*       break; */
/*     case DIA_ALIGN_TOP: */
/*       fprintf(renderer->file, "%s strh %f ex sub m", buffer, pos->y + OY); */
/*       break; */
/*     } */
  
  fprintf(renderer->file, " gs 1 -1 sc sh gr\n");

  g_free(buffer);
}

static void
draw_image(DiaRendererEps *renderer,
	   Point *point,
	   gfloat width, gfloat height,
	   DiaImage *image)
{
  int img_width, img_height;
  int v;
  int                 x, y;
  unsigned char      *ptr;
  gfloat ratio;
  guint8 *rgb_data;

  img_width = dia_image_width(image);
  img_height = dia_image_height(image);

  rgb_data = dia_image_rgb_data(image);
  
  ratio = height/width;

  fprintf(renderer->file, "gs\n");
  if (1) { /* Color output */
    fprintf(renderer->file, "/pix %i string def\n", img_width * 3);
    fprintf(renderer->file, "/grays %i string def\n", img_width);
    fprintf(renderer->file, "/npixls 0 def\n");
    fprintf(renderer->file, "/rgbindx 0 def\n");
    fprintf(renderer->file, "%f %f tr\n", point->x + OX, point->y + OX);
    fprintf(renderer->file, "%f %f sc\n", width, height);
    fprintf(renderer->file, "%i %i 8\n", img_width, img_height);
    fprintf(renderer->file, "[%i 0 0 %i 0 0]\n", img_width, img_height);
    fprintf(renderer->file, "{currentfile pix readhexstring pop}\n");
    fprintf(renderer->file, "false 3 colorimage\n");
    fprintf(renderer->file, "\n");
    ptr = rgb_data;
    for (y = 0; y < img_width; y++) {
      for (x = 0; x < img_height; x++) {
	fprintf(renderer->file, "%02x", (int)(*ptr++));
	fprintf(renderer->file, "%02x", (int)(*ptr++));
	fprintf(renderer->file, "%02x", (int)(*ptr++));
      }
      fprintf(renderer->file, "\n");
    }
  } else { /* Grayscale */
    fprintf(renderer->file, "/pix %i string def\n", img_width);
    fprintf(renderer->file, "/grays %i string def\n", img_width);
    fprintf(renderer->file, "/npixls 0 def\n");
    fprintf(renderer->file, "/rgbindx 0 def\n");
    fprintf(renderer->file, "%f %f tr\n", point->x + OX, point->y + OX);
    fprintf(renderer->file, "%f %f sc\n", width, height);
    fprintf(renderer->file, "%i %i 8\n", img_width, img_height);
    fprintf(renderer->file, "[%i 0 0 %i 0 0]\n", img_width, img_height);
    fprintf(renderer->file, "{currentfile pix readhexstring pop}\n");
    fprintf(renderer->file, "image\n");
    fprintf(renderer->file, "\n");
    ptr = rgb_data;
    for (y = 0; y < img_height; y++) {
      for (x = 0; x < img_width; x++) {
	v = (int)(*ptr++);
	v += (int)(*ptr++);
	v += (int)(*ptr++);
	v /= 3;
	fprintf(renderer->file, "%02x", v);
      }
      fprintf(renderer->file, "\n");
    }
  }
  /*  fprintf(renderer->file, "%f %f scale\n", 1.0, 1.0/ratio);*/
  fprintf(renderer->file, "gr\n");
  fprintf(renderer->file, "\n");
}

static void
draw_handle (DiaRendererEps *renderer, Point *position, DiaColor *color,
	     gboolean is_connectable)
{
  /* No handles! :-) */
}

static void
draw_cp (DiaRendererEps *renderer, Point *position, DiaColor *color)
{
  /* No cp's! :-) */
}









