/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <gdk/gdk.h>

#include "diarenderergdk.h"


static void begin_render (DiaRendererGdk *renderer, DiaLayer *layer);
static void end_render (DiaRendererGdk *renderer);
static void set_origin (DiaRendererGdk *renderer, gfloat x, gfloat y);
static void set_linewidth (DiaRendererGdk *renderer, gfloat linewidth);
static void set_linecaps (DiaRendererGdk *renderer, DiaLineCaps mode);
static void set_linejoin (DiaRendererGdk *renderer, DiaLineJoin mode);
static void set_linestyle (DiaRendererGdk *renderer, DiaLineStyle mode);
static void set_dashlength (DiaRendererGdk *renderer, gfloat length);
static void set_fillstyle (DiaRendererGdk *renderer, DiaFillStyle mode);
static void set_font (DiaRendererGdk *renderer, DiaFont *font, gfloat height);
static void draw_line (DiaRendererGdk *renderer, 
		       Point *start, Point *end, 
		      DiaColor *line_color);
static void draw_polyline (DiaRendererGdk *renderer, 
			   Point *points, gint num_points, 
			   DiaColor *line_color);
static void draw_polygon (DiaRendererGdk *renderer, 
			  Point *points, gint num_points, 
			  DiaColor *line_color);
static void fill_polygon (DiaRendererGdk *renderer, 
			  Point *points, gint num_points, 
			  DiaColor *line_color);
static void draw_rect (DiaRendererGdk *renderer, 
		       Point *ul_corner, Point *lr_corner,
		       DiaColor *color);
static void fill_rect (DiaRendererGdk *renderer, 
		       Point *ul_corner, Point *lr_corner,
		       DiaColor *color);
static void draw_arc (DiaRendererGdk *renderer, 
		      Point *center,
		      gfloat width, gfloat height,
		      gfloat angle1, gfloat angle2,
		      DiaColor *color);
static void fill_arc (DiaRendererGdk *renderer, 
		      Point *center,
		      gfloat width, gfloat height,
		      gfloat angle1, gfloat angle2,
		      DiaColor *color);
static void draw_ellipse (DiaRendererGdk *renderer, 
			  Point *center,
			  gfloat width, gfloat height,
			  DiaColor *color);
static void fill_ellipse (DiaRendererGdk *renderer, 
			  Point *center,
			  gfloat width, gfloat height,
			  DiaColor *color);
static void draw_bezier (DiaRendererGdk *renderer, 
			 Point *points,
			 gint numpoints, /* numpoints = 4+3*n, n=>0 */
			 DiaColor *color);
static void fill_bezier (DiaRendererGdk *renderer, 
			 Point *points, /* Last point must be same as first point */
			 int numpoints, /* numpoints = 4+3*n, n=>0 */
			 DiaColor *color);
static void draw_string (DiaRendererGdk *renderer,
			 const char *text,
			 Point *pos,
/* 			 DiaHAlignment halign, */
/* 			 DiaVAlignment valign, */
			 DiaColor *color);
static void draw_image (DiaRendererGdk *renderer,
			Point *point,
			gfloat width, gfloat height,
			DiaImage *image);
static void draw_handle (DiaRendererGdk *renderer, Point *pos, DiaColor *color,
			 gboolean is_connectable);
static void draw_cp (DiaRendererGdk *renderer, Point *pos, DiaColor *color);
static void get_handle_bb (DiaRendererGdk *renderer, Point *pos,
			   Rectangle *bb);
static void get_cp_bb (DiaRendererGdk *renderer, Point *pos, Rectangle *bb);

static gfloat get_text_width (DiaRendererGdk *renderer,
			    const char *text, int length);

static void clip_region_clear (DiaRendererGdk *renderer);
static void clip_region_add_rect (DiaRendererGdk *renderer,
				  Rectangle *rect);

static void draw_pixel_line (DiaRendererGdk *renderer,
			     gint x1, gint y1,
			     gint x2, gint y2,
			     DiaColor *color);
static void draw_pixel_rect (DiaRendererGdk *renderer,
			     gint x, gint y,
			     gint width, gint height,
			     DiaColor *color);
static void fill_pixel_rect (DiaRendererGdk *renderer,
			     gint x, gint y,
			     gint width, gint height,
			     DiaColor *color);

static DiaRenderOps GdkRenderOps = {
  (DiaBeginRenderFunc) begin_render,
  (DiaEndRenderFunc) end_render,
  (DiaSetOriginFunc) set_origin,
  (DiaSetLineWidthFunc) set_linewidth,
  (DiaSetLineCapsFunc) set_linecaps,
  (DiaSetLineJoinFunc) set_linejoin,
  (DiaSetLineStyleFunc) set_linestyle,
  (DiaSetDashLengthFunc) set_dashlength,
  (DiaSetFillStyleFunc) set_fillstyle,
  (DiaSetFontFunc) set_font,
  
  (DiaDrawLineFunc) draw_line,
  (DiaDrawPolyLineFunc) draw_polyline,
  
  (DiaDrawPolygonFunc) draw_polygon,
  (DiaFillPolygonFunc) fill_polygon,

  (DiaDrawRectangleFunc) draw_rect,
  (DiaFillRectangleFunc) fill_rect,

  (DiaDrawArcFunc) draw_arc,
  (DiaFillArcFunc) fill_arc,

  (DiaDrawEllipseFunc) draw_ellipse,
  (DiaFillEllipseFunc) fill_ellipse,

  (DiaDrawBezierFunc) draw_bezier,
  (DiaFillBezierFunc) fill_bezier,

  (DiaDrawStringFunc) draw_string,

  (DiaDrawImageFunc) draw_image,

  (DiaDrawHandleFunc) draw_handle,
  (DiaDrawConnectionPointFunc) draw_cp,
};

static DiaInteractiveRenderOps GdkInteractiveRenderOps = 
{
  (DiaGetTextWidthFunc) get_text_width,

  (DiaGetHandleBoundingBoxFunc) get_handle_bb,
  (DiaGetCPBoundingBoxFunc) get_cp_bb,

  (DiaClipRegionClearFunc) clip_region_clear,
  (DiaClipRegionAddRectangleFunc) clip_region_add_rect,

  (DiaDrawPixelLineFunc) draw_pixel_line,
  (DiaDrawPixelRectangleFunc) draw_pixel_rect,
  (DiaFillPixelRectangleFunc) fill_pixel_rect,
};

DiaRendererGdk*
dia_renderer_gdk_new (DiaDisplay *ddisp)
{
  DiaRendererGdk *renderer;

  renderer = g_new (DiaRendererGdk, 1);
  renderer->renderer.ops = &GdkRenderOps;
  renderer->renderer.origin.x = 0.0;
  renderer->renderer.origin.y = 0.0;
  renderer->renderer.is_interactive = 1;
  renderer->renderer.interactive_ops = &GdkInteractiveRenderOps;
  renderer->ddisp = ddisp;
  renderer->render_gc = NULL;

  renderer->pixmap = NULL;
  renderer->renderer.pixel_width = 0;
  renderer->renderer.pixel_height = 0;
  renderer->clip_region = NULL;

  renderer->line_width = 1;
  renderer->line_style = GDK_LINE_SOLID;
  renderer->cap_style = GDK_CAP_BUTT;
  renderer->join_style = GDK_JOIN_MITER;
  
  renderer->saved_line_style = DIA_LINE_STYLE_SOLID;
  renderer->dash_length = 10;
  renderer->dot_length = 2;
  
  return renderer;
}

void
dia_renderer_gdk_destroy (DiaRendererGdk *renderer)
{
  if (renderer->pixmap != NULL)
    gdk_pixmap_unref(renderer->pixmap);

  if (renderer->render_gc != NULL)
    gdk_gc_unref(renderer->render_gc);

  if (renderer->clip_region != NULL)
    gdk_region_destroy(renderer->clip_region);

  g_free (renderer);
}

void
dia_renderer_gdk_set_size (DiaRendererGdk *renderer, GdkWindow *window,
			   gint width, gint height)
{
  
  if (renderer->pixmap != NULL) {
    gdk_pixmap_unref(renderer->pixmap);
  }

  renderer->pixmap = gdk_pixmap_new (window,  width, height, -1);
  renderer->renderer.pixel_width = width;
  renderer->renderer.pixel_height = height;

  if (renderer->render_gc == NULL) {
    renderer->render_gc = gdk_gc_new (renderer->pixmap);

    gdk_gc_set_line_attributes (renderer->render_gc,
				renderer->line_width,
				renderer->line_style,
				renderer->cap_style,
				renderer->join_style);
  }
}

void
dia_renderer_gdk_copy_to_window (DiaRendererGdk *renderer, GdkWindow *window,
				 gint x, gint y, gint width, gint height)
{
  static GdkGC *copy_gc = NULL;
  
  if (copy_gc == NULL)
    {
      copy_gc = gdk_gc_new (window);
    }
  
  /* check canvas sizes if auto-resize is turned off: */
  if (renderer->ddisp->diagram && !renderer->ddisp->diagram->auto_resize)
    {
      Rectangle extents;
      gint tx, ty, tw, th;
      
      extents = renderer->ddisp->diagram->extents;
      rectangle_intersection (&extents, &renderer->ddisp->visible);
      
      dia_display_transform_coords (renderer->ddisp,
				    extents.left, extents.top, &tx, &ty);
      
      tw = dia_display_transform_length (renderer->ddisp,
					 extents.right - extents.left);
      th = dia_display_transform_length (renderer->ddisp,
					 extents.bottom - extents.top);
    
      /* The background is cleaned when the zoom factor changes
       * ( dia_display_zoom () ). */

      if (tx > x)
	x = tx;
      if (ty > y)
	y = ty;

      tw -= x - tx;
      th -= y - ty;
      
      if (tw < width)
	width = tw;
      if (th < height)
	height = th;
    }

  gdk_draw_pixmap (window,
		   copy_gc,
		   renderer->pixmap,
		   x, y, x, y, width, height);
}

static void
begin_render(DiaRendererGdk *renderer, DiaLayer *layer)
{
}

static void
end_render(DiaRendererGdk *renderer)
{
}

static void
set_origin (DiaRendererGdk *renderer, gfloat x, gfloat y)
{
  ((DiaRenderer*)renderer)->origin.x = x;
  ((DiaRenderer*)renderer)->origin.y = y;
}

static void
set_linewidth(DiaRendererGdk *renderer, gfloat linewidth)
{  /* 0 == hairline **/
  renderer->line_width =
    dia_display_transform_length(renderer->ddisp, linewidth);

  if (renderer->line_width<=0)
    renderer->line_width = 1; /* Minimum 1 pixel. */

  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     renderer->join_style);
}

static void
set_linecaps(DiaRendererGdk *renderer, DiaLineCaps mode)
{
  switch(mode) {
  case DIA_LINE_CAPS_BUTT:
    renderer->cap_style = GDK_CAP_BUTT;
    break;
  case DIA_LINE_CAPS_ROUND:
    renderer->cap_style = GDK_CAP_ROUND;
    break;
  case DIA_LINE_CAPS_PROJECTING:
    renderer->cap_style = GDK_CAP_PROJECTING;
    break;
  }
 
  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     renderer->join_style);
}

static void
set_linejoin(DiaRendererGdk *renderer, DiaLineJoin mode)
{
  switch(mode) {
  case DIA_LINE_JOIN_MITER:
    renderer->cap_style = GDK_JOIN_MITER;
    break;
  case DIA_LINE_JOIN_ROUND:
    renderer->cap_style = GDK_JOIN_ROUND;
    break;
  case DIA_LINE_JOIN_BEVEL:
    renderer->cap_style = GDK_JOIN_BEVEL;
    break;
  }
 
  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     renderer->join_style);
}

static void
set_linestyle(DiaRendererGdk *renderer, DiaLineStyle mode)
{
  char dash_list[6];
  int hole_width;
  
  renderer->saved_line_style = mode;
  switch(mode) {
  case DIA_LINE_STYLE_SOLID:
    renderer->line_style = GDK_LINE_SOLID;
    break;
  case DIA_LINE_STYLE_DASHED:
    renderer->line_style = GDK_LINE_ON_OFF_DASH;
    dash_list[0] = renderer->dash_length;
    dash_list[1] = renderer->dash_length;
    gdk_gc_set_dashes(renderer->render_gc, 0, dash_list, 2);
    break;
  case DIA_LINE_STYLE_DASH_DOT:
    renderer->line_style = GDK_LINE_ON_OFF_DASH;
    hole_width = (renderer->dash_length - renderer->dot_length) / 2;
    if (hole_width==0)
      hole_width = 1;
    dash_list[0] = renderer->dash_length;
    dash_list[1] = hole_width;
    dash_list[2] = renderer->dot_length;
    dash_list[3] = hole_width;
    gdk_gc_set_dashes(renderer->render_gc, 0, dash_list, 4);
    break;
  case DIA_LINE_STYLE_DASH_DOT_DOT:
    renderer->line_style = GDK_LINE_ON_OFF_DASH;
    hole_width = (renderer->dash_length - 2*renderer->dot_length) / 3;
    if (hole_width==0)
      hole_width = 1;
    dash_list[0] = renderer->dash_length;
    dash_list[1] = hole_width;
    dash_list[2] = renderer->dot_length;
    dash_list[3] = hole_width;
    dash_list[4] = renderer->dot_length;
    dash_list[5] = hole_width;
    gdk_gc_set_dashes(renderer->render_gc, 0, dash_list, 6);
    break;
  case DIA_LINE_STYLE_DOTTED:
    renderer->line_style = GDK_LINE_ON_OFF_DASH;
    dash_list[0] = renderer->dot_length;
    dash_list[1] = renderer->dot_length;
    gdk_gc_set_dashes(renderer->render_gc, 0, dash_list, 2);
    break;
  }
  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     renderer->join_style);
}

static void
set_dashlength(DiaRendererGdk *renderer, gfloat length)
{  /* dot = 10% of len */
  gfloat ddisp_len;

  ddisp_len =
    dia_display_transform_length(renderer->ddisp, length);
  
  renderer->dash_length = (int)floor(ddisp_len+0.5);
  renderer->dot_length = (int)floor(ddisp_len*0.1+0.5);
  
  if (renderer->dash_length<=0)
    renderer->dash_length = 1;
  if (renderer->dash_length>255)
    renderer->dash_length = 255;
  if (renderer->dot_length<=0)
    renderer->dot_length = 1;
  if (renderer->dot_length>255)
    renderer->dot_length = 255;
  set_linestyle(renderer, renderer->saved_line_style);
}

static void
set_fillstyle(DiaRendererGdk *renderer, DiaFillStyle mode)
{
  switch(mode) {
  case DIA_FILL_STYLE_SOLID:
    break;
  default:
    g_warning ("DIA_RENDERER_GDK: Unsupported fill mode specified!\n");
  }
}

static void
set_font(DiaRendererGdk *renderer, DiaFont *font, gfloat height)
{
  renderer->font_height =
    dia_display_transform_length(renderer->ddisp, height);

  renderer->gdk_font = dia_font_get_gdkfont(font, renderer->font_height);
}

static void
draw_line(DiaRendererGdk *renderer, 
	  Point *start, Point *end, 
	  DiaColor *line_color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor color;
  gint x1,y1,x2,y2;
  
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + end->x, 
			       ((DiaRenderer*)renderer)->origin.y + end->y,
			       &x2, &y2);
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + start->x,
			       ((DiaRenderer*)renderer)->origin.y + start->y,
			       &x1, &y1);

  dia_color_convert(line_color, &color);
  gdk_gc_set_foreground(gc, &color);
  
  gdk_draw_line(renderer->pixmap, gc,
		x1, y1,	x2, y2);
}

static void
draw_polyline(DiaRendererGdk *renderer, 
	      Point *points, gint num_points, 
	      DiaColor *line_color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor color;
  GdkPoint *gdk_points;
  gint i,x,y;

  gdk_points = g_new(GdkPoint, num_points);

  for (i=0;i<num_points;i++) {
    dia_display_transform_coords(ddisp,
				 ((DiaRenderer*)renderer)->origin.x + points[i].x, 
				 ((DiaRenderer*)renderer)->origin.y + points[i].y, &x, &y);
    gdk_points[i].x = x;
    gdk_points[i].y = y;
  }
  
  dia_color_convert(line_color, &color);
  gdk_gc_set_foreground(gc, &color);
  
  gdk_draw_lines(renderer->pixmap, gc, gdk_points, num_points);
  g_free(gdk_points);
}

static void
draw_polygon(DiaRendererGdk *renderer, 
	      Point *points, int num_points, 
	      DiaColor *line_color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor color;
  GdkPoint *gdk_points;
  int i,x,y;
  
  gdk_points = g_new(GdkPoint, num_points);

  for (i=0;i<num_points;i++) {
    dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + points[i].x, 
				 ((DiaRenderer*)renderer)->origin.y + points[i].y, &x, &y);
    gdk_points[i].x = x;
    gdk_points[i].y = y;
  }
  
  dia_color_convert(line_color, &color);
  gdk_gc_set_foreground(gc, &color);
  
  gdk_draw_polygon(renderer->pixmap, gc, FALSE, gdk_points, num_points);
  g_free(gdk_points);
}

static void
fill_polygon(DiaRendererGdk *renderer, 
	      Point *points, int num_points, 
	      DiaColor *line_color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor color;
  GdkPoint *gdk_points;
  int i,x,y;
  
  gdk_points = g_new(GdkPoint, num_points);

  for (i=0;i<num_points;i++) {
    dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + points[i].x,
				 ((DiaRenderer*)renderer)->origin.y + points[i].y, &x, &y);
    gdk_points[i].x = x;
    gdk_points[i].y = y;
  }
  
  dia_color_convert(line_color, &color);
  gdk_gc_set_foreground(gc, &color);
  
  gdk_draw_polygon(renderer->pixmap, gc, TRUE, gdk_points, num_points);
  g_free(gdk_points);
}

static void
draw_rect(DiaRendererGdk *renderer, 
	  Point *ul_corner, Point *lr_corner,
	  DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  gint top, bottom, left, right;
    
  dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + ul_corner->x, 
			       ((DiaRenderer*)renderer)->origin.y + ul_corner->y,
			       &left, &top);
  dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + lr_corner->x,
			       ((DiaRenderer*)renderer)->origin.y + lr_corner->y,
			       &right, &bottom);
  
  if ((left>right) || (top>bottom))
    return;

  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);

  gdk_draw_rectangle (renderer->pixmap,
		      gc, FALSE,
		      left, top,
		      right-left,
		      bottom-top);
}

static void
fill_rect(DiaRendererGdk *renderer, 
	  Point *ul_corner, Point *lr_corner,
	  DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  gint top, bottom, left, right;
    
  dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + ul_corner->x,
			       ((DiaRenderer*)renderer)->origin.y + ul_corner->y,
			    &left, &top);
  dia_display_transform_coords(ddisp, ((DiaRenderer*)renderer)->origin.x + lr_corner->x,
			       ((DiaRenderer*)renderer)->origin.y + lr_corner->y,
			    &right, &bottom);
  
  if ((left>right) || (top>bottom))
    return;
  
  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);

  gdk_draw_rectangle (renderer->pixmap,
		      gc, TRUE,
		      left, top,
		      right-left,
		      bottom-top);
}

static void
draw_arc(DiaRendererGdk *renderer, 
	 Point *center,
	 gfloat width, gfloat height,
	 gfloat angle1, gfloat angle2,
	 DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  gint top, left, bottom, right;
  gfloat dangle;
  
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + center->x - width/2,
			       ((DiaRenderer*)renderer)->origin.y + center->y - height/2,
			       &left, &top);
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + center->x + width/2,
			       ((DiaRenderer*)renderer)->origin.y + center->y + height/2,
			       &right, &bottom);
  
  if ((left>right) || (top>bottom))
    return;

  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);

  dangle = angle2-angle1;
  if (dangle<0)
    dangle += 360.0;
  
  gdk_draw_arc(renderer->pixmap,
	       gc, FALSE,
	       left, top, right-left, bottom-top,
	       (int) (angle1*64.0), (int) (dangle*64.0));
}

static void
fill_arc(DiaRendererGdk *renderer, 
	 Point *center,
	 gfloat width, gfloat height,
	 gfloat angle1, gfloat angle2,
	 DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  gint top, left, bottom, right;
  gfloat dangle;
  
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + center->x - width/2,
			       ((DiaRenderer*)renderer)->origin.y + center->y - height/2,
			       &left, &top);
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + center->x + width/2,
			       ((DiaRenderer*)renderer)->origin.y + center->y + height/2,
			       &right, &bottom);
  
  if ((left>right) || (top>bottom))
    return;

  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);
  
  dangle = angle2-angle1;
  if (dangle<0)
    dangle += 360.0;
  
  gdk_draw_arc(renderer->pixmap,
	       gc, TRUE,
	       left, top, right-left, bottom-top,
	       (int) (angle1*64), (int) (dangle*64));
}

static void
draw_ellipse(DiaRendererGdk *renderer, 
	     Point *center,
	     gfloat width, gfloat height,
	     DiaColor *color)
{
  draw_arc(renderer, center, width, height, 0.0, 360.0, color); 
}

static void
fill_ellipse(DiaRendererGdk *renderer, 
	     Point *center,
	     gfloat width, gfloat height,
	     DiaColor *color)
{
  fill_arc(renderer, center, width, height, 0.0, 360.0, color); 
}

struct bezier_curve {
  GdkPoint *gdk_points;
  int numpoints;
  int currpoint;
};

#define BEZIER_SUBDIVIDE_LIMIT 0.03
#define BEZIER_SUBDIVIDE_LIMIT_SQ (BEZIER_SUBDIVIDE_LIMIT*BEZIER_SUBDIVIDE_LIMIT)

static void
bezier_add_point(DiaDisplay *ddisp,
		 struct bezier_curve *bezier,
		 Point *point)
{
  int x,y;
  
  /* Grow if needed: */
  if (bezier->currpoint == bezier->numpoints) {
    bezier->numpoints += 40;
    bezier->gdk_points = g_realloc(bezier->gdk_points,
				   bezier->numpoints*sizeof(GdkPoint));
  }
  
  dia_display_transform_coords(ddisp, ((DiaRenderer*)ddisp->renderer)->origin.x + point->x,
			       ((DiaRenderer*)ddisp->renderer)->origin.y + point->y, &x, &y);
  
  bezier->gdk_points[bezier->currpoint].x = x;
  bezier->gdk_points[bezier->currpoint].y = y;
  
  bezier->currpoint++;
}

static void
bezier_add_lines(DiaDisplay *ddisp,
		 Point points[4],
		 struct bezier_curve *bezier)
{
  Point u, v, x, y;
  Point r[4];
  Point s[4];
  Point middle;
  gfloat delta;

  /* Check if almost flat: */
  u = points[1];
  point_sub(&u, &points[0]);
  v = points[3];
  point_sub(&v, &points[0]);
  y = v;
  point_scale(&y, point_dot(&u,&v)/point_dot(&v,&v));
  x = u;
  point_sub(&x,&y);
  delta = dia_display_transform_length(ddisp, point_dot(&x,&x));
  if (delta < BEZIER_SUBDIVIDE_LIMIT_SQ) {
    u = points[2];
    point_sub(&u, &points[3]);
    v = points[0];
    point_sub(&v, &points[3]);
    y = v;
    point_scale(&y, point_dot(&u,&v)/point_dot(&v,&v));
    x = u;
    point_sub(&x,&y);
    delta = dia_display_transform_length(ddisp, point_dot(&x,&x));
    if (delta < BEZIER_SUBDIVIDE_LIMIT_SQ) {
      bezier_add_point(ddisp, bezier, &points[3]);
      return;
    }
  }
  /* Subdivide into two bezier curves: */

  middle = points[1];
  point_add(&middle, &points[2]);
  point_scale(&middle, 0.5);
  
  r[0] = points[0];
  
  r[1] = points[0];
  point_add(&r[1], &points[1]);
  point_scale(&r[1], 0.5);

  r[2] = r[1];
  point_add(&r[2], &middle);
  point_scale(&r[2], 0.5);

  s[3] = points[3];

  s[2] = points[2];
  point_add(&s[2], &points[3]);
  point_scale(&s[2], 0.5);
  
  s[1] = s[2];
  point_add(&s[1], &middle);
  point_scale(&s[1], 0.5);

  r[3] = r[2];
  point_add(&r[3], &s[1]);
  point_scale(&r[3], 0.5);
  
  s[0] = r[3];

  bezier_add_lines(ddisp, r, bezier);
  bezier_add_lines(ddisp, s, bezier);
}

static struct bezier_curve bezier = { NULL, 0, 0 };

static void
draw_bezier(DiaRendererGdk *renderer, 
	    Point *points,
	    gint numpoints, /* numpoints = 4+3*n, n=>0 */
	    DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdk_color;
  int i;
  
  if (bezier.gdk_points == NULL) {
    bezier.numpoints = 30;
    bezier.gdk_points = g_malloc(bezier.numpoints*sizeof(GdkPoint));
  }

  bezier.currpoint = 0;
  
  bezier_add_point(renderer->ddisp, &bezier, &points[0]);
  i = 0;
  while (i<=numpoints-3) {
    bezier_add_lines(ddisp, &points[i], &bezier);
    i += 3;
  }
  
  dia_color_convert(color, &gdk_color);
  gdk_gc_set_foreground(gc, &gdk_color);
  
  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     GDK_JOIN_ROUND);

  gdk_draw_lines(renderer->pixmap, gc, bezier.gdk_points, bezier.currpoint);

  gdk_gc_set_line_attributes(renderer->render_gc,
			     renderer->line_width,
			     renderer->line_style,
			     renderer->cap_style,
			     renderer->join_style);
}

static void
fill_bezier(DiaRendererGdk *renderer, 
	    Point *points, /* Last point must be same as first point */
	    int numpoints, /* numpoints = 4+3*n, n=>0 */
	    DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdk_color;
  int i;
  
  if (bezier.gdk_points == NULL) {
    bezier.numpoints = 30;
    bezier.gdk_points = g_malloc(bezier.numpoints*sizeof(GdkPoint));
  }

  bezier.currpoint = 0;
  
  bezier_add_point(renderer->ddisp, &bezier, &points[0]);
  i = 0;
  while (i<=numpoints-3) {
    bezier_add_lines(ddisp, &points[i], &bezier);
    i += 3;
  }
  
  dia_color_convert(color, &gdk_color);
  gdk_gc_set_foreground(gc, &gdk_color);
  
  gdk_draw_polygon(renderer->pixmap, gc, TRUE,
		   bezier.gdk_points, bezier.currpoint);
}

static void
draw_string(DiaRendererGdk *renderer,
	    const gchar *text,
	    Point *pos, 
	    DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  gint x,y;
  
  dia_display_transform_coords (ddisp,
				((DiaRenderer*)renderer)->origin.x + pos->x,
				((DiaRenderer*)renderer)->origin.y + pos->y,
				&x, &y);
    
  dia_color_convert (color, &gdkcolor);
  gdk_gc_set_foreground (gc, &gdkcolor);
      
  gdk_draw_string (renderer->pixmap,
		   renderer->gdk_font, gc,
		   x, y, text);
}

static void
draw_image (DiaRendererGdk *renderer,
	    Point *point,
	    gfloat width, gfloat height,
	    DiaImage *image)
{
  gint real_width, real_height, real_x, real_y;
  
  real_width = dia_display_transform_length(renderer->ddisp, width);
  real_height = dia_display_transform_length(renderer->ddisp, height);
  dia_display_transform_coords(renderer->ddisp,
			       ((DiaRenderer*)renderer)->origin.x + point->x,
			       ((DiaRenderer*)renderer)->origin.x + point->y,
			       &real_x, &real_y);

  dia_image_draw(image,  renderer->pixmap, real_x, real_y,
		 real_width, real_height);
}

static void
draw_handle (DiaRendererGdk *renderer, Point *pos, DiaColor *color,
	     gboolean is_connectable)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc;
  GdkColor gdk_color;
  gint x, y;
  gfloat half_size = DIA_HANDLE_SIZE/2;
  
  gc = gdk_gc_new (ddisp->canvas->window);
  gdk_gc_set_line_attributes (gc,
			      1, /* line width */
			      GDK_LINE_SOLID, /* line style */
			      GDK_CAP_ROUND, /* cap style */
			      GDK_JOIN_MITER); /* join style */
  
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + pos->x,
			       ((DiaRenderer*)renderer)->origin.y + pos->y,
			       &x, &y);

  dia_color_convert(color, &gdk_color);
  gdk_gc_set_foreground(gc, &gdk_color);
  gdk_draw_rectangle (renderer->pixmap,
		      gc, TRUE,
		      x - half_size, y - half_size,
		      DIA_HANDLE_SIZE,
		      DIA_HANDLE_SIZE);
  
  gdk_gc_set_foreground(gc, &dia_color_gdk_black);
  gdk_draw_rectangle (renderer->pixmap,
		      gc, FALSE,
		      x - half_size, y - half_size,
		      DIA_HANDLE_SIZE,
		      DIA_HANDLE_SIZE);
  if (is_connectable)
    {
      gdk_draw_line(renderer->pixmap, gc,
		    x - half_size,
		    y - half_size,
		    x + half_size,
		    y + half_size);
      gdk_draw_line(renderer->pixmap, gc,
		    x - half_size,
		    y + half_size + 1,
		    x + half_size + 1,
		    y - half_size);
    }
  gdk_gc_destroy (gc);
}

static void
draw_cp (DiaRendererGdk *renderer, Point *pos, DiaColor *color)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkGC *gc = NULL;
  GdkColor gdk_color;
  gint x, y;
  
  gc = gdk_gc_new (ddisp->canvas->window);
  gdk_gc_set_line_attributes (gc,
			      1, /* line width */
			      GDK_LINE_SOLID, /* line style */
			      GDK_CAP_ROUND, /* cap style */
			      GDK_JOIN_MITER); /* join style */
  
  dia_display_transform_coords(ddisp,
			       ((DiaRenderer*)renderer)->origin.x + pos->x,
			       ((DiaRenderer*)renderer)->origin.y + pos->y,
			       &x, &y);

  dia_color_convert(color, &gdk_color);
  gdk_gc_set_foreground(gc, &gdk_color);

  gdk_draw_line(renderer->pixmap, gc,
		x - DIA_CONNECTION_POINT_SIZE/2,
		y - DIA_CONNECTION_POINT_SIZE/2,
		x + DIA_CONNECTION_POINT_SIZE/2,
		y + DIA_CONNECTION_POINT_SIZE/2);
  gdk_draw_line(renderer->pixmap, gc,
		x - DIA_CONNECTION_POINT_SIZE/2,
		y + DIA_CONNECTION_POINT_SIZE/2,
		x + DIA_CONNECTION_POINT_SIZE/2,
		y - DIA_CONNECTION_POINT_SIZE/2);

  gdk_gc_destroy (gc);
}

static gfloat
get_text_width (DiaRendererGdk *renderer,
		const gchar *text, gint length)
{
  gint iwidth;
  
  iwidth = gdk_text_width (renderer->gdk_font, text, length);

  return dia_display_untransform_length (renderer->ddisp, (gfloat) iwidth);
}

/* retreive bounding box information */
static void
get_handle_bb (DiaRendererGdk *renderer, Point *pos, Rectangle *bb)
{
  DiaDisplay *ddisp = renderer->ddisp;
  //DiaRenderer *ren = (DiaRenderer*) renderer;
  gfloat x = ((gfloat)(DIA_HANDLE_SIZE + 1))/(ddisp->zoom_factor * 2);
  
  bb->left = pos->x - x;
  bb->right = pos->x + x;
  bb->top = pos->y - x;
  bb->bottom = pos->y + x;
}

static void
get_cp_bb (DiaRendererGdk *renderer, Point *pos, Rectangle *bb)
{
  DiaDisplay *ddisp = renderer->ddisp;
  //DiaRenderer *ren = (DiaRenderer*) renderer;
  gfloat x = ((gfloat)DIA_CONNECTION_POINT_SIZE + 1)/(ddisp->zoom_factor*2);

  bb->left = pos->x - x;
  bb->right = pos->x + x;
  bb->top = pos->y - x;
  bb->bottom = pos->y + x;
}

static void
clip_region_clear (DiaRendererGdk *renderer)
{
  if (renderer->clip_region != NULL)
    gdk_region_destroy (renderer->clip_region);

  renderer->clip_region =  gdk_region_new();

  gdk_gc_set_clip_region (renderer->render_gc, renderer->clip_region);
}

static void
clip_region_add_rect (DiaRendererGdk *renderer,
		      Rectangle *rect)
{
  DiaDisplay *ddisp = renderer->ddisp;
  GdkRegion *old_reg;
  GdkRectangle clip_rect;
  gint x1,y1;
  gint x2,y2;

  dia_display_transform_coords (ddisp, rect->left, rect->top,  &x1, &y1);
  dia_display_transform_coords (ddisp, rect->right, rect->bottom,  &x2, &y2);
  
  
  clip_rect.x = x1;
  clip_rect.y = y1;
  clip_rect.width = x2 - x1 + 1;
  clip_rect.height = y2 - y1 + 1;

  old_reg = renderer->clip_region;

  renderer->clip_region =
    gdk_region_union_with_rect (renderer->clip_region, &clip_rect);
  
  gdk_region_destroy (old_reg);

  gdk_gc_set_clip_region (renderer->render_gc, renderer->clip_region);
}

static void
draw_pixel_line (DiaRendererGdk *renderer,
		 gint x1, gint y1,
		 gint x2, gint y2,
		 DiaColor *color)
{
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
  
  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);
  
  gdk_draw_line(renderer->pixmap, gc, x1, y1, x2, y2);
}

static void
draw_pixel_rect (DiaRendererGdk *renderer,
		 gint x, gint y,
		 gint width, gint height,
		 DiaColor *color)
{
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
    
  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);

  gdk_draw_rectangle (renderer->pixmap, gc, FALSE,
		      x, y,  width, height);
}

static void
fill_pixel_rect (DiaRendererGdk *renderer,
		 gint x, gint y,
		 gint width, gint height,
		 DiaColor *color)
{
  GdkGC *gc = renderer->render_gc;
  GdkColor gdkcolor;
    
  dia_color_convert(color, &gdkcolor);
  gdk_gc_set_foreground(gc, &gdkcolor);

  gdk_draw_rectangle (renderer->pixmap, gc, TRUE,
		      x, y,  width, height);
}











