/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "geometry.h"

void
point_add (Point *p1, Point *p2)
{
  p1->x += p2->x;
  p1->y += p2->y;
}

void
point_sub (Point *p1, Point *p2)
{
  p1->x -= p2->x;
  p1->y -= p2->y;
}

gfloat
point_dot (Point *p1, Point *p2)
{
  return (p1->x * p2->x) + (p1->y * p2->y);
}
    
void
point_scale (Point *p, gfloat alpha)
{
  p->x *= alpha;
  p->y *= alpha;
}

void
point_normalize (Point *p)
{
  gfloat len;

  len = sqrt(p->x*p->x + p->y*p->y);
  
  p->x /= len;
  p->y /= len;
}

gboolean
line_line_intersection (Point *start1, Point *end1,
			Point *start2, Point *end2,
			Point *intersect)
{
  gfloat a1 = 0.0, a2 = 0.0;
  gfloat b1 = 0.0, b2 = 0.0;
  gboolean vertical1 = FALSE;
  gboolean vertical2 = FALSE;
  Rectangle bb1, bb2;
  Point p;
  
  /* first line: y1 = a1*x + b1 */
  if ((start1->x - end1->x) != 0.0) /* line is vertical? */
    {
      a1 = (start1->y - end1->y) / (start1->x - end1->x);
      b1 = start1->y - a1 * start1->x;
    }
  else
    {
      vertical1 = TRUE;
    }
  
  if ((start2->x - end2->x) != 0.0)
    {
      a2 = (start2->y - end2->y)/ (start2->x - end2->x);
      b2 = start2->y - a2 * start2->x;
    }
  else
    {
      vertical2 = TRUE;
    }
  
  
  if (vertical1 && vertical2)       /* two parallel vertical lines */ 
     return FALSE;

  if (vertical1)                    /* line one is vertical */
    {
      p.x = start1->x;
      p.y = a2 * p.x + b2;
    }
  else
    if (vertical2)                  /* line two is vertical */
      {
	p.x = start2->x;
	p.y = a1 * p.x + b1;
      }
    else
      if (a1 == a2)     /* two parallel lines */
	return FALSE;
      else                          /* two normal crossing lines */
	{
	  p.x = (b2 - b1) / (a1 - a2);
	  p.y = a1 * p.x + b1;
	}
  
  /* If the intersection point is within both bounding boxes, there is a
     valid intersection. */
  bb1.top = MIN (start1->y, end1->y);
  bb1.bottom = MAX (start1->y, end1->y);
  bb1.left = MIN (start1->x, end1->x);
  bb1.right = MAX (start1->x, end1->x);

  bb2.top = MIN (start2->y, end2->y);
  bb2.bottom = MAX (start2->y, end2->y);
  bb2.left = MIN (start2->x, end2->x);
  bb2.right = MAX (start2->x, end2->x);
  
  if (point_in_rectangle (&bb1, &p) &&
      point_in_rectangle (&bb2, &p))
    {
      if (intersect)
	*intersect = p;
      return TRUE;
    }

  return FALSE;
}

/* Find the intersection points of a line and a rectangle by comparing each
 * border line of the rectangle with the line's intersection.
 */
gboolean
line_rectangle_intersection (Point *start, Point *end,
			     Rectangle *rect, Point **intersect)
{
  Point tl; /* top-left */
  Point tr; /* top-right */
  Point bl; /* bottom... ah well */
  Point br;
  Point p;
  gint count = 0;
  Point *is;
  
  g_return_val_if_fail (start != NULL, FALSE);
  g_return_val_if_fail (end != NULL, FALSE);
  g_return_val_if_fail (rect != NULL, FALSE);
  g_return_val_if_fail (intersect != NULL, FALSE);
  
  /* A line can intersect with no more than two points of the rectangle
   * (plus one for the delimiter. */
  *intersect = g_new0 (Point, 3);
  is = *intersect;
  
  tl.x = bl.x = rect->left;
  tr.x = br.x = rect->right;
  tl.y = tr.y = rect->top;
  bl.y = br.y = rect->bottom;
  
  /* top */
  if (line_line_intersection (start, end, &tl, &tr, &p))
    {
      //count++;
      //*intersect = g_renew (Point, *intersect, count);
      is[count++] = p;
    }
  
  /* bottom */
  if (line_line_intersection (start, end, &bl, &br, &p))
    {
      //count++;
      //*intersect = g_renew (Point, *intersect, count);
      is[count++] = p;
    }
  //*intersect[count++] = p;
  if (count > 1)
    return TRUE;
  /* left */
  if (line_line_intersection (start, end, &tl, &bl, &p))
    {
      //count++;
      //*intersect = g_renew (Point, *intersect, count);
      is[count++] = p;
    }
  //*intersect[count++] = p;
  if (count > 1)
    return TRUE;
  /* right */
  if (line_line_intersection (start, end, &tr, &br, &p))
    {
      //count++;
      //*intersect = g_renew (Point, *intersect, count);
      is[count++] = p;
    }
  //    *intersect[count++] = p;
  //if (count)
  //{
  //  count++;
  //  *intersect = g_renew (Point, *intersect, count);
  //  ((Point)*intersect[count - 1]).x = 0.0;
  //  ((Point)*intersect[count - 1]).y = 0.0;
  //}
     
  return (count > 0) ? TRUE : FALSE;
}

void
rectangle_union (Rectangle *r1, Rectangle *r2)
{
  g_return_if_fail (r1 != NULL);
  g_return_if_fail (r2 != NULL);
  
  r1->top = MIN( r1->top, r2->top );
  r1->bottom = MAX( r1->bottom, r2->bottom );
  r1->left = MIN( r1->left, r2->left );
  r1->right = MAX( r1->right, r2->right );
}

void
rectangle_intersection (Rectangle *r1, Rectangle *r2)
{
  g_return_if_fail (r1 != NULL);
  g_return_if_fail (r2 != NULL);
  
  r1->top = MAX( r1->top, r2->top );
  r1->bottom = MIN( r1->bottom, r2->bottom );
  r1->left = MAX( r1->left, r2->left );
  r1->right = MIN( r1->right, r2->right );

  /* Is rectangle empty? */
  if ((r1->top > r1->bottom)
      || (r1->left > r1->right))
    {
      r1->top = r1->bottom = r1->left = r1->right = 0.0;
    }
}

int
rectangle_intersects (Rectangle *r1, Rectangle *r2)
{
  g_return_val_if_fail (r1 != NULL, FALSE);
  g_return_val_if_fail (r2 != NULL, FALSE);
  
  if ((r1->right < r2->left)
      || (r1->left > r2->right)
      || (r1->top > r2->bottom)
      || (r1->bottom < r2->top))
    return FALSE;

  return TRUE;
}

int
point_in_rectangle (Rectangle* r, Point *p)
{
  g_return_val_if_fail (r != NULL, FALSE);
  g_return_val_if_fail (p != NULL, FALSE);
  
  if ((p->x < r->left)
      || (p->x > r->right)
      || (p->y > r->bottom)
      || (p->y < r->top))
    return FALSE;

  return TRUE;
}

int
rectangle_in_rectangle (Rectangle* outer, Rectangle *inner)
{
  g_return_val_if_fail (inner != NULL, FALSE);
  g_return_val_if_fail (outer != NULL, FALSE);
  
  if ((inner->left <= outer->left)
      || (inner->right >= outer->right)
      || (inner->top <= outer->top)
      || (inner->bottom >= outer->bottom))
    return FALSE;
  
  return TRUE;
}

void
rectangle_add_point (Rectangle *r, Point *p)
{
  g_return_if_fail (r != NULL);
  g_return_if_fail (p != NULL);
  
  if (p->x < r->left)
    r->left = p->x;
  else
    if (p->x > r->right)
      r->right = p->x;

  if (p->y < r->top)
    r->top = p->y;
  else
    if (p->y > r->bottom)
      r->bottom = p->y;
}


gfloat
distance_point_point (Point *p1, Point *p2)
{
  gfloat dx;
  gfloat dy;

  g_return_val_if_fail (p1 != NULL, G_MAXFLOAT);
  g_return_val_if_fail (p2 != NULL, G_MAXFLOAT);
  
  dx = p1->x - p2->x;
  dy = p1->y - p2->y;
  return sqrt( dx*dx+dy*dy );
}

gfloat
distance_point_point_manhattan (Point *p1, Point *p2)
{
  gfloat dx;
  gfloat dy;

  g_return_val_if_fail (p1 != NULL, G_MAXFLOAT);
  g_return_val_if_fail (p2 != NULL, G_MAXFLOAT);
  
  dx = p1->x - p2->x;
  dy = p1->y - p2->y;
  return ABS(dx) + ABS(dy);
}

/*
 * This function estimates the distance from a point to a rectangle.
 * If the point is in the rectangle, 0.0 is returned. Otherwise the
 * distance in a manhattan metric from the point to the nearest point
 * on the rectangle is returned.
 */
gfloat
distance_rectangle_point (Rectangle *rect, Point *point)
{
  gfloat dx = 0.0;
  gfloat dy = 0.0;

  g_return_val_if_fail (rect != NULL, G_MAXFLOAT);
  g_return_val_if_fail (point != NULL, G_MAXFLOAT);
  
  if (point->x < rect->left ) {
    dx = rect->left - point->x;
  } else if (point->x > rect->right ) {
    dx = point->x - rect->right;
  }

  if (point->y < rect->top ) {
    dy = rect->top - point->y;
  } else if (point->y > rect->bottom ) {
    dy = point->y - rect->bottom;
  }

  return dx + dy;
}

/*
 * This function estimates the distance from a point to a line segment
 * specified by two endpoints.
 * If the point is on the line, 0.0 is returned. Otherwise the
 * distance in the R^2 metric from the point to the nearest point
 * on the line segment is returned. Does one sqrt per call.
 */
gfloat
distance_line_point (Point *line_start, Point *line_end,
		     gfloat line_width, Point *point, Point *point_on_line)
{
  Point v1, v2;
  gfloat v1_lensq;
  gfloat projlen;
  gfloat perp_dist;

  g_return_val_if_fail (line_start != NULL, G_MAXFLOAT);
  g_return_val_if_fail (line_end != NULL, G_MAXFLOAT);
  g_return_val_if_fail (point != NULL, G_MAXFLOAT);
  
  v1 = *line_end;
  point_sub(&v1,line_start);

  v2 = *point;
  point_sub(&v2, line_start);

  v1_lensq = point_dot(&v1,&v1);

  if (v1_lensq<0.000001) {
    return sqrt(point_dot(&v2,&v2));
  }

  projlen = point_dot(&v1,&v2) / v1_lensq;
  
  if (projlen<0.0) {
    if (point_on_line)
      *point_on_line = *line_start;
    return sqrt(point_dot(&v2,&v2));
  }
  
  if (projlen>1.0) {
    Point v3 = *point;
    point_sub(&v3,line_end);
    if (point_on_line)
      *point_on_line = *line_end;
    return sqrt(point_dot(&v3,&v3));
  }

  point_scale(&v1, projlen);
  if (point_on_line)
    {
      *point_on_line = v1;
      point_add (point_on_line, line_start);
    }
    
  point_sub(&v1, &v2);
  perp_dist = sqrt(point_dot(&v1,&v1));

  perp_dist -= line_width / 2.0;
  if (perp_dist < 0.0) {
    perp_dist = 0.0;
  }
  
  return perp_dist;
}









