/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/******************************************
 * This is The DiaCanvas Test application *
 ******************************************/ 
#include <stdlib.h>
#include <gtk/gtk.h>
#include "diamain.h"
#include "diadiagram.h"
#include "diadisplay.h"
#include "diamodifytool.h"
#include "diamagnifytool.h"
#include "diacreatetool.h"
//#include "diagraph.h"
#include "diadynline.h"
#include "diabox.h"
#include "diarenderereps.h"
#include "diabasetext.h"
#include "diaboxelement.h"
#include "diamultiline.h"

/* This defines the size of the canvas, in pixels */
#define CANVAS_SIZE 450


GtkWidget* create_preference_window (DiaDisplay *ddisp);

void
dump_object (DiaObject *object)
{
  gint i;
  GList *l;
  
  g_print ("-----------------------------\n"
	   "name: %s\nstate: 0x%x\n", object->object_type->name,
	   object->flags);
  g_print ("bounding-box: (%f, %f) (%f, %f)\n",
	   object->bounding_box.left, object->bounding_box.top,
	   object->bounding_box.right, object->bounding_box.bottom);

  g_print ("handles:\n");
  for (i = 0; i < object->handles->len; i++)
    if (DIA_OBJECT_GET_HANDLE (object, i)->connected_to)
      g_print ("\t%d: connected to %s (%x)\n", i,
	       DIA_OBJECT_GET_HANDLE (object, i)->connected_to->object->object_type->name,
	       DIA_OBJECT_GET_HANDLE (object, i)->connected_to->object);
    else
      g_print ("\t%d: unconnected\n", i);
      
  g_print ("connection points:\n");
  for (i = 0; i < object->connections->len; i++)
    {
      g_print ("\t%d: connects:\n", i);
      l = DIA_OBJECT_GET_CP (object, i)->connected;
      while (l)
	{
	  g_print ("\t\t%s (%x)\n", 
		   ((DiaHandle*) l->data)->object->object_type->name,
		   ((DiaHandle*) l->data)->object);
	  l = l->next;
	}
    }
            
  g_print ("-----------------------------\n");
}

/* Callback for the clicked signal of the Exit button */
static void
exit_clicked (GtkWidget *widget, gpointer data)
{
  gtk_main_quit ();
}

static void
cb_select_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{ 
  g_message ("Object selected: %s", obj->object_type->name); 
}

static void
cb_unselect_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{
  g_message ("Object unselected: %s", obj->object_type->name);
}

static void
cb_focus_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{
  g_message ("Object focused: %s", obj->object_type->name);
}

static void
cb_unfocus_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{
  g_message ("Object unfocused: %s", obj->object_type->name);
}

static void
cb_grab_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{
  g_message ("Object grabbed: %s", obj->object_type->name);
}

static void
cb_ungrab_object (DiaDisplay *display, DiaObject *obj, gpointer data)
{
  g_message ("Object ungrabbed: %s", obj->object_type->name);
}

static void
cb_add_item (DiaDiagram *dia, DiaLayer *layer, DiaObject *obj, gpointer data)
{
  g_message ("Object added: %s", obj->object_type->name);
}

static void
cb_remove_item (DiaDiagram *dia, DiaLayer *layer,
		DiaObject *obj, gpointer data)
{
  g_message ("Object removed: %s", obj->object_type->name);
}

static void
cb_active_display_change (DiaDisplay *display, gpointer data)
{
  g_message ("active_display_change: %x", (uint)display);
}

/* void main (int argc, char **argv)
 * ---------------------------------
 * In this test program `main' creates two displays an connects the callback
 * functions described above for the first canvas. 
 */
int
main (int argc, char **argv)
{
  GtkWidget *window;
  GtkWidget *canvas;
  //GtkWidget *hbox;
  DiaLayer  *layer;
  DiaDiagram *dia;
  //DiaTool *temp_tool;
  DiaBaseLine *bline;
  DiaDynLine *line;
  DiaObject *obj;
  //Point    *point;
  //DiaHandle *handle;
  Point start_point = {0.0, 0.0};
  DiaTool *tool;
    
  /* init GTK (you've seen this before...) */
  gtk_init (&argc, &argv);

  /* Optional: use your favourite theme */
  gtk_rc_parse ("/usr/share/gtkrc");
  
  //gtk_themes_init (&argc, &argv);
  /* initialize DIA stuff */
  dia_init ();

  /* First create a diagram */
  dia = dia_diagram_new ("name");
  gtk_signal_connect (GTK_OBJECT (dia), "add_item",
		      GTK_SIGNAL_FUNC (cb_add_item), NULL);
  gtk_signal_connect (GTK_OBJECT (dia), "remove_item",
		      GTK_SIGNAL_FUNC (cb_remove_item), NULL);
 
  /* Create a new layer */
  layer = dia_layer_new ("");
    
  dia_diagram_add_layer (dia, layer);
  
  /* Add some objects to the layer. */
  /* A line (this is a base line: has no CP's and can't create them): */
  bline = DIA_BASE_LINE (dia_base_line_new (&start_point));
  dia_layer_add_object (layer, DIA_OBJECT (bline));

  /* check the simple typecast mechanism on the line... */
  if (dia_object_check_type (DIA_OBJECT (bline), dia_object_get_type ()))
    g_message ("typecast OK... (as it should be)");
  else
    g_message ("typecast failed (this isn't supposed to happen)");

  /* ...add another line (can create CP's)... */
  start_point.x = 1.0;
  line = DIA_DYN_LINE (dia_dyn_line_new (&start_point));
  dia_layer_add_object (layer, DIA_OBJECT (line));

  /* ...yet another line... */
  start_point.x = 2.0;
  line = DIA_DYN_LINE (dia_dyn_line_new (&start_point));
  dia_layer_add_object (layer, DIA_OBJECT (line));

  /* ...and a box (note the order!): */
  start_point.y = 2.0;
  obj = dia_box_new (&start_point);
  dia_layer_add_object (layer, obj);
  
  /* add some static text: */
   start_point.x = start_point.y = 4.0;
   dia_base_text_set_default_text ("Center-Middle");
   dia_base_text_set_default_height (0.8);
   obj = dia_base_text_new (&start_point);
   //DIA_BASE_TEXT (obj)->editable = FALSE;
   /* dia_base_text_set_halign (DIA_BASE_TEXT (obj), DIA_ALIGN_CENTER); */
   /* dia_base_text_set_valign (DIA_BASE_TEXT (obj), DIA_ALIGN_MIDDLE); */
  dia_layer_add_object (layer, obj);

  /* add some static text: */
/*   start_point.x = start_point.y = 6.0; */
/*   dia_base_text_set_default_text ("Left-Bottom (normal)"); */
/*   obj = dia_base_text_new (&start_point); */
/*   dia_layer_add_object (layer, obj); */
  /* add some static text: */
/*   start_point.x = start_point.y = 8.0; */
/*   dia_base_text_set_default_text ("Right-Bottom text"); */
/*   obj = dia_base_text_new (&start_point); */
/*   dia_base_text_set_halign (DIA_BASE_TEXT (obj), DIA_ALIGN_RIGHT); */
  //dia_base_text_set_valign (DIA_BASE_TEXT (obj), DIA_ALIGN_MIDDLE);
/*   dia_layer_add_object (layer, obj); */
  /* add an entry: */
/*   start_point.x = start_point.y = 10.0; */
/*   dia_base_text_set_default_text ("Left-Top text"); */
/*   obj = dia_entry_new (&start_point); */
/*   //dia_base_text_set_halign (DIA_BASE_TEXT (obj), DIA_ALIGN_CENTER); */
/*   dia_base_text_set_valign (DIA_BASE_TEXT (obj), DIA_ALIGN_TOP); */
/*   dia_layer_add_object (layer, obj); */

  /* update the extents (say bounding box) of the diagram, for scrollbars
   * etc. */
  dia_diagram_update_extents (dia);
    
  /* create first window */
  canvas = dia_display_new (dia, CANVAS_SIZE, CANVAS_SIZE);
  gtk_signal_connect (GTK_OBJECT (canvas), "select_object",
		      GTK_SIGNAL_FUNC (cb_select_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "unselect_object",
		      GTK_SIGNAL_FUNC (cb_unselect_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "focus_object",
		      GTK_SIGNAL_FUNC (cb_focus_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "unfocus_object",
		      GTK_SIGNAL_FUNC (cb_unfocus_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "grab_object",
		      GTK_SIGNAL_FUNC (cb_grab_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "ungrab_object",
		      GTK_SIGNAL_FUNC (cb_ungrab_object), NULL);
  gtk_signal_connect (GTK_OBJECT (canvas), "active_display_change",
		      GTK_SIGNAL_FUNC (cb_active_display_change), NULL);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, TRUE);
  
  gtk_window_set_title (GTK_WINDOW (window), 
			"TestCanvas (owns preference window)");
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (exit_clicked), window);    
  gtk_container_add (GTK_CONTAINER (window), canvas);
  dia_display_update_scrollbars (DIA_DISPLAY (canvas));

  gtk_widget_show_all (window);

  /* never forget to set the focus, otherwise the canvas can't
   * recieve key events. */
  gtk_widget_grab_focus (DIA_DISPLAY (canvas)->canvas);
  
  /* create the property window (this is generated by GLADE :-) ) */
  window = create_preference_window (DIA_DISPLAY (canvas));
  gtk_widget_show (window);

  /* create a second display here... (the easy way) */
  window = dia_display_new_in_window (dia, CANVAS_SIZE, CANVAS_SIZE,
  				      "TestCanvas (should have the same "
                                    "contents as the other one... ALWAYS!)");
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
  		      GTK_SIGNAL_FUNC (exit_clicked), window);
  gtk_widget_show (window);

  dia_display_set_default_cursor (GDK_DRAFT_LARGE);

  /* Optional: set a fixed size for the canvas */
  //dia_diagram_set_auto_resize (dia, FALSE);
  //dia_diagram_set_extents (dia, 0.0, 0.0, 13.0, 9.5);
 

  /* Create a standard tool (or use the preference window): */
  tool = dia_modify_tool_new ();
  //tool = dia_magnify_tool_new ();
  dia_tool_set_active (tool);
  
  gtk_main ();

  return 0;
}

/* prototypes for event handlers: */
void on_item_added                          (DiaDiagram      *diagram,
					     DiaLayer        *layer,
					     DiaObject       *object,
					     gpointer         data);
void on_item_not_added                      (DiaDiagram      *diagram,
					     DiaLayer        *layer,
					     DiaObject       *object,
					     gpointer         data);
void on_snap_toggled                        (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_visible_toggled                     (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_point_grid_toggled                  (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_line_grid_toggled                   (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_yinterval_changed                   (GtkEditable     *editable,
					     gpointer         user_data);
void on_xinterval_changed                   (GtkEditable     *editable,
					     gpointer         user_data);
void on_ywidth_changed                      (GtkEditable     *editable,
					     gpointer         user_data);
void on_xwidth_changed                      (GtkEditable     *editable,
					     gpointer         user_data);
void on_modify_tool_toggled                 (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_zoom_tool_toggled                   (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_refresh_clicked                     (GtkButton       *button,
					     gpointer         user_data);
void on_relations_clicked                   (GtkButton       *button,
					     gpointer         user_data);
void on_base_line_toggled                   (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_dyn_line_toggled                    (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_multi_line_toggled                  (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_box_toggled                         (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_box_element_toggled                 (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_graph_toggled                       (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_base_text_toggled                   (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_entry_toggled                       (GtkToggleButton *togglebutton,
					     gpointer         user_data);
void on_exit_button_clicked                 (GtkButton       *button,
					     gpointer         user_data);

GtkWidget*
create_preference_window (DiaDisplay *ddisp)
{
  GtkWidget *preference_window;
  GtkWidget *vbox1;
  GtkWidget *frame5;
  GtkWidget *table3;
  GtkWidget *snap;
  GtkWidget *visible;
  GSList *grid_type_group = NULL;
  GtkWidget *point_grid;
  GtkWidget *line_grid;
  GtkObject *yinterval_adj;
  GtkWidget *yinterval;
  GtkObject *xinterval_adj;
  GtkWidget *xinterval;
  GtkObject *ywidth_adj;
  GtkWidget *ywidth;
  GtkObject *xwidth_adj;
  GtkWidget *xwidth;
  GtkWidget *label8;
  GtkWidget *refresh;
  GtkWidget *label7;
  GtkWidget *label6;
  GtkWidget *label5;
  GtkWidget *hbox4;
  GtkWidget *frame6;
  GtkWidget *vbox2;
  GSList *objects_type_group = NULL;
  GtkWidget *modify_tool;
  GtkWidget *zoom_tool;
  GtkWidget *relations;
  GtkWidget *label9;
  GtkWidget *base_line;
  GtkWidget *dyn_line;
  GtkWidget *box;
  GtkWidget *box_element;
  GtkWidget *multi_line;
  GtkWidget *graph;
  GtkWidget *base_text;
  GtkWidget *entry;
  GtkWidget *exit_button;
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();

  preference_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_object_set_data (GTK_OBJECT (preference_window), "preference_window", preference_window);
  gtk_container_border_width (GTK_CONTAINER (preference_window), 5);
  gtk_window_set_title (GTK_WINDOW (preference_window), "Preferences");
  gtk_window_set_policy (GTK_WINDOW (preference_window), TRUE, TRUE, FALSE);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (preference_window), "vbox1", vbox1);
  gtk_widget_show (vbox1);
  gtk_container_add (GTK_CONTAINER (preference_window), vbox1);

  frame5 = gtk_frame_new ("Grid");
  gtk_object_set_data (GTK_OBJECT (preference_window), "frame5", frame5);
  gtk_widget_show (frame5);
  gtk_box_pack_start (GTK_BOX (vbox1), frame5, TRUE, TRUE, 0);

  table3 = gtk_table_new (6, 2, FALSE);
  gtk_object_set_data (GTK_OBJECT (preference_window), "table3", table3);
  gtk_widget_show (table3);
  gtk_container_add (GTK_CONTAINER (frame5), table3);
  gtk_container_border_width (GTK_CONTAINER (table3), 5);

  snap = gtk_check_button_new_with_label ("Snap to grid");
  gtk_object_set_data (GTK_OBJECT (preference_window), "snap", snap);
  gtk_widget_show (snap);
  gtk_table_attach (GTK_TABLE (table3), snap, 1, 2, 5, 6,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (snap), "toggled",
                      GTK_SIGNAL_FUNC (on_snap_toggled),
                      ddisp);

  visible = gtk_check_button_new_with_label ("Visible");
  gtk_object_set_data (GTK_OBJECT (preference_window), "visible", visible);
  gtk_widget_show (visible);
  gtk_table_attach (GTK_TABLE (table3), visible, 1, 2, 4, 5,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (visible), "toggled",
                      GTK_SIGNAL_FUNC (on_visible_toggled),
                      ddisp);

  point_grid = gtk_radio_button_new_with_label (grid_type_group, "Point grid");
  grid_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (point_grid));
  gtk_object_set_data (GTK_OBJECT (preference_window), "point_grid", point_grid);
  gtk_widget_show (point_grid);
  gtk_table_attach (GTK_TABLE (table3), point_grid, 0, 1, 5, 6,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (point_grid), "toggled",
                      GTK_SIGNAL_FUNC (on_point_grid_toggled),
                      ddisp);

  line_grid = gtk_radio_button_new_with_label (grid_type_group, "Line grid");
  grid_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (line_grid));
  gtk_object_set_data (GTK_OBJECT (preference_window), "line_grid", line_grid);
  gtk_widget_show (line_grid);
  gtk_table_attach (GTK_TABLE (table3), line_grid, 0, 1, 4, 5,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (line_grid), "toggled",
                      GTK_SIGNAL_FUNC (on_line_grid_toggled),
                      ddisp);

  yinterval_adj = gtk_adjustment_new (1, 1, 63, 1, 10, 10);
  yinterval = gtk_spin_button_new (GTK_ADJUSTMENT (yinterval_adj), 1, 0);
  gtk_object_set_data (GTK_OBJECT (preference_window), "yinterval", yinterval);
  gtk_widget_show (yinterval);
  gtk_table_attach (GTK_TABLE (table3), yinterval, 1, 2, 3, 4,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (yinterval), "changed",
                      GTK_SIGNAL_FUNC (on_yinterval_changed),
                      ddisp);

  xinterval_adj = gtk_adjustment_new (1, 1, 63, 1, 10, 10);
  xinterval = gtk_spin_button_new (GTK_ADJUSTMENT (xinterval_adj), 1, 0);
  gtk_object_set_data (GTK_OBJECT (preference_window), "xinterval", xinterval);
  gtk_widget_show (xinterval);
  gtk_table_attach (GTK_TABLE (table3), xinterval, 1, 2, 2, 3,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (xinterval), "changed",
                      GTK_SIGNAL_FUNC (on_xinterval_changed),
                      ddisp);

  ywidth_adj = gtk_adjustment_new (1, 0.01, 100, 0.1, 1, 1);
  ywidth = gtk_spin_button_new (GTK_ADJUSTMENT (ywidth_adj), 1, 2);
  gtk_object_set_data (GTK_OBJECT (preference_window), "ywidth", ywidth);
  gtk_widget_show (ywidth);
  gtk_table_attach (GTK_TABLE (table3), ywidth, 1, 2, 1, 2,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (ywidth), "changed",
                      GTK_SIGNAL_FUNC (on_ywidth_changed),
                      ddisp);

  xwidth_adj = gtk_adjustment_new (1, 0.01, 100, 0.1, 1, 1);
  xwidth = gtk_spin_button_new (GTK_ADJUSTMENT (xwidth_adj), 1, 2);
  gtk_object_set_data (GTK_OBJECT (preference_window), "xwidth", xwidth);
  gtk_widget_show (xwidth);
  gtk_table_attach (GTK_TABLE (table3), xwidth, 1, 2, 0, 1,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (xwidth), "changed",
                      GTK_SIGNAL_FUNC (on_xwidth_changed),
                      ddisp);

  label8 = gtk_label_new ("Y interval:");
  gtk_object_set_data (GTK_OBJECT (preference_window), "label8", label8);
  gtk_widget_show (label8);
  gtk_table_attach (GTK_TABLE (table3), label8, 0, 1, 3, 4,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label8), 0, 0.5);

  label7 = gtk_label_new ("X interval:");
  gtk_object_set_data (GTK_OBJECT (preference_window), "label7", label7);
  gtk_widget_show (label7);
  gtk_table_attach (GTK_TABLE (table3), label7, 0, 1, 2, 3,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label7), 0, 0.5);

  label6 = gtk_label_new ("Y width:");
  gtk_object_set_data (GTK_OBJECT (preference_window), "label6", label6);
  gtk_widget_show (label6);
  gtk_table_attach (GTK_TABLE (table3), label6, 0, 1, 1, 2,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label6), 0, 0.5);

  label5 = gtk_label_new ("X width:");
  gtk_object_set_data (GTK_OBJECT (preference_window), "label5", label5);
  gtk_widget_show (label5);
  gtk_table_attach (GTK_TABLE (table3), label5, 0, 1, 0, 1,
                    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label5), 0, 0.5);

  hbox4 = gtk_hbox_new (FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (preference_window), "hbox4", hbox4);
  gtk_widget_show (hbox4);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox4, TRUE, TRUE, 0);

  frame6 = gtk_frame_new ("Objects");
  gtk_object_set_data (GTK_OBJECT (preference_window), "frame6", frame6);
  gtk_widget_show (frame6);
  gtk_box_pack_start (GTK_BOX (hbox4), frame6, TRUE, TRUE, 0);

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (preference_window), "vbox2", vbox2);
  gtk_widget_show (vbox2);
  gtk_container_add (GTK_CONTAINER (frame6), vbox2);
  gtk_container_border_width (GTK_CONTAINER (vbox2), 5);

  modify_tool = gtk_radio_button_new_with_label (objects_type_group, "Modify (standard)");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (modify_tool));
  gtk_object_set_data (GTK_OBJECT (preference_window), "modify_tool", modify_tool);
  gtk_widget_show (modify_tool);
  gtk_box_pack_start (GTK_BOX (vbox2), modify_tool, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, modify_tool, "This is the standard tool (with the mouse cursor icon :-)", NULL);
  gtk_signal_connect (GTK_OBJECT (modify_tool), "toggled",
                      GTK_SIGNAL_FUNC (on_modify_tool_toggled),
                      ddisp);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (modify_tool), TRUE);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (modify_tool), FALSE);

  zoom_tool = gtk_radio_button_new_with_label (objects_type_group, "Zoom");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (zoom_tool));
  gtk_object_set_data (GTK_OBJECT (preference_window), "zoom_tool", zoom_tool);
  gtk_widget_show (zoom_tool);
  gtk_box_pack_start (GTK_BOX (vbox2), zoom_tool, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, zoom_tool, "Zoom in (left mb) and out (shift-left-mb) and box zoom", NULL);
  gtk_signal_connect (GTK_OBJECT (zoom_tool), "toggled",
                      GTK_SIGNAL_FUNC (on_zoom_tool_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (zoom_tool), FALSE);

  refresh = gtk_button_new_with_label ("Refresh");
  gtk_object_set_data (GTK_OBJECT (preference_window), "refresh", refresh);
  gtk_widget_show (refresh);
  gtk_box_pack_start (GTK_BOX (vbox2), refresh, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (refresh), "clicked",
                      GTK_SIGNAL_FUNC (on_refresh_clicked),
                      ddisp);
  
  relations = gtk_button_new_with_label ("Relations");
  gtk_object_set_data (GTK_OBJECT (preference_window), "relations", relations);
  gtk_widget_show (relations);
  gtk_box_pack_start (GTK_BOX (vbox2), relations, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (relations), "clicked",
                      GTK_SIGNAL_FUNC (on_relations_clicked),
                      ddisp);
  
  label9 = gtk_label_new ("Create objects:");
  gtk_object_set_data (GTK_OBJECT (preference_window), "label9", label9);
  gtk_widget_show (label9);
  gtk_box_pack_start (GTK_BOX (vbox2), label9, TRUE, TRUE, 0);
  gtk_misc_set_alignment (GTK_MISC (label9), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label9), 0, 2);

  base_line = gtk_radio_button_new_with_label (objects_type_group, "BaseLine");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (base_line));
  gtk_object_set_data (GTK_OBJECT (preference_window), "base_line", base_line);
  gtk_widget_show (base_line);
  gtk_box_pack_start (GTK_BOX (vbox2), base_line, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, base_line, "A simple line", NULL);
  gtk_signal_connect (GTK_OBJECT (base_line), "toggled",
                      GTK_SIGNAL_FUNC (on_base_line_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (base_line), FALSE);

  dyn_line = gtk_radio_button_new_with_label (objects_type_group, "DynLine");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (dyn_line));
  gtk_object_set_data (GTK_OBJECT (preference_window), "dyn_line", dyn_line);
  gtk_widget_show (dyn_line);
  gtk_box_pack_start (GTK_BOX (vbox2), dyn_line, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, dyn_line, "A line that can dynamically connect other objects", NULL);
  gtk_signal_connect (GTK_OBJECT (dyn_line), "toggled",
                      GTK_SIGNAL_FUNC (on_dyn_line_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (dyn_line), FALSE);

  multi_line = gtk_radio_button_new_with_label (objects_type_group, "MultiLine");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (multi_line));
  gtk_object_set_data (GTK_OBJECT (preference_window), "multiline", multi_line);
  gtk_widget_show (multi_line);
  gtk_box_pack_start (GTK_BOX (vbox2), multi_line, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, multi_line, "Adds a new handle instead of"
			"dragging... use shift-left-mb to drag the line",
			NULL);
  gtk_signal_connect (GTK_OBJECT (multi_line), "toggled",
                      GTK_SIGNAL_FUNC (on_multi_line_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (multi_line), FALSE);

  box = gtk_radio_button_new_with_label (objects_type_group, "Box");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (box));
  gtk_object_set_data (GTK_OBJECT (preference_window), "box", box);
  gtk_widget_show (box);
  gtk_box_pack_start (GTK_BOX (vbox2), box, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, box, "A rectangular box", NULL);
  gtk_signal_connect (GTK_OBJECT (box), "toggled",
                      GTK_SIGNAL_FUNC (on_box_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (box), FALSE);

  box_element = gtk_radio_button_new_with_label (objects_type_group, "BoxElement");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (box_element));
  gtk_object_set_data (GTK_OBJECT (preference_window), "box_element", box_element);
  gtk_widget_show (box_element);
  gtk_box_pack_start (GTK_BOX (vbox2), box_element, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, box_element, "Elements are used to create graph diagrams", NULL);
  gtk_signal_connect (GTK_OBJECT (box_element), "toggled",
                      GTK_SIGNAL_FUNC (on_box_element_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (box_element), FALSE);

  graph = gtk_radio_button_new_with_label (objects_type_group, "Graph");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (graph));
  gtk_object_set_data (GTK_OBJECT (preference_window), "graph", graph);
  gtk_widget_show (graph);
  gtk_box_pack_start (GTK_BOX (vbox2), graph, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, graph, "Graphs can connect two elements", NULL);
  gtk_signal_connect (GTK_OBJECT (graph), "toggled",
                      GTK_SIGNAL_FUNC (on_graph_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (graph), FALSE);

  base_text = gtk_radio_button_new_with_label (objects_type_group, "BaseText");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (base_text));
  gtk_object_set_data (GTK_OBJECT (preference_window), "base_text", base_text);
  gtk_widget_show (base_text);
  gtk_box_pack_start (GTK_BOX (vbox2), base_text, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, base_text, "Simple text widget", NULL);
  gtk_signal_connect (GTK_OBJECT (base_text), "toggled",
                      GTK_SIGNAL_FUNC (on_base_text_toggled),
                      ddisp);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (base_text), FALSE);

  entry = gtk_radio_button_new_with_label (objects_type_group, "Entry");
  objects_type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (entry));
  gtk_object_set_data (GTK_OBJECT (preference_window), "entry", entry);
  gtk_widget_show (entry);
  gtk_box_pack_start (GTK_BOX (vbox2), entry, TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, entry, "Text widget with selection", NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "toggled",
                      GTK_SIGNAL_FUNC (on_entry_toggled),
                      NULL);
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (entry), FALSE);

  exit_button = gtk_button_new_with_label ("Exit");
  gtk_object_set_data (GTK_OBJECT (preference_window), "exit_button", exit_button);
  gtk_widget_show (exit_button);
  gtk_box_pack_start (GTK_BOX (vbox1), exit_button, TRUE, TRUE, 0);
  gtk_container_border_width (GTK_CONTAINER (exit_button), 5);
  gtk_signal_connect (GTK_OBJECT (exit_button), "clicked",
                      GTK_SIGNAL_FUNC (on_exit_button_clicked),
                      ddisp);

  gtk_signal_connect (GTK_OBJECT (ddisp->diagram), "add_item",
		      GTK_SIGNAL_FUNC (on_item_added), modify_tool);
  gtk_signal_connect (GTK_OBJECT (ddisp->diagram), "add_item_fail",
		      GTK_SIGNAL_FUNC (on_item_not_added), modify_tool);
  
  return preference_window;
}

/* Preference window event handlers: */
void
on_item_added (DiaDiagram *diagram, DiaLayer *layer,
	       DiaObject *object, gpointer data)
{
  g_message ("A %s has been successfully added to the canvas!",
	     object->object_type->name);
  
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data), TRUE);
}

void
on_item_not_added (DiaDiagram *diagram, DiaLayer *layer,
		   DiaObject *object, gpointer data)
{
  g_message ("A %s has not been added to the canvas!",
	     object->object_type->name);
  
  //gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data), TRUE);
}

void
on_exit_button_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  DiaDisplay *ddisp = DIA_DISPLAY (user_data);
  DiaRendererEps *renderer;
  DiaDiagram *dia = ddisp->diagram;
  
  renderer = dia_renderer_eps_new (dia, "test.eps");
  
  dia_diagram_render (dia, (DiaRenderer*)renderer,
		   NULL /* Can be NULL */,
		     NULL, NULL);
  
  dia_renderer_eps_destroy (renderer);
  
  renderer = dia_renderer_eps_new_scale (dia, "scale.eps", 0.5);
  
  dia_diagram_render(dia, (DiaRenderer*)renderer,
		   NULL /* Can be NULL */,
		     NULL, NULL);
  
  dia_renderer_eps_destroy (renderer);
  
  gtk_main_quit ();
}

void
on_refresh_clicked (GtkButton *button, gpointer user_data)
{
  DiaDisplay *ddisp = DIA_DISPLAY (user_data);
  
  dia_display_add_update_all (ddisp);
  //dia_display_flush (ddisp);
}

void on_relations_clicked (GtkButton *button, gpointer user_data)
{
  DiaDisplay *ddisp = DIA_DISPLAY (user_data);
  //DiaObject *object;
  //gint i;
  GList *l;
  
  l = DIA_LAYER (ddisp->diagram->layers->data)->objects;
  
  while (l)
    {
      dump_object (DIA_OBJECT (l->data));
      l = l->next;
    }
}


void
on_point_grid_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaGrid *grid = &DIA_DISPLAY (user_data)->grid;
      
      grid->point_grid = TRUE;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }
}

void
on_line_grid_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaGrid *grid = &DIA_DISPLAY (user_data)->grid;
      
      grid->point_grid = FALSE;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }
}

void
on_snap_toggled                        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  DIA_DISPLAY (user_data)->grid.snap = togglebutton->active;
}

void
on_visible_toggled                     (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  DIA_DISPLAY (user_data)->grid.visible = togglebutton->active;
  dia_display_add_update_all (DIA_DISPLAY (user_data));
  //dia_display_flush (DIA_DISPLAY (user_data));
}

void
on_yinterval_changed                   (GtkEditable     *editable,
                                        gpointer         user_data)
{
  gchar* s =  gtk_editable_get_chars (editable, 0, -1);
  gint i;
  
  i = atoi (s);
  
  if ((i >= GRID_MIN_INTERVAL) && (i <= GRID_MAX_INTERVAL))
    {
      DIA_DISPLAY (user_data)->grid.interval_y = i;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }
  g_free (s);
}

void
on_xinterval_changed                   (GtkEditable     *editable,
                                        gpointer         user_data)
{
  gchar* s =  gtk_editable_get_chars (editable, 0, -1);
  gint i;
  
  i = atoi (s);
  
  if ((i >= GRID_MIN_INTERVAL) && (i <= GRID_MAX_INTERVAL))
    {
      DIA_DISPLAY (user_data)->grid.interval_x = i;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }
  g_free (s);
}

void
on_ywidth_changed                      (GtkEditable     *editable,
                                        gpointer         user_data)
{
  gchar* s =  gtk_editable_get_chars (editable, 0, -1);
  double i;
  
  i = atof (s);
  
  if ((i >= GRID_MIN_SIZE) && (i <= GRID_MAX_SIZE))
    {
      DIA_DISPLAY (user_data)->grid.width_y = (gfloat) i;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }  
  g_free (s);
}

void
on_xwidth_changed                      (GtkEditable     *editable,
                                        gpointer         user_data)
{
  gchar* s =  gtk_editable_get_chars (editable, 0, -1);
  double i;
  
  i = atof (s);

  if ((i >= GRID_MIN_SIZE) && (i <= GRID_MAX_SIZE))
    {
      DIA_DISPLAY (user_data)->grid.width_x = (gfloat)i;
      dia_display_add_update_all (DIA_DISPLAY (user_data));
      //dia_display_flush (DIA_DISPLAY (user_data));
    }  
  g_free (s);
}


void
on_zoom_tool_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_magnify_tool_new ();
      dia_tool_set_active (tool);
    }
}

void
on_modify_tool_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_modify_tool_new ();
      dia_tool_set_active (tool);
    }
}

void
on_base_line_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_base_line_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_dyn_line_toggled                    (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_dyn_line_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_multi_line_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_multi_line_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_box_toggled                         (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_box_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_box_element_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_box_element_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_graph_toggled                       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_graph_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_base_text_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (togglebutton->active)
    {
      DiaTool *tool;
      tool = dia_create_tool_new (dia_base_text_get_type ());
      dia_tool_set_active (tool);
    }
}

void
on_entry_toggled                       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
/*   if (togglebutton->active) */
/*     { */
/*       DiaTool *tool; */
/*       gfloat height = 1.0; */
/*       dia_base_text_set_default_text ("entry"); */
/*       tool = dia_create_tool_new ((DiaNewObjectFunc) dia_entry_new); */
/*       dia_tool_set_active (tool); */
/*     } */
}












