/*
Copyright (c) 2003-2005 by Radim HSN Kolar (hsn@netmag.cz)

You may copy or modify this file in any manner you wish, provided
that this notice is always included, and that you hold the author
harmless for any loss or damage resulting from the installation or
use of this software.

		     This is a free software.  Be creative. 
		    Let me know of any bugs and suggestions.

Version 1.0rc6
*/
import java.io.OutputStream;
import java.io.IOException;
import java.util.Vector;

/** Utilities for easier working with FSP.
 * <p>
 * There are some higher level function for working with FSP.
 *
 * @author Radim Kolar
 * @version 1.0
 */
public class FSPutil
{
   /** Downloads a file from FSP server.
    * <p>
    * This procedure download a file from FSP server, file is written to
    * OutputStream.  OutputStream is not closed at end of transfer.
    * 
    * @since 1.0
    * @param session active FSP session
    * @param filename filename on FSP server
    * @param os write file to this stream
    * @param start_from offset where to start dowload
    * @param byteswanted how many bytes to download, &lt; 0 for all
    */ 
   public static void download(FSPsession session,String filename,OutputStream os,long start_from,long byteswanted) throws IOException
   {
       FSPpacket pkt;
       byte fname[]=stringToASCIIZ(filename);
       while(true)
       {
	   pkt=session.interact(FSPpacket.CC_GET_FILE,start_from,fname,0,fname.length,null,0,0);
	   if(pkt.cmd==FSPpacket.CC_ERR)
	       throw new IOException("ERR: "+new String(pkt.buf,0,pkt.bb_len-1));
	   else
	   if(pkt.cmd!=FSPpacket.CC_GET_FILE)
	       throw new IOException("unexpected FSP server response");
	   if(pkt.bb_len==0) return;
	   if(
	       (pkt.bb_len > byteswanted) &&
	       ( byteswanted >= 0 )
	     )
	     {
		 pkt.bb_len = (short) byteswanted;
	     }	 
	   os.write(pkt.buf,0,pkt.bb_len);
	   start_from+=pkt.bb_len;
	   byteswanted-=pkt.bb_len;
	   if(byteswanted == 0 ) return; 
       }
   }

   /** Gets information about file or directory.
    * <p>
    * This function requests information about specific path from FSP server.
    * Server must support CC_STAT command, which is supported from FSP 2.8.1
    * Beta 11.
    *
    * @since 1.0
    * @param session FSPsession
    * @param path path for getting information
    * @return stat object or null if file is not found
    * @see FSPstat
    */
   public static FSPstat stat(FSPsession ses,String path) throws IOException
   {
       byte fname[]=stringToASCIIZ(path);
       FSPpacket pkt;

       pkt=ses.interact(FSPpacket.CC_STAT,0,fname,0,fname.length,null,0,0);
       if(pkt.cmd==FSPpacket.CC_ERR)
	       throw new IOException("ERR: "+new String(pkt.buf,0,pkt.bb_len-1));
       else
	   if(pkt.cmd!=FSPpacket.CC_STAT)
	       throw new IOException("unexpected FSP server response");
       
       if(pkt.buf[8]==0) return null; /* does not exists */
       FSPstat stat=new FSPstat();
       stat.name=path;
       stat.lastmod=( (pkt.buf[0]<<8) | (pkt.buf[1] & 0xFF)) <<16;
       stat.lastmod|= ((pkt.buf[2] & 0xFF)<<8) | (pkt.buf[3] & 0xFF);
       stat.lastmod&=0xffffffffL;
       stat.lastmod*=1000L;
       stat.length  =((pkt.buf[4] << 8) | (pkt.buf[5] & 0xFF)) << 16;
       stat.length |=((pkt.buf[6] & 0xFF)<< 8) | (pkt.buf[7] & 0xFF);
       stat.length &= 0xffffffffL;
       stat.type = pkt.buf[8];
       return stat;
   }

   /** check if FSP server supports CC_STAT command.
    * 
    * @param ses FSPsession
    * @since 1.0
    * @return true if CC_STAT command is supported
    */
   public static boolean statSupported(FSPsession ses) throws IOException
   {
       byte fname[]=stringToASCIIZ("/");
       FSPpacket pkt;

       pkt=ses.interact(FSPpacket.CC_STAT,0,fname,0,fname.length,null,0,0);
       if(pkt.cmd!=FSPpacket.CC_STAT)
	   return false;
       else
	   return true;
   }

   /** Converts String to ASCIIZ byte array.
    * @since 1.0
    * @param filename string to be converted
    * @return converted byte array NULL terminated
    */ 
   public static byte[] stringToASCIIZ(String filename)
   {
       if(filename==null) return new byte[1];
       byte fname[]=new byte[filename.length()+1];
       System.arraycopy(filename.getBytes(),0,fname,0,filename.length());
       fname[fname.length-1]=0;
       return fname;
   }

   /** get a file list from server.
    *
    * @since 1.0
    * @param session live FSPsession
    * @param directory directory to be listed
    * @return file list or null on error
    */ 
   public static String[] list(FSPsession session,String directory)
   {
       byte fname[]=stringToASCIIZ(directory);
       Vector dirlist=new Vector(20);
       FSPpacket pkt;
       int pos=0;
       int i,j;

       dirlister:while(true)
       {
	   try
	   {
	       pkt=session.interact(FSPpacket.CC_GET_DIR,pos,fname,0,fname.length,null,0,0);
	   }
	   catch (IOException err) { return null;}
	   if(pkt.cmd!=FSPpacket.CC_GET_DIR)
	       return null;
	   if(pkt.bb_len==0) break;
	   pos+=pkt.bb_len;
	   i=0;
nextentry:while(i<pkt.bb_len-9)
	   {
	       /* check entry type */
	       switch(pkt.buf[i+8])
	       {
		   case 0x2A: 
		       /* RDTYPE_SKIP */
		       break nextentry;
		   case 0x00:
		       break dirlister;
	       }
	       i+=9;
	       /* read ASCIIZ fname */
	       j=i;
	       while(pkt.buf[j]!=0) 
		   j++;
	       dirlist.addElement(new String(pkt.buf,i,j-i));
	       i=j+1;
	       /* move to next 4byte boundary */
	       while((i & 0x3)>0) 
		   i++;
	   }
       }
       /* convert Vector to array */
       String list[]=new String[dirlist.size()];
       for(i=0;i<dirlist.size();i++)
       {
	   list[i]=(String)dirlist.elementAt(i);
       }
       return list;

   }

   /** get a stat list from server.
    *
    * @since 1.0
    * @param session live FSPsession
    * @param directory directory to be listed
    * @return stat list or null on error
    */ 
   public static FSPstat[] statlist(FSPsession session,String directory)
   {
       byte fname[]=stringToASCIIZ(directory);
       Vector dirlist=new Vector(20);
       FSPpacket pkt;
       FSPstat stat;
       int pos=0;
       int i,j;

       dirlister:while(true)
       {
	   try
	   {
	       pkt=session.interact(FSPpacket.CC_GET_DIR,pos,fname,0,fname.length,null,0,0);
	   }
	   catch (IOException err) { return null;}
	   if(pkt.cmd!=FSPpacket.CC_GET_DIR)
	       return null;
	   if(pkt.bb_len==0) break;
	   pos+=pkt.bb_len;
	   i=0;
nextentry:while(i<pkt.bb_len-9)
	  {
	       /* check entry type */
	       switch(pkt.buf[i+8])
	       {
		   case 0x2A: 
		       /* RDTYPE_SKIP */
		       break nextentry;
		   case 0x00:
		       /* END OF LIST */
		       break dirlister;
	       }
	       /* create a new stat object */
	       stat=new FSPstat();
               /* extract date */
               stat.lastmod =((pkt.buf[i] << 8) | (pkt.buf[i+1] & 0xFF)) << 16;
	       i+=2;
	       stat.lastmod|=((pkt.buf[i] & 0xFF)<< 8) | (pkt.buf[i+1] & 0xFF);
	       i+=2;
               stat.lastmod &=0xffffffffL;
               stat.lastmod*=1000L;
	       /* extract size */
               stat.length  =((pkt.buf[i] << 8) | (pkt.buf[i+1] & 0xFF)) << 16;
	       i+=2;
	       stat.length |=((pkt.buf[i] & 0xFF)<< 8) | (pkt.buf[i+1] & 0xFF);
	       i+=2;
               /* extract type */
               stat.type=pkt.buf[i++];
	       /* read ASCIIZ fname */
	       j=i;
	       while(pkt.buf[j]!=0) 
		   j++;
	       stat.name=new String(pkt.buf,i,j-i);
	       dirlist.addElement(stat);
	       i=j+1;
	       /* move to next 4byte boundary */
	       while((i & 0x3)>0) 
		   i++;
	   }
       }

       /* convert Vector to array */
       FSPstat list[]=new FSPstat[dirlist.size()];
       for(i=0;i<dirlist.size();i++)
       {
	   list[i]=(FSPstat)dirlist.elementAt(i);
       }
       return list;

   }
   
}
