/*
Copyright (c) 2003-2005 by Radim HSN Kolar (hsn@netmag.cz)

You may copy or modify this file in any manner you wish, provided
that this notice is always included, and that you hold the author
harmless for any loss or damage resulting from the installation or
use of this software.

		     This is a free software.  Be creative. 
		    Let me know of any bugs and suggestions.

Version 1.0rc6
*/
import java.net.InetAddress;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.io.InterruptedIOException;
import java.io.IOException;
import java.util.Hashtable;

/** This class represents one live FSP session.
 * <p>
 * FSPsession class takes care about FSP session management. It handles
 * packet resends on timeouts and key management.
 *
 * @author Radim Kolar
 * @see FSPpacket
 * @version 1.0
 */
public class FSPsession
{
 private DatagramSocket socket;
 private DatagramPacket udp;
 private FSPpacket   packet;

 private short seq;  /* sequence number */
 
 private int timeout;
 private int delay;
 private int maxdelay;

 private int port;
 private InetAddress host;
 private String hostadr;  /* 1.2.3.4:2234  */

 private static Hashtable locks;
 
 /** minimum resend delay */
 public final static int MIN_DELAY=1000;
 /** default resend delay */
 public final static int DEFAULT_DELAY=1340;
 /** maximum resend delay. FSP protocol has max delay 300s, we are
     using 60s for faster recovery on common Internet/Wifi lines */
 public final static int MAX_DELAY=60000;

 /** default timeout (msec) */
 public final static int DEFAULT_TIMEOUT=300000;

 /** Creates a new FSP session
  *
  * @param host hostname of fsp server
  * @param port port number on fsp server
  * @since 1.0
  */
 public FSPsession(String host, int port) throws java.net.SocketException,java.net.UnknownHostException
 {
  this(InetAddress.getByName(host),port);
 }

 /** Creates a new FSP session
  *
  * @param host address of fsp server
  * @param port port number on fsp server
  * @since 1.0
  */
 public FSPsession(InetAddress host, int port) throws java.net.SocketException
 {
  socket=new DatagramSocket();
  udp=new DatagramPacket(new byte[FSPpacket.MAXSIZE],FSPpacket.MAXSIZE,host,port);
  if(port==0) port=21;
  udp.setPort(port);
  udp.setAddress(host);
  packet=new FSPpacket();
  seq=(short)((int)(Math.random()*0xffff) & 0xfff8);
  timeout=DEFAULT_TIMEOUT;
  delay=DEFAULT_DELAY;
  maxdelay=MAX_DELAY;
  
  if(locks==null) locks=new Hashtable();
  hostadr=host.getHostAddress()+":"+port;
  synchronized(locks)
  {
      Short k=(Short)locks.get(hostadr);
      if(k==null)
      {
	  k=new Short((short)0);
	  locks.put(hostadr,k);
      }
      /* Get object from hashtable, we need it for correct locking */
      java.util.Enumeration en=locks.keys();
      while(en.hasMoreElements())
      {
	  String hkey;
	  hkey=(String)en.nextElement();
	  if(hostadr.equals(hkey))
	  {
	      hostadr=hkey;
	      break;
	  }
      }
  }
  this.port=port;
  this.host=host;
 }

 /** Sends FSP packet and waits for reply, resend packet if needed
  *
  * @since 1.0
  * @throws java.io.InterruptedIOException
  */

 public FSPpacket interact(byte cmd,long filepos, byte data1[],int offset1,int length1, byte data2[],int offset2,int length2) throws InterruptedIOException
 {
     int rdelay=delay;
     int rtimeout=0;
     Short k;
     
     /* setup the packet */
     packet.setData(data1,offset1,length1,data2,offset2,length2);
     k=new Short((short)((int)(Math.random()*0xffff) & 0xfff8));
     if (k.shortValue()==seq)
	 seq ^=0x1080;
     else
	 seq=k.shortValue();
     packet.bb_seq=seq;
     packet.bb_pos=(int)(filepos & 0xffffffff);
     packet.cmd=cmd;
     /* get key for the host */
     synchronized(hostadr)
     {
         k=(Short)locks.get(hostadr);
	 packet.bb_key=k.shortValue();

	 while(true)
	 {
	     /* increase a sequence number */
	     packet.bb_seq=(short)(seq + (++packet.bb_seq & 0x07));
	     packet.assemble(udp);
	     try
	     {
		socket.setSoTimeout(rdelay);
	     }
	     catch (java.net.SocketException ex) {}
	     try
	     {
		 socket.send(udp);
		 udp.setLength(FSPpacket.MAXSIZE);
		 socket.receive(udp);
		 if(packet.disassemble(udp)==true)
		 {
		     // System.out.println("received valid fsp packet seq="+packet.bb_seq+" our seq="+seq);
		     /* check reply type */
		     if( (packet.cmd != cmd) && (packet.cmd != FSPpacket.CC_ERR))
			 continue;
		     /* check position */
		     if(packet.bb_pos != filepos && ( cmd == FSPpacket.CC_GET_DIR || cmd == FSPpacket.CC_GET_FILE || cmd == FSPpacket.CC_UP_LOAD || cmd == FSPpacket.CC_GRAB_FILE || cmd == FSPpacket.CC_INFO) )
			 continue;
		     /* check sequence number */
		     if( (packet.bb_seq & 0xfff8) == (seq & 0xfff8) )
		     {

			 locks.put(hostadr,new Short(packet.bb_key));
			 return packet;
		     }
		 }
	     } catch (InterruptedIOException ioe) {}
	       catch (IOException ioe) {}

	     rtimeout+=rdelay;
	     // System.out.println(rtimeout);
	     if(rtimeout>=timeout) throw new InterruptedIOException("Timeout");
	     /* increase delay */
	     rdelay*=1.5f;
	     if(rdelay>maxdelay) rdelay=maxdelay;
	 }
     }
 }

 /** close a session.
  * <p>
  * Session object can't be used after session is closed. This also sends
  * CC_BYE command to server.
  * 
  * @since 1.0
  */ 
 public void close()
 {
     if(socket!=null)
	 try
	 {
	   interact(FSPpacket.CC_BYE,0,null,0,0,null,0,0);
	   socket.close();
	 }
	 catch (Exception e) {}

     socket=null;
     udp=null;
     packet=null;
 }
 
 /** gets the delay.
  * @since 1.0
  */
 public int getDelay()
 {
     return delay;
 }
 /** sets the delay.
  * <p>
  * This functions sets delay parameter of FSP protocol stack. This
  * is timeout value for first packet. If packet is lost delay is multiplied
  * by 1.5. Delay is in milliseconds and must be between MIN_DELAY and MAX_DELAY
  *
  * @since 1.0
  * @param ndelay new delay in milliseconds
  */ 
 public void setDelay(int ndelay)
 {
     if(ndelay<MIN_DELAY) ndelay=MIN_DELAY;
     else
	 if(ndelay>MAX_DELAY) ndelay=MAX_DELAY;

     this.delay=ndelay;
 }
 /** gets the maximum delay between packets.
  *
  * @since 1.0
  */
 public int getMaxDelay()
 {
     return maxdelay;
 }
 /** sets the maximum delay between packets.
  *
  * @param newdelay new max delay time in milliseconds.
  * @since 1.0
  */
 public void setMaxDelay(int newdelay)
 {
     if(newdelay>MAX_DELAY)
	 newdelay=MAX_DELAY;
     this.maxdelay=newdelay;
 }

 /** get timeout value.
  * @since 1.0
  */
 public int getTimeout()
 {
     return timeout;
 }
 /** set timeout value.
  * <p>
  * If no packet from server is received in this time, session times out.
  * 
  * @since 1.0
  * @param timeout timeout value in milliseconds
  */ 
 public void setTimeout(int ntimeout)
 {
     if(ntimeout>=0) this.timeout=ntimeout;
 }

 /** closes session before doing GC.
  * <p>
  * If session is not closed, close it before doing GC on this object.
  * Timeout is set to 7 sec when doing it.
  * 
  * @since 1.0
  */ 
 public void finalize()
 {
     timeout=7000;
     close();
 }

}
