/* Copyright (C) 2003-2006 Datapark corp. All rights reserved.
   Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/
#include "dps_common.h"
#include "dps_utils.h"
#include "dps_db.h"
#include "dps_sqldbms.h"
#include "dps_agent.h"
#include "dps_env.h"
#include "dps_conf.h"
#include "dps_services.h"
#include "dps_sdp.h"
#include "dps_log.h"
#include "dps_xmalloc.h"
#include "dps_doc.h"
#include "dps_result.h"
#include "dps_searchtool.h"
#include "dps_vars.h"
#include "dps_match.h"
#include "dps_spell.h"
#include "dps_mutex.h"
#include "dps_signals.h"
#include "dps_socket.h"
#include "dps_store.h"
#include "dps_hash.h"
#include "dps_charsetutils.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <fcntl.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef HAVE_SYS_MSG_H
#include <sys/ipc.h>
#include <sys/msg.h>
#endif

/* This should be last include */
#ifdef DMALLOC
#include "dmalloc.h"
#endif

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long)-1)
#endif


#define DEBUG_SEARCH 1


static	fd_set mask;

/********************* SIG Handlers ****************/
static void sighandler(int sign);
static void TrackSighandler(int sign);
static void init_signals(void){
	/* Set up signals handler*/
	DpsSignal(SIGPIPE, sighandler);
	DpsSignal(SIGCHLD, sighandler);
	DpsSignal(SIGALRM, sighandler);
	DpsSignal(SIGUSR1, sighandler);
	DpsSignal(SIGUSR2, sighandler);
	DpsSignal(SIGHUP, sighandler);
	DpsSignal(SIGINT, sighandler);
	DpsSignal(SIGTERM, sighandler);
}

static void sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif
	switch(sign){
		case SIGPIPE:
			have_sigpipe = 1;
			break;
		case SIGCHLD:
			while (waitpid(-1, &status, WNOHANG) > 0);
			break;
		case SIGHUP:
			have_sighup=1;
			break;
		case SIGINT:
			have_sigint=1;
			break;
		case SIGTERM:
			have_sigterm=1;
			break;
	        case SIGALRM:
		        _exit(0);
			break;
		case SIGUSR1:
		  have_sigusr1 = 1;
		  break;
		case SIGUSR2:
		  have_sigusr2 = 1;
		  break;
		default:
			break;
	}
	init_signals();
}

static void init_TrackSignals(void){
	/* Set up signals handler*/
	DpsSignal(SIGPIPE, TrackSighandler);
	DpsSignal(SIGCHLD, TrackSighandler);
	DpsSignal(SIGALRM, TrackSighandler);
	DpsSignal(SIGUSR1, TrackSighandler);
	DpsSignal(SIGUSR2, TrackSighandler);
	DpsSignal(SIGHUP, TrackSighandler);
	DpsSignal(SIGINT, TrackSighandler);
	DpsSignal(SIGTERM, TrackSighandler);
}

static void TrackSighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif
	switch(sign){
		case SIGPIPE:
			have_sigpipe = 1;
			break;
		case SIGCHLD:
			while (waitpid(-1, &status, WNOHANG) > 0);
			break;
		case SIGHUP:
			have_sighup=1;
			break;
		case SIGINT:
			have_sigint=1;
			break;
		case SIGTERM:
			have_sigterm=1;
			break;
	        case SIGALRM:
		        _exit(0);
			break;
		case SIGUSR1:
		  have_sigusr1 = 1;
		  break;
		case SIGUSR2:
		  have_sigusr2 = 1;
		  break;
		default:
			break;
	}
	init_signals();
}

/*************************************************************/


static ssize_t DpsSearchdSendPacket(int fd,const DPS_SEARCHD_PACKET_HEADER *hdr,const void *data){
	ssize_t nsent;
	
	nsent=DpsSend(fd,hdr,sizeof(*hdr),0);
	if(nsent!=sizeof(*hdr))
		return(nsent);
	if(data){
		nsent+=DpsSend(fd,data,hdr->len,0);
	}
	return nsent;
}

static int do_client(DPS_ENV * Conf, int client){
	DPS_SEARCHD_PACKET_HEADER hdr;
	DPS_AGENT   *Agent;
	DPS_RESULT  *Res;
	char buf[1024]="";
	char *words = NULL;
	ssize_t nrecv,nsent;
	int verb=-1;
	int done=0;
	const char *bcharset;
	size_t ExcerptSize, ExcerptPadding;
#ifdef DEBUG_SEARCH
	unsigned long ticks;
	unsigned long total_ticks = ticks = DpsStartTimer();
#endif
	
	Agent=DpsAgentInit(NULL,Conf,1000);
	
	if (Agent == NULL) {
	  fprintf(stderr, "Can't alloc Agent at %s:%d", __FILE__, __LINE__);
	  return DPS_ERROR;
	}

	Res=DpsResultInit(NULL);
	if (Res == NULL) return DPS_ERROR;

	while(!done){
		size_t dlen=0,ndocs,i;
		int * doc_id=NULL;
		char * dinfo=NULL;
		char * tok, * lt;
		
		DpsLog(Agent,verb,"Waiting for command header");
		nrecv=DpsRecvall(client,&hdr,sizeof(hdr));
		if(nrecv!=sizeof(hdr)){
			DpsLog(Agent,verb,"Received incomplete header nrecv=%d",(int)nrecv);
			break;
		}else{
			DpsLog(Agent,verb,"Received header cmd=%d len=%d",hdr.cmd,hdr.len);
		}
		switch(hdr.cmd){
			case DPS_SEARCHD_CMD_DOCINFO:
				dinfo = (char*)DpsRealloc(dinfo, hdr.len + 1);
				if (dinfo == NULL) {
					DPS_FREE(doc_id);
					done=1;
					break;
				}
				nrecv = DpsRecvall(client, dinfo, hdr.len);
				if((size_t)nrecv != hdr.len){
					DpsLog(Agent,verb,"Received incomplete data nbytes=%d nrecv=%d",hdr.len,(int)nrecv);
					DPS_FREE(doc_id);
					done=1;
					break;
				}
				dinfo[hdr.len]='\0';
				ndocs = 0;
				tok = dps_strtok_r(dinfo, "\r\n", &lt);
				
				while(tok){
					Res->Doc = (DPS_DOCUMENT*)DpsRealloc(Res->Doc, sizeof(DPS_DOCUMENT) * (ndocs + 1));
					if (Res->Doc == NULL) {
					  DPS_FREE(doc_id);
					  done=1;
					  break;
					}
					DpsDocInit(&Res->Doc[ndocs]);
					DpsDocFromTextBuf(&Res->Doc[ndocs], tok);
					
					tok = dps_strtok_r(NULL, "\r\n", &lt);
					ndocs++;
				}
				
				Res->num_rows = ndocs;
				DpsLog(Agent,verb,"Received DOCINFO command len=%d ndocs=%d",hdr.len,ndocs);
#ifdef DEBUG_SEARCH
				total_ticks = DpsStartTimer();
#endif
				
				if(DPS_OK != DpsResAction(Agent, Res, DPS_RES_ACTION_DOCINFO)){
					DpsResultFree(Res);
					dps_snprintf(buf,sizeof(buf)-1,"%s",DpsEnvErrMsg(Agent->Conf));
					DpsLog(Agent,verb,"%s",DpsEnvErrMsg(Agent->Conf));
					hdr.cmd=DPS_SEARCHD_CMD_ERROR;
					hdr.len=dps_strlen(buf);
					nsent=DpsSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
#ifdef DEBUG_SEARCH
				total_ticks = DpsStartTimer() - total_ticks;
				DpsLog(Agent, DPS_LOG_EXTRA, "ResAction in %.2f sec.", (float)total_ticks / 1000);
#endif
				dlen=0;
				
#ifdef DEBUG_SEARCH
				total_ticks = DpsStartTimer();
#endif
				DpsAgentStoredConnect(Agent);
				ExcerptSize = (size_t)DpsVarListFindInt(&Agent->Vars, "ExcerptSize", 256);
				ExcerptPadding = (size_t)DpsVarListFindInt(&Agent->Vars, "ExcerptPadding", 40);
	

				for(i=0;i<Res->num_rows;i++){
					size_t		ulen;
					size_t		olen;
					char		*textbuf, *Excerpt = NULL, *al;
					size_t		nsec, r;
					DPS_DOCUMENT	*D=&Res->Doc[i];
					
					al = DpsVarListFindStrTxt(&D->Sections, "URL", "");
					DpsLog(Agent, DPS_LOG_DEBUG, "Start excerpts for %s", al);
					DpsVarListReplaceInt(&D->Sections, "URL_ID", DpsStrHash32(al));

					if (Agent->Flags.do_excerpt) Excerpt = DpsExcerptDoc(Agent, Res, D, ExcerptSize, ExcerptPadding);

					if ((Excerpt != NULL) && (dps_strlen(Excerpt) > 6)) {
					  DpsVarListReplaceStr(&D->Sections, "body", Excerpt);
					}
					if (DpsVarListFindStr(&D->Sections, "Z", NULL) != NULL) {
					  DpsVarListReplaceStr(&D->Sections, "ST", "0");
					}
					DPS_FREE(Excerpt);

					for (r = 0; r < 256; r++)
					for (nsec = 0; nsec < D->Sections.Root[r].nvars; nsec++)
						D->Sections.Root[r].Var[nsec].section = 1;
					
					textbuf = DpsDocToTextBuf(D);
					if (textbuf == NULL) break;
/*					
					fprintf(stderr, "%s\n\n", textbuf);
*/
					ulen=dps_strlen(textbuf)+2;
					olen=dlen;
					dlen=dlen+ulen;
					dinfo=(char*)DpsRealloc(dinfo,dlen+1);
					if (dinfo == NULL) {
					  DPS_FREE(doc_id);
					  done=1;
					  break;
					}
					dinfo[olen]='\0';
					sprintf(dinfo+olen,"%s\r\n",textbuf);
					DpsFree(textbuf);
				}
				
#ifdef DEBUG_SEARCH
				total_ticks = DpsStartTimer() - total_ticks;
				DpsLog(Agent, DPS_LOG_EXTRA, "Excerpts in %.2f sec.", (float)total_ticks / 1000);
#endif
				if(!dinfo) dinfo = (char*)DpsStrdup("nodocinfo");
			
				hdr.cmd=DPS_SEARCHD_CMD_DOCINFO;
				hdr.len=dps_strlen(dinfo);
				nsent=DpsSearchdSendPacket(client,&hdr,dinfo);
				DpsLog(Agent,verb,"Sent doc_info packet %d bytes",(int)nsent);
				DPS_FREE(dinfo);
				
				break;

		        case DPS_SEARCHD_CMD_CLONES:
			  {
			    DPS_RESULT *Cl;
			    DPS_DOCUMENT *D;
			    int origin_id;
			    char cl_buf[128];

			    if (hdr.len > 128) {
			      DpsLog(Agent, verb, "Received too many data bytes=%d", hdr.len);
			      done = 1;
			      break;
			    }
			    nrecv = DpsRecvall(client, cl_buf, hdr.len);
			    if((size_t)nrecv != hdr.len){
			      DpsLog(Agent, verb, "Received incomplete data nbytes=%d nrecv=%d", hdr.len, (int)nrecv);
			      done = 1;
			      break;
			    }
			    cl_buf[nrecv] = '\0';
			    sscanf(cl_buf, "%d", &origin_id);
			    D = DpsDocInit(NULL);
			    DpsVarListAddInt(&D->Sections, "DP_ID", origin_id);
			    Cl = DpsCloneList(Agent, &Agent->Vars, D);

			    DpsDocFree(D);
			    dlen=0;
				
			    if (Cl) for(i = 0; i < Cl->num_rows; i++) {
					size_t		ulen;
					size_t		olen;
					char		*textbuf;
					size_t		nsec, r;
					
					D = &Cl->Doc[i];
					
					for (r = 0; r < 256; r++)
					for (nsec = 0; nsec < D->Sections.Root[r].nvars; nsec++)
						D->Sections.Root[r].Var[nsec].section = 1;
					
					textbuf = DpsDocToTextBuf(D);
					if (textbuf == NULL) break;
					
					ulen=dps_strlen(textbuf)+2;
					olen=dlen;
					dlen=dlen+ulen;
					dinfo=(char*)DpsRealloc(dinfo,dlen+1);
					if (dinfo == NULL) {
					  DPS_FREE(doc_id);
					  done=1;
					  break;
					}
					dinfo[olen]='\0';
					sprintf(dinfo+olen,"%s\r\n",textbuf);
					DpsFree(textbuf);
			    }
			    
			    if (!dinfo) dinfo = (char*)DpsStrdup("nocloneinfo");
				
			    hdr.cmd = DPS_SEARCHD_CMD_DOCINFO;
			    hdr.len = dps_strlen(dinfo); 
			    nsent = DpsSearchdSendPacket(client,&hdr,dinfo);
			    DpsLog(Agent, verb, "Sent clone_info packet %d bytes", (int)nsent);
			    DPS_FREE(dinfo);
			    DpsResultFree(Cl);
			  }				
			  break;

 		        case DPS_SEARCHD_CMD_CATINFO:
			  {
			        DPS_CATEGORY Cat;
				int cmd;
				
				bzero((void*)&Cat, sizeof(DPS_CATEGORY));
				nrecv = DpsRecvall(client, &cmd, sizeof(int));
				nrecv = DpsRecvall(client, Cat.addr, hdr.len - sizeof(int));
				Cat.addr[hdr.len - sizeof(int)] = '\0';
				DpsLog(Agent,verb,"Received CATINFO command len=%d, cmd=%d, addr='%s'", hdr.len, cmd, Cat.addr);

				DpsCatAction(Agent, &Cat, cmd);
				
				dlen = Cat.ncategories * 1024;
				dinfo = (char*)DpsMalloc(dlen + 1);
				if (dinfo == NULL) {
					DPS_FREE(doc_id);
					done=1;
					break;
				}
				DpsCatToTextBuf(&Cat, dinfo, dlen);
				
				hdr.cmd = DPS_SEARCHD_CMD_CATINFO;
				hdr.len = dps_strlen(dinfo);
				nsent = DpsSearchdSendPacket(client, &hdr, dinfo);
				DpsLog(Agent,verb,"Sent cat_info packet %d bytes",(int)nsent);
				DPS_FREE(dinfo);

				DPS_FREE(Cat.Category);
				
			  }
			  break;
				
 		        case DPS_SEARCHD_CMD_URLACTION:
			  {
				int cmd;
				
				DpsLog(Agent,verb,"Received URLACTION command len=%d", hdr.len);
				nrecv = DpsRecvall(client, &cmd, sizeof(int));

				DpsURLAction(Agent, NULL, cmd);
				DpsLog(Agent,verb,"Received URLACTION command len=%d", hdr.len);
				
				hdr.cmd = DPS_SEARCHD_CMD_DOCCOUNT;
				hdr.len = sizeof(Agent->doccount);
				nsent = DpsSearchdSendPacket(client, &hdr, (char*)&Agent->doccount);
				DpsLog(Agent,verb,"Sent doccount packet %d bytes",(int)nsent);

			  }
			  break;
				
			case DPS_SEARCHD_CMD_WORDS:
			        words = (char*)DpsRealloc(words, hdr.len + 1);
				if (words == NULL) {
					dps_snprintf(buf, sizeof(buf)-1, "Can't alloc memory for query");
					DpsLog(Agent, verb, "Can't alloc memory for query");
					hdr.cmd=DPS_SEARCHD_CMD_ERROR;
					hdr.len=dps_strlen(buf);
					DpsSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				nrecv=DpsRecvall(client, words, hdr.len); /* FIXME: check */
				words[nrecv]='\0';
				DpsLog(Agent,verb,"Received words len=%d words='%s'",nrecv,words);
				
				DpsParseQueryString(Agent, &Agent->Vars, words);
				bcharset = DpsVarListFindStr(&Agent->Vars, "BrowserCharset", "iso-8859-1");
				if (!(Agent->Conf->bcs = DpsGetCharSet(bcharset))) {
					dps_snprintf(buf,sizeof(buf)-1,"Unknown BrowserCharset: %s", bcharset);
					DpsLog(Agent,verb,"Unknown BrowserCharset: %s", bcharset);
					hdr.cmd=DPS_SEARCHD_CMD_ERROR;
					hdr.len=dps_strlen(buf);
					DpsSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
				DpsLog(Agent, verb, "Query: %s [Charset: %s]", 
					DpsVarListFindStr(&Agent->Vars, "q", ""), bcharset);
				
				DpsPrepare(Agent, Res);		/* Prepare search query */
#ifdef HAVE_ASPELL
				if (Agent->Flags.use_aspellext && Agent->aspell_pid != 0) {
#ifdef UNIONWAIT
				  union wait status;
#else
				  int status;
#endif
				  kill(Agent->aspell_pid, SIGTERM);
				  while(waitpid(Agent->aspell_pid, &status, WNOHANG) > 0);
				  Agent->aspell_pid = 0;
				}
#endif /* HAVE_ASPELL */
				if(DPS_OK != DpsFindWords(Agent, Res)) {
					dps_snprintf(buf,sizeof(buf)-1,"%s",DpsEnvErrMsg(Agent->Conf));
					DpsLog(Agent,verb,"%s",DpsEnvErrMsg(Agent->Conf));
					hdr.cmd=DPS_SEARCHD_CMD_ERROR;
					hdr.len=dps_strlen(buf);
					DpsSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
				dps_snprintf(buf,sizeof(buf)-1,"Total_found=%d(%d)",Res->total_found,Res->CoordList.ncoords);
				hdr.cmd=DPS_SEARCHD_CMD_MESSAGE;
				hdr.len=dps_strlen(buf);
				nsent=DpsSearchdSendPacket(client,&hdr,buf);
				DpsLog(Agent,verb,"Sent total_found packet %d bytes buf='%s'",(int)nsent,buf);

/*				if (Res->offset) {
				  hdr.cmd = DPS_SEARCHD_CMD_WITHOFFSET;
				  hdr.len = 0;
				  nsent = DpsSearchdSendPacket(client,&hdr,buf);
				  DpsLog(Agent,verb,"Sent withoffset packet %d bytes",(int)nsent);
				}*/

				{
				  char *wbuf, *p;
				  p = DpsVarListFindStr(&Agent->Vars, "q", "");
				  hdr.cmd = DPS_SEARCHD_CMD_QLC;
				  hdr.len = dps_strlen(p);
				  nsent = DpsSearchdSendPacket(client, &hdr, p);

				  hdr.cmd = DPS_SEARCHD_CMD_WWL;
				  hdr.len = sizeof(DPS_WIDEWORDLIST);
				  for (i = 0; i < Res->WWList.nwords; i++) {
				    hdr.len += sizeof(DPS_WIDEWORD) 
				      + sizeof(char) * (Res->WWList.Word[i].len + 1) 
				      + sizeof(dpsunicode_t) * (Res->WWList.Word[i].ulen + 1) 
				      + sizeof(int);
				  }
				  p = wbuf = (char *)DpsXmalloc(hdr.len + 1); /* need DpsXmalloc */
				  if (p == NULL) {
					done=1;
					break;
				  }
				  dps_memmove(p, &(Res->WWList), sizeof(DPS_WIDEWORDLIST));
				  p += sizeof(DPS_WIDEWORDLIST);
				  for (i = 0; i < Res->WWList.nwords; i++) {
				    dps_memmove(p, &(Res->WWList.Word[i]), sizeof(DPS_WIDEWORD));
				    p += sizeof(DPS_WIDEWORD);
				    dps_memmove(p, Res->WWList.Word[i].word, Res->WWList.Word[i].len + 1);
				    p += Res->WWList.Word[i].len + 1;
				    p += sizeof(dpsunicode_t) - ((SDPALIGN)p % sizeof(dpsunicode_t));
				    dps_memmove(p, Res->WWList.Word[i].uword, sizeof(dpsunicode_t) * (Res->WWList.Word[i].ulen + 1));
				    p += sizeof(dpsunicode_t) * (Res->WWList.Word[i].ulen + 1);
				  }
				  nsent = DpsSearchdSendPacket(client, &hdr, wbuf);
				  DpsLog(Agent, verb, "Sent WWL packet %d bytes cmd=%d len=%d nwords=%d", 
					 (int)nsent, hdr.cmd, hdr.len, Res->WWList.nwords);

				  DPS_FREE(wbuf);
				}

				if (Res->PerSite) {
				  hdr.cmd = DPS_SEARCHD_CMD_PERSITE;
				  hdr.len = Res->CoordList.ncoords * sizeof(*Res->PerSite);
				  nsent = DpsSearchdSendPacket(client, &hdr, Res->PerSite);
				  DpsLog(Agent, verb, "Sent PerSite packet %u bytes cmd=%d len=%d", nsent, hdr.cmd, hdr.len);
				}

				hdr.cmd = DPS_SEARCHD_CMD_DATA;
				hdr.len = Res->CoordList.ncoords * sizeof(*Res->CoordList.Data);
				nsent = DpsSearchdSendPacket(client, &hdr, Res->CoordList.Data);
				DpsLog(Agent, verb, "Sent URLDATA packet %u bytes cmd=%d len=%d", nsent, hdr.cmd, hdr.len);

#if HAVE_ASPELL
				if (Res->Suggest != NULL) {
				  hdr.cmd = DPS_SEARCHD_CMD_SUGGEST;
				  hdr.len = dps_strlen(Res->Suggest);
				  nsent = DpsSearchdSendPacket(client, &hdr, Res->Suggest);
				  DpsLog(Agent, verb, "Sent Suggest packet %d bytes cmd=%d len=%d", (int)nsent, hdr.cmd, hdr.len);
				}
#endif

				hdr.cmd=DPS_SEARCHD_CMD_WORDS;
				hdr.len=Res->CoordList.ncoords*sizeof(DPS_URL_CRD);
				nsent=DpsSearchdSendPacket(client,&hdr,Res->CoordList.Coords);
				DpsLog(Agent,verb,"Sent words packet %d bytes cmd=%d len=%d nwords=%d",(int)nsent,hdr.cmd,hdr.len,Res->CoordList.Coords);

				/*
				for(i=0;i<Agent->total_found;i++){
					dps_snprintf(buf,sizeof(buf)-1,"u=%08X c=%08X",wrd[i].url_id,wrd[i].coord);
					DpsLog(Agent,verb,"%s",buf);
				}
				*/
				break;
			case DPS_SEARCHD_CMD_GOODBYE:
				Res->work_time = DpsStartTimer() - ticks;
				DpsTrackSearchd(Agent, Res);
/*				DpsTrack(Agent, &Agent->Conf->Vars, Res);*/
				DpsLog(Agent,verb,"Received goodbye command. Work time: %.3f sec.", (float)Res->work_time / 1000);
				done=1;
				break;
			default:
				DpsLog(Agent,verb,"Unknown command %d",hdr.cmd);
				done=1;
				break;
		}
	}
	close(client);
	client=0;
	DpsLog(Agent,verb,"Quit");
	DpsAgentFree(Agent);
	DpsResultFree(Res);
	DPS_FREE(words);
	return 0;
}

/*************************************************************/

static void usage(void){

	fprintf(stderr, "\nsearchd from %s-%s-%s\n(C)1998-2003, LavTech Corp.\n(C)2003-2004, Datapark Corp.\n\n\
Usage: searchd [OPTIONS]\n\
\n\
Options are:\n\
  -l		do not log to stderr\n\
  -v n          verbose level, 0-5\n\
  -s n          sleep n seconds before starting\n\ 
  -w /path      choose alternative working /var directory\n\
  -h,-?         print this help page and exit\n\
\n\
\n", PACKAGE,VERSION,DPS_DBTYPE);

	return;
}

#define SEARCHD_EXIT    0
#define SEARCHD_RELOAD  1

static int conf_main(int ctl_sock,DPS_AGENT * Agent){
	int mdone=0;
	int verb=-1;
	int res=SEARCHD_EXIT;

	FD_ZERO(&mask);
	FD_SET(ctl_sock,&mask);

	while(!mdone) {
		pid_t fres;
		int sel;
		struct timeval tval;
		fd_set msk;
		
		tval.tv_sec=300;
		tval.tv_usec=0;
		msk=mask;
		sel = select(FD_SETSIZE, &msk, 0, 0, &tval);

		if(have_sighup){
			DpsLog(Agent,verb,"SIGHUP arrived");
			have_sighup=0;
			res=SEARCHD_RELOAD;
			mdone=1;
		}
		if(have_sigint){
			DpsLog(Agent,verb,"SIGINT arrived");
			have_sigint=0;
			mdone=1;
		}
		if(have_sigterm){
			DpsLog(Agent,verb,"SIGTERM arrived");
			have_sigterm=0;
			mdone=1;
		}
		if(have_sigpipe){
			DpsLog(Agent, verb, "SIGPIPE arrived. Broken pipe!");
			have_sigpipe = 0;
			mdone = 1;
		}
		if (have_sigusr1) {
		  DpsIncLogLevel(Agent);
		  have_sigusr1 = 0;
		}
		if (have_sigusr2) {
		  DpsDecLogLevel(Agent);
		  have_sigusr2 = 0;
		}

		if(mdone)break;
		
		if(sel==0)continue;
		if(sel==-1){
			switch(errno){
				case EINTR:	/* Child */
					break;
				default:
					DpsLog(Agent,verb,"FIXME select error %d %s",errno,strerror(errno));
			}
			continue;
		}
		
		if(FD_ISSET(ctl_sock,&msk)){
			int		ns;
			struct sockaddr_in client_addr;
			socklen_t	addrlen=sizeof(client_addr);
			char		addr[128]="";
			int		method;
			DPS_MATCH	*M;
			DPS_MATCH_PART	P[10];
			
			if ((ns = accept(ctl_sock, (struct sockaddr *) &client_addr, &addrlen)) == -1) {
				DpsLog(Agent,verb,"accept() error %d %s",errno,strerror(errno));
				DpsEnvFree(Agent->Conf);
				DpsAgentFree(Agent);
				unlink(dps_pid_name);
				exit(1);
			}
			DpsLog(Agent,verb,"Connect %s", inet_ntoa(client_addr.sin_addr));
			
			dps_snprintf(addr,sizeof(addr)-1,inet_ntoa(client_addr.sin_addr));
			M=DpsMatchListFind(&Agent->Conf->Filters,addr,10,P);
			method=M?DpsMethod(M->arg):DPS_METHOD_GET;
			DpsLog(Agent,verb,"%s %s %s%s",M?M->arg:"",addr,M?M->pattern:"",M?"":"Allow by default");
			
			if(method==DPS_METHOD_DISALLOW){
				DpsLog(Agent,verb,"Reject client");
				close(ns);
				continue;
			}
			
			Agent->handle++;
			if((fres=fork())==0){
			        DPS_ENV *Env;
				DPS_DB *db;
				size_t i, dbfrom = 0, dbto = (Agent->flags & DPS_FLAG_UNOCON) ? Agent->Conf->dbl.nitems : Agent->dbl.nitems;

				for (i = dbfrom; i < dbto; i++) {
				  db = (Agent->flags & DPS_FLAG_UNOCON) ? &Agent->Conf->dbl.db[i] : &Agent->dbl.db[i];
				  db->connected = 0;
				}
#ifdef WITH_TRACE
				char filename[PATH_MAX];
				fclose(Agent->TR);
				dps_snprintf(filename, sizeof(filename), "/tmp/dps_agent.%d.trace", Agent->handle);
				Agent->TR = fopen(filename, "w");
#endif

				alarm(2400); /* 40 min. - maximum time of child execution */
				init_signals();
				
				dps_closesocket(ctl_sock);
				do_client(Env = Agent->Conf, ns);

				DpsEnvFree(Env);
				DpsAgentFree(Agent);
/*				unlink(dps_pid_name);*/
				exit(0);
			}else{
				close(ns);
				if(fres>0){
					
				}else{
					DpsLog(Agent,verb,"fork error %d",fres);
				}
			}
			
		}
	}
	return res;
}

#ifdef HAVE_SYS_MSG_H

#define MAXMSG (4096 + sizeof(long))

static void SearchdTrack(DPS_AGENT *Agent) {
  DPS_SQLRES      sqlRes;
  DPS_DB *db;
  int mq, rec_id, res = DPS_OK;
  ssize_t n;
  char query[MAXMSG];
  char *lt;
  char qbuf[4096 + MAXMSG];
  char *IP, *qwords, *qtime, *total_found, *wtime, *var, *val;
  size_t i, dbfrom = 0, dbto =  (Agent->flags & DPS_FLAG_UNOCON) ? Agent->Conf->dbl.nitems : Agent->dbl.nitems; 

  init_TrackSignals();

  DpsSQLResInit(&sqlRes);

#ifdef HAVE_SETPROCTITLE
  /* To see the URL being indexed in "ps" output on xBSD */
  setproctitle("Query Tracker");
#endif

  for (i = dbfrom; i < dbto; i++) {
    db = (Agent->flags & DPS_FLAG_UNOCON) ? &Agent->Conf->dbl.db[i] : &Agent->dbl.db[i];
    db->connected = 0;
  }

  if ((mq = msgget(ftok(dps_pid_name, /*"/tmp/dpsearchd",*/ 0), 

#if defined(IPC_R) && defined(IPC_W)
		   IPC_R | IPC_W 
#else
		   0600
#endif
		   | IPC_CREAT)) == -1) {
    DpsLog(Agent, DPS_LOG_ERROR, "DpsTrackSearchd: couldn't open mqueue ftok(%s) for reading: %s", dps_pid_name, strerror(errno) );
    return;
  }
  while(1) {
    if(have_sigint){
      DpsLog(Agent, DPS_LOG_ERROR, "Query Tracker: SIGINT arrived");
      have_sigint = 0;
      break;
    }
    if(have_sigterm){
      DpsLog(Agent, DPS_LOG_ERROR, "Query Tracker: SIGTERM arrived");
      have_sigterm = 0;
      break;
    }
    if(have_sigpipe){
      DpsLog(Agent, DPS_LOG_ERROR, "Query Tracker: SIGPIPE arrived. Broken pipe !");
      have_sigpipe = 0;
      break;
    }
    if (have_sigusr1) {
      DpsIncLogLevel(Agent);
      have_sigusr1 = 0;
    }
    if (have_sigusr2) {
      DpsDecLogLevel(Agent);
      have_sigusr2 = 0;
    }
    DpsLog(Agent, DPS_LOG_DEBUG, "Query Track: waiting msg: %d", mq);
    n = msgrcv(mq, query, MAXMSG, 0, 0);

    DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: query[%d]: %s", dps_strlen(query + sizeof(long)), query + sizeof(long) );


    if (n > 0) {
      query[n] = '\0';

      for (i = dbfrom; (mq > 0) && (i < dbto); i++) {
	db = (Agent->flags & DPS_FLAG_UNOCON) ? &Agent->Conf->dbl.db[i] : &Agent->dbl.db[i];
	if(db->TrackQuery) {
	  const char      *qu = (db->DBType == DPS_DB_PGSQL) ? "'" : "";

	  IP = dps_strtok_r(query + sizeof(long), "\2", &lt);
/*	  DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: IP: %s", IP);*/
	  qwords = dps_strtok_r(NULL, "\2", &lt);
/*	  DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: qwords: %s", qwords);*/
	  qtime = dps_strtok_r(NULL, "\2", &lt);
/*	  DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: qtime: %s", qtime);*/
	  total_found = dps_strtok_r(NULL, "\2", &lt); 
/*	  DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: total: %s", total_found);*/
	  wtime = dps_strtok_r(NULL, "\2", &lt); 
/*	  DpsLog(Agent, DPS_LOG_EXTRA, "Query Track: wtime: %s", wtime);*/
      
	  dps_snprintf(qbuf, sizeof(qbuf), "INSERT INTO qtrack (ip,qwords,qtime,found,wtime) VALUES ('%s','%s',%s,%s,%s)",
		       IP, qwords, qtime, total_found, wtime );
 
	  res = DpsSQLAsyncQuery(db, NULL, qbuf);
	  if (res != DPS_OK) continue;

	  dps_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id FROM qtrack WHERE ip='%s' AND qtime=%s", IP, qtime);
	  res = DpsSQLQuery(db, &sqlRes, qbuf);
	  if (res != DPS_OK) continue;
	  if (DpsSQLNumRows(&sqlRes) == 0) { DpsSQLFree(&sqlRes); res = DPS_ERROR; continue; }
	  rec_id = DPS_ATOI(DpsSQLValue(&sqlRes, 0, 0));
	  DpsSQLFree(&sqlRes);

	  do {
	    var = dps_strtok_r(NULL, "\2", &lt);
	    if (var != NULL) {
	      val = dps_strtok_r(NULL, "\2", &lt); 
	      dps_snprintf(qbuf, sizeof(qbuf), "INSERT INTO qinfo (q_id,name,value) VALUES (%s%i%s,'%s','%s')", 
			   qu, rec_id, qu, var, val);
	      res = DpsSQLAsyncQuery(db, NULL, qbuf);
	      if (res != DPS_OK) continue;
	    }
	  } while (var != NULL);

	}
      }
    }
  }
}

#endif


int main(int argc, char **argv) {
	int done=0;
	int ch=0, pid_fd;
	int nport=DPS_SEARCHD_PORT;
	const char * config_name = DPS_CONF_DIR "/searchd.conf";
	int debug_level = DPS_LOG_INFO;
	char	pidbuf[64];
	const char	*var_dir = NULL, *pvar_dir = DPS_VAR_DIR;
	pid_t track_pid = 0;

	log2stderr = 1;
	DpsInit(); /* Initialize library */
	DpsInitMutexes();

	while ((ch = getopt(argc, argv, "hl?v:w:s:")) != -1){
		switch (ch) {
			case 'l':
				log2stderr=0;
				break;
		        case 'v': debug_level = atoi(optarg); break;
			case 'w':
				pvar_dir = optarg;
				break;
		        case 's':
			        sleep((unsigned int)atoi(optarg));
				break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;
	if(argc==1)config_name=argv[0];

	if (dps_demonize() < 0) {
	  fprintf(stderr, "Can't demonize\n");
	  exit(1);
	}

	dps_snprintf(dps_pid_name, PATH_MAX, "%s%s%s", pvar_dir, DPSSLASHSTR, "searchd.pid");
	pid_fd = open(dps_pid_name, O_CREAT|O_EXCL|O_WRONLY, 0644);
	if(pid_fd < 0) {
		fprintf(stderr, "Can't create '%s': %s\n", dps_pid_name, strerror(errno));
		if(errno == EEXIST){
		  fprintf(stderr, "It seems that another searchd is already running!\nRemove '%s' if it is not true.", dps_pid_name);
		}
		return DPS_ERROR;
	}
	sprintf(pidbuf, "%d\n", (int)getpid());
	write(pid_fd, &pidbuf, dps_strlen(pidbuf));
	close(pid_fd);

	while(!done){
		struct sockaddr_in server_addr;
		int ctl_sock, on=1;
		DPS_AGENT * Agent;
		DPS_ENV * Conf=NULL;
		int verb=-1;
		dps_uint8 flags = DPS_FLAG_SPELL | DPS_FLAG_UNOCON | DPS_FLAG_ADD_SERV;
		int res=0;
		const char *lstn;

		if (track_pid != 0) {
		  kill(track_pid, SIGTERM);
		}

		Conf = DpsEnvInit(NULL);
		if (Conf == NULL) {
		  fprintf(stderr, "Can't alloc Env at %s:%d", __FILE__, __LINE__);
		  return DPS_ERROR;
		}
		Agent=DpsAgentInit(NULL,Conf,0);
		
		if (Agent == NULL) {
		  fprintf(stderr, "Can't alloc Agent at %s:%d", __FILE__, __LINE__);
		  return DPS_ERROR;
		}

		Agent->flags = Conf->flags = flags; /*DPS_FLAG_UNOCON | DPS_FLAG_ADD_SERV;*/

		res = DpsEnvLoad(Agent, config_name, flags);
		
		if (pvar_dir == NULL) var_dir = DpsVarListFindStr(&Conf->Vars, "VarDir", DPS_VAR_DIR);
		else var_dir = pvar_dir;

		DpsUniRegCompileAll(Conf);

		DpsOpenLog("searchd",Conf,log2stderr);
		
		DpsSetLogLevel(Agent, debug_level);
		Agent->flags = Conf->flags = flags;

		if(res!=DPS_OK){
			DpsLog(Agent,verb,"%s",DpsEnvErrMsg(Conf));
			DpsAgentFree(Agent);
			DpsEnvFree(Conf);
			unlink(dps_pid_name);
			exit(1);
		}

		if (Conf->Flags.PreloadURLData) {
		  DpsLog(Agent, verb, "Preloading url data");
		  DpsURLDataPreload(Agent);
		}
		DpsLog(Agent,verb,"searchd started with '%s'", config_name);
		DpsLog(Agent,verb,"VarDir: '%s'", DpsVarListFindStr(&Agent->Conf->Vars, "VarDir", DPS_VAR_DIR));
		DpsLog(Agent,verb,"Affixes: %d, Spells: %d, Synonyms: %d, Acronyms: %d, Stopwords: %d",
		       Conf->Affixes.naffixes,Conf->Spells.nspell,
		       Conf->Synonyms.nsynonyms,
		       Conf->Acronyms.nacronyms,
		       Conf->StopWords.nstopwords);
		DpsLog(Agent,verb,"Chinese dictionary with %d entries", Conf->Chi.nwords);
		DpsLog(Agent,verb,"Korean dictionary with %d entries", Conf->Korean.nwords);
		DpsLog(Agent,verb,"Thai dictionary with %d entries", Conf->Thai.nwords);

#ifdef HAVE_SYS_MSG_H

		if ((track_pid = fork() ) == -1) {
		  DpsLog(Agent, DPS_LOG_ERROR, "fork() error %d %s", errno, strerror(errno));
		  DpsAgentFree(Agent);
		  DpsEnvFree(Conf);
		  unlink(dps_pid_name);
		  exit(1);
		}
		if (track_pid == 0) { /* child process */
		  SearchdTrack(Agent);
		  DpsAgentFree(Agent);
		  DpsEnvFree(Conf);
		  exit(0);
		}
		DpsLog(Agent,verb,"Query tracker child started.");
#endif
		

#if defined(HAVE_PTHREAD) && defined(HAVE_THR_SETCONCURRENCY_PROT)
		if (thr_setconcurrency(16) != NULL) { /* FIXME: Does 16 is enough ? */
		  DpsLog(A, DPS_LOG_ERROR, "Can't set %d concurrency threads", maxthreads);
		  return DPS_ERROR;
		}
#endif
		if ((ctl_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			DpsLog(Agent,verb,"socket() error %d",errno);
			DpsAgentFree(Agent);
			DpsEnvFree(Conf);
			unlink(dps_pid_name);
			exit(1);
		}

		if (setsockopt(ctl_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
			DpsLog(Agent,verb,"setsockopt() error %d",errno);
			DpsAgentFree(Agent);
			DpsEnvFree(Conf);
			unlink(dps_pid_name);
			exit(1);
		}

		bzero((void*)&server_addr, sizeof(server_addr));
		server_addr.sin_family = AF_INET;
		if((lstn=DpsVarListFindStr(&Agent->Conf->Vars,"Listen",NULL))){
			char * cport;
			
			if((cport=strchr(lstn,':'))){
				DpsLog(Agent,verb,"Listening '%s'",lstn);
				*cport='\0';
				server_addr.sin_addr.s_addr = inet_addr(lstn);
				nport=atoi(cport+1);
			}else{
				nport=atoi(lstn);
				server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
				DpsLog(Agent,verb,"Listening port %d",nport);
			}
		}else{
			DpsLog(Agent,verb,"Listening port %d",nport);
			server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
		}
		server_addr.sin_port = htons((u_short)nport);

		if (bind(ctl_sock, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
			DpsLog(Agent,verb,"Can't bind: error %d %s",errno,strerror(errno));
			DpsAgentFree(Agent);
			DpsEnvFree(Conf);
			unlink(dps_pid_name);
			exit(1);
		}

		/* Backlog 32 is enough */
		if (listen(ctl_sock, 32) == -1) {
			DpsLog(Agent,verb,"listen() error %d %s",errno,strerror(errno));
			DpsAgentFree(Agent);
			DpsEnvFree(Conf);
			unlink(dps_pid_name);
			exit(1);
		}
		
		DpsLog(Agent,verb,"Ready");
		
		init_signals();
		res=conf_main(ctl_sock,Agent);
		dps_closesocket(ctl_sock);

		if(res!=SEARCHD_RELOAD){
			done=1;
			DpsLog(Agent,verb,"Shutdown");
		}else{
			DpsLog(Agent,verb,"Reloading conf");
		}
		DpsAgentFree(Agent);
		DpsEnvFree(Conf);
	}
		
	if (track_pid != 0) {
	  kill(track_pid, SIGTERM);
	}

	unlink(dps_pid_name);
	DpsDestroyMutexes();

#ifdef EFENCE
	fprintf(stderr, "Memory leaks checking\n");
	DpsEfenceCheckLeaks();
#endif
#ifdef FILENCE
	fprintf(stderr, "FD leaks checking\n");
	DpsFilenceCheckLeaks(NULL);
#endif

#ifdef BOEHMGC
	CHECK_LEAKS();
#endif
     
	return DPS_OK;
}
