// Creates and resgisters Tcl's stdout & stderr channels for e93
// (currently using John Ousterhout's Tcl)
// Copyright (C) 2000 Core Technologies.

// This file is part of e93.
//
// e93 is free software; you can redistribute it and/or modify
// it under the terms of the e93 LICENSE AGREEMENT.
//
// e93 is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// e93 LICENSE AGREEMENT for more details.
//
// You should have received a copy of the e93 LICENSE AGREEMENT
// along with e93; see the file "LICENSE.TXT".

// This code was contributed by Michael Manning

// This could use some more work to support some of the things Tcl might
// want to do with its channels. Like blocking and setting watches on
// them.


#include	"includes.h"

static EDITORBUFFERHANDLE
	stdoutBufferHandle,
	stderrBufferHandle;

static int CommonBlockModeProc(ClientData instanceData,int mode)
// Set blocking or non-blocking mode
{
    return(-1);			// can't set blocking, return non-zero
}

static void CommonWatchProc(ClientData instanceData,int mask)
// Called by the notifier to set up to watch for events on this
// channel.
{
}

static int CommonGetHandleProc(ClientData instanceData,int direction,ClientData *handlePtr)
// Called from Tcl_GetChannelHandle to retrieve OS handles from
// inside a command console line based channel.
{
	*handlePtr=NULL;

	return(TCL_ERROR);
}

static int CommonCloseProc(ClientData instanceData,Tcl_Interp *interp)
// Closes a console based IO channel.
{
	return(TCL_OK);		// nothing for us to do, but this proc must exist
}

static int StdoutOutputProc(ClientData clientData,char *buf,int charsToWrite,int *errorCode)
// Writes the given output on the IO channel. Returns count of how
// many characters were actually written, and an error indication.
{
	int
		charsWritten;
	EDITORBUFFER
		*theBuffer;

	theBuffer=EditorGetBufferFromHandle(&stdoutBufferHandle);
	if(theBuffer&&!BufferBusy(theBuffer))		// get the output buffer if there is one, make sure it is not otherwise busy
	{
		EditorInsert(theBuffer,(UINT8 *)buf,charsToWrite);
		charsWritten=charsToWrite;				// tell tcl how many characters we wrote
	}
	else
	{
		charsWritten=-1;
	}
	return(charsWritten);
}

static int StderrOutputProc(ClientData clientData,char *buf,int charsToWrite,int *errorCode)
// Writes the given output on the IO channel. Returns count of how
// many characters were actually written, and an error indication.
{
	int
		charsWritten;
	EDITORBUFFER
		*theBuffer;

	theBuffer=EditorGetBufferFromHandle(&stderrBufferHandle);
	if(theBuffer&&!BufferBusy(theBuffer))		// get the output buffer if there is one, make sure it is not otherwise busy
	{
		EditorInsert(theBuffer,(UINT8 *)buf,charsToWrite);
		charsWritten=charsToWrite;				// tell tcl how many characters we wrote
	}
	else
	{
		charsWritten=-1;
	}
	return(charsWritten);
}

static Tcl_ChannelType
	stdoutChannelType=
	{
		"window",					// Type name
		CommonBlockModeProc,		// Set blocking or non-blocking mode
		CommonCloseProc,			// Close proc
		NULL,						// Input proc (we don't do input)
		StdoutOutputProc,			// Output proc
		NULL,						// Seek proc
		NULL,						// Set option proc
		NULL,						// Get option proc
		CommonWatchProc,			// Set up notifier to watch the channel
		CommonGetHandleProc,		// Get an OS handle from channel
		NULL,						// Close2 proc
	};

static Tcl_ChannelType
	stderrChannelType=
	{
		"window",					// Type name
		CommonBlockModeProc,		// Set blocking or non-blocking mode
		CommonCloseProc,			// Close proc
		NULL,						// Input proc (we don't do input)
		StderrOutputProc,			// Output proc
		NULL,						// Seek proc
		NULL,						// Set option proc
		NULL,						// Get option proc
		CommonWatchProc,			// Set up notifier to watch the channel
		CommonGetHandleProc,		// Get an OS handle from channel
		NULL,						// Close2 proc
	};


EDITORBUFFER *GetTclStdoutBuffer(Tcl_Interp *localInterpreter)
// set the buffer being used for stdout
{
	return(EditorGetBufferFromHandle(&stdoutBufferHandle));
}

void SetTclStdoutBuffer(Tcl_Interp *localInterpreter,EDITORBUFFER *newBuffer)
// set the buffer to be used for stdout
{
	Tcl_Channel
		channel;

	Tcl_Eval(localInterpreter,"close stdout");	// if a stdout has been opened, close it ignore any error
	channel=Tcl_CreateChannel(&stdoutChannelType,"stdout",NULL,TCL_WRITABLE);
	Tcl_RegisterChannel(localInterpreter,channel);
	Tcl_SetChannelOption(localInterpreter,channel,"-buffering","none");	// don't buffer stderr, write as soon as you get data (Tcl convention)
	Tcl_SetChannelOption(localInterpreter,channel,"-translation","lf");	// we only want to see linefeeds in our output
	Tcl_ResetResult(localInterpreter);			// do not leave error messages hanging around

	EditorReleaseBufferHandle(&stdoutBufferHandle);	// release our grip on any buffer we held
	if(newBuffer)
	{
		EditorGrabBufferHandle(&stdoutBufferHandle,newBuffer);				// hang onto this one
	}
}

EDITORBUFFER *GetTclStderrBuffer(Tcl_Interp *localInterpreter)
// set the buffer to be used for stderr
{
	return(EditorGetBufferFromHandle(&stderrBufferHandle));
}

void SetTclStderrBuffer(Tcl_Interp *localInterpreter,EDITORBUFFER *newBuffer)
// set the buffer to be used for stderr
{
	Tcl_Channel
		channel;

	Tcl_Eval(localInterpreter,"close stderr");	// if a stderr has been opened, close it ignore any error
	channel=Tcl_CreateChannel(&stderrChannelType,"stderr",NULL,TCL_WRITABLE);
	Tcl_RegisterChannel(localInterpreter,channel);
	Tcl_SetChannelOption(localInterpreter,channel,"-buffering","none");	// don't buffer stderr, write as soon as you get data (Tcl convention)
	Tcl_SetChannelOption(localInterpreter,channel,"-translation","lf");	// we only want to see linefeeds in our output
	Tcl_ResetResult(localInterpreter);			// do not leave error messages hanging around

	EditorReleaseBufferHandle(&stderrBufferHandle);	// release our grip on any buffer we held
	if(newBuffer)
	{
		EditorGrabBufferHandle(&stderrBufferHandle,newBuffer);				// hang onto this one
	}
}

void UnInitChannels(Tcl_Interp *localInterpreter)
// undo what InitChannels did
{
//	SetTclStderrBuffer(localInterpreter,NULL);
	SetTclStdoutBuffer(localInterpreter,NULL);
}

bool InitChannels(Tcl_Interp *localInterpreter)
{
	EditorInitBufferHandle(&stdoutBufferHandle);
	EditorInitBufferHandle(&stderrBufferHandle);

	// start with no channels (this will mess up stderr for the editor)
//	SetTclStdoutBuffer(localInterpreter,NULL);
//	SetTclStderrBuffer(localInterpreter,NULL);

	return(true);
}
