/* $Id: themes.c,v 1.318 2004/04/28 09:12:49 wojtekka Exp $ */

/*
 *  (C) Copyright 2001-2002 Wojtek Kaniewski <wojtekka@irc.pl>
 * 			    Piotr Kupisiewicz <deli@rzepaknet.us>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#include <limits.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "dynstuff.h"
#include "stuff.h"
#include "themes.h"
#include "xmalloc.h"
#include "ui.h"

#ifndef PATH_MAX
#  define PATH_MAX _POSIX_PATH_MAX
#endif

char *prompt_cache = NULL, *prompt2_cache = NULL, *error_cache = NULL;
const char *timestamp_cache = NULL;

int no_prompt_cache = 0;

list_t formats = NULL;

/*
 * format_find()
 *
 * odnajduje warto danego formatu. jeli nie znajdzie, zwraca pusty cig,
 * eby nie musie uwaa na adne null-references.
 *
 *  - name.
 */
const char *format_find(const char *name)
{
	const char *tmp;
	int hash;
	list_t l;

	if (!name)
		return "";

	hash = ekg_hash(name);

	if (config_speech_app && !strchr(name, ',')) {
		char *name2 = saprintf("%s,speech", name);
		const char *tmp;
		
		if (strcmp((tmp = format_find(name2)), "")) {
			xfree(name2);
			return tmp;
		}
		
		xfree(name2);
	}

	if (config_theme && (tmp = strchr(config_theme, ',')) && !strchr(name, ',')) {
		char *name2 = saprintf("%s,%s", name, tmp + 1);
		const char *tmp;
		
		if (strcmp((tmp = format_find(name2)), "")) {
			xfree(name2);
			return tmp;
		}
		
		xfree(name2);
	}
	
	for (l = formats; l; l = l->next) {
		struct format *f = l->data;

		if (hash == f->name_hash && !strcasecmp(f->name, name))
			return f->value;
	}
	
	return "";
}

/*
 * format_ansi()
 *
 * zwraca sekwencj ansi odpowiadajc danemu kolorkowi z thememw ekg.
 */
const char *format_ansi(char ch)
{
	if (ch == 'k')
		return "\033[0;30m";
	if (ch == 'K')
		return "\033[1;30m";
	if (ch == 'l')
		return "\033[40m";
	if (ch == 'r')
		return "\033[0;31m";
	if (ch == 'R')
		return "\033[1;31m";
	if (ch == 's')
		return "\033[41m";
	if (ch == 'g')
		return "\033[0;32m";
	if (ch == 'G')
		return "\033[1;32m";
	if (ch == 'h')
		return "\033[42m";
	if (ch == 'y')
		return "\033[0;33m";
	if (ch == 'Y')
		return "\033[1;33m";
	if (ch == 'z')
		return "\033[43m";
	if (ch == 'b')
		return "\033[0;34m";
	if (ch == 'B')
		return "\033[1;34m";
	if (ch == 'e')
		return "\033[44m";
	if (ch == 'm' || ch == 'p')
		return "\033[0;35m";
	if (ch == 'M' || ch == 'P')
		return "\033[1;35m";
	if (ch == 'q')
		return "\033[45m";
	if (ch == 'c')
		return "\033[0;36m";
	if (ch == 'C')
		return "\033[1;36m";
	if (ch == 'd')
		return "\033[46m";
	if (ch == 'w')
		return "\033[0;37m";
	if (ch == 'W')
		return "\033[1;37m";
	if (ch == 'x')
		return "\033[47m";
	if (ch == 'i')
		return "\033[5m";
	if (ch == 'n')
		return "\033[0m";
	if (ch == 'T')
		return "\033[1m";

	return "";
}

/*
 * va_format_string()
 *
 * formatuje zgodnie z podanymi parametrami cig znakw.
 *
 *  - format - warto, nie nazwa formatu,
 *  - ap - argumenty.
 */
char *va_format_string(const char *format, va_list ap)
{
	static int dont_resolve = 0;
	string_t buf = string_init(NULL);
	const char *p, *args[9];
	int i, argc = 0;

	/* liczymy ilo argumentw */
	for (p = format; *p; p++) {
		if (*p != '%')
			continue;

		p++;

		if (!*p)
			break;

		if (*p == '@') {
			p++;

			if (!*p)
				break;

			if ((*p - '0') > argc)
				argc = *p - '0';
			
		} else if (*p == '(' || *p == '[') {
			if (*p == '(') {
				while (*p && *p != ')')
					p++;
			} else {
				while (*p && *p != ']')
					p++;
			}

			if (*p)
				p++;
			
			if (!*p)
				break;
			
			if ((*p - '0') > argc)
				argc = *p - '0';
		} else {
			if (*p >= '1' && *p <= '9' && (*p - '0') > argc)
				argc = *p - '0';
		}
	}

	for (i = 0; i < 9; i++)
		args[i] = NULL;

	for (i = 0; i < argc; i++)
		args[i] = va_arg(ap, char*);

	if (!dont_resolve) {
		dont_resolve = 1;
		if (no_prompt_cache) {
			/* zawsze czytaj */
			timestamp_cache = format_find("timestamp");
			prompt_cache = format_string(format_find("prompt"));
			prompt2_cache = format_string(format_find("prompt2"));
			error_cache = format_string(format_find("error"));
		} else {
			/* tylko jeli nie s keszowanie */
			if (!timestamp_cache)
				timestamp_cache = format_find("timestamp");
			if (!prompt_cache)
				prompt_cache = format_string(format_find("prompt"));
			if (!prompt2_cache)
				prompt2_cache = format_string(format_find("prompt2"));
			if (!error_cache)
				error_cache = format_string(format_find("error"));
		}
		dont_resolve = 0;
	}
	
	p = format;
	
	while (*p) {
		if (*p == '%') {
			int fill_before, fill_after, fill_soft, fill_length;
			char fill_char;

			p++;
			if (!*p)
				break;
			if (*p == '%')
				string_append_c(buf, '%');
			if (*p == '>')
				string_append(buf, prompt_cache);
			if (*p == ')')
				string_append(buf, prompt2_cache);
			if (*p == '!')
				string_append(buf, error_cache);
			if (*p == '|')
				string_append(buf, "\033[00m");	/* gupie, wiem */
			if (*p == ']')
				string_append(buf, "\033[000m");	/* jeszcze gupsze */
			if (*p == '#')
				string_append(buf, timestamp(timestamp_cache));
			else if (config_display_color) {
				string_append(buf, format_ansi(*p));
			}

			if (*p == '@') {
				char *str = (char*) args[*(p + 1) - '1'];

				if (str) {
					char *q = str + strlen(str) - 1;

					while (q >= str && !isalpha_pl_PL(*q))
						q--;

					if (*q == 'a')
						string_append(buf, "a");
					else
						string_append(buf, "y");
				}
				p += 2;
				continue;
			}

			fill_before = 0;
			fill_after = 0;
			fill_length = 0;
			fill_char = ' ';
			fill_soft = 1;

			if (*p == '[' || *p == '(') {
				char *q;

				fill_soft = (*p == '(');

				p++;
				fill_char = ' ';

				if (*p == '.') {
					fill_char = '0';
					p++;
				} else if (*p == ',') {
					fill_char = '.';
					p++;
				} else if (*p == '_') {
					fill_char = '_';
					p++;
				}

				fill_length = strtol(p, &q, 0);
				p = q;
				if (fill_length > 0)
					fill_after = 1;
				else {
					fill_length = -fill_length;
					fill_before = 1;
				}
				p++;
			}

			if (*p >= '1' && *p <= '9') {
				char *str = (char *) args[*p - '1'];
				int i, len;

				if (!str)
					str = "";
				len = strlen(str);

				if (fill_length) {
					if (len >= fill_length) {
						if (!fill_soft)
							len = fill_length;
						fill_length = 0;
					} else
						fill_length -= len;
				}

				if (fill_before)
					for (i = 0; i < fill_length; i++)
						string_append_c(buf, fill_char);

				string_append_n(buf, str, len);

				if (fill_after) 
					for (i = 0; i < fill_length; i++)
						string_append_c(buf, fill_char);

			}
		} else
			string_append_c(buf, *p);

		p++;
	}

	if (!dont_resolve && no_prompt_cache)
		theme_cache_reset();

	if (!config_display_pl_chars)
		iso_to_ascii(buf->str);

	return string_free(buf, 0);
}

/*
 * reformat_string()
 *
 * zamienia sformatowany cig znakw ansi na Nowy-i-Lepszy(tm).
 *
 *  - str - cig znakw,
 * 
 * zwraca zaalokowan fstring_t.
 */
fstring_t reformat_string(const char *str)
{
	fstring_t res = xnew(struct fstring_s);
	unsigned char attr = 128;
	int i, j, len = 0;
	
	for (i = 0; str[i]; i++) {
		if (str[i] == 27) {
			if (str[i + 1] != '[')
				continue;

			while (str[i] && !isalpha_pl_PL(str[i]))
				i++;

			i--;
			
			continue;
		}

		if (str[i] == 9) {
			len += (8 - (len % 8));
			continue;
		}

		if (str[i] == 13)
			continue;

		len++;
	}

	res->str = xmalloc(len + 1);
	res->attr = xmalloc(len + 1);
	res->prompt_len = 0;
	res->prompt_empty = 0;

	for (i = 0, j = 0; str[i]; i++) {
		if (str[i] == 27) {
			int tmp = 0;

			if (str[i + 1] != '[')
				continue;

			i += 2;

			/* obsuguje tylko "\033[...m", tak ma by */
			
			for (; str[i]; i++) {
				if (str[i] >= '0' && str[i] <= '9') {
					tmp *= 10;
					tmp += (str[i] - '0');
				}

				if (str[i] == ';' || isalpha_pl_PL(str[i])) {
					if (tmp == 0) {
						attr = 128;

						/* prompt jako \033[00m */
						if (str[i - 1] == '0' && str[i - 2] == '0')
							res->prompt_len = j;

						/* odstp jako \033[000m */
						if (i > 3 && str[i - 1] == '0' && str[i - 2] == '0' && str[i - 3] == 0) {
							res->prompt_len = j;
							res->prompt_empty = 1;
						}
					}

					if (tmp == 1)
						attr |= 64;

					if (tmp >= 30 && tmp <= 37) {
						attr &= 127;
						attr |= (tmp - 30);
					}

					if (tmp >= 40 && tmp <= 47) {
						attr &= 127;
						attr |= (tmp - 40) << 3;
					}

					tmp = 0;
				}

				if (isalpha_pl_PL(str[i]))
					break;
			}

			continue;
		}

		if (str[i] == 13)
			continue;

		if (str[i] == 9) {
			int k = 0, l = 8 - (j % 8);

			for (k = 0; k < l; j++, k++) {
				res->str[j] = ' ';
				res->attr[j] = attr;
			}

			continue;
		}

		res->str[j] = str[i];
		res->attr[j] = attr;

		j++;
	}

	res->str[j] = 0;
	res->attr[j] = 0;

	return res;
}

/*
 * format_string()
 *
 * j.w. tyle e nie potrzeba dawa mu va_list, a wystarcz zwyke parametry.
 *
 *  - format... - j.w.,
 */
char *format_string(const char *format, ...)
{
	va_list ap;
	char *tmp;
	
	va_start(ap, format);
	tmp = va_format_string(format, ap);
	va_end(ap);

	return tmp;
}

/*
 * print()
 *
 * drukuje na stdout tekst, biorc pod uwag nazw, nie warto formatu.
 * parametry takie jak zdefiniowano. pierwszy to %1, drugi to %2.
 */
void print(const char *theme, ...)
{
	va_list ap;
	char *tmp;
	
	va_start(ap, theme);
	tmp = va_format_string(format_find(theme), ap);
	
	ui_print("__current", 0, (tmp) ? tmp : "");
	
	xfree(tmp);
	va_end(ap);
}

/*
 * print_status()
 *
 * wywietla tekst w oknie statusu.
 */
void print_status(const char *theme, ...)
{
	va_list ap;
	char *tmp;
	
	va_start(ap, theme);
	tmp = va_format_string(format_find(theme), ap);
	
	ui_print("__status", 0, (tmp) ? tmp : "");
	
	xfree(tmp);
	va_end(ap);
}

/*
 * print_window()
 *
 * wywietla tekst w podanym oknie.
 *  
 *  - target - nazwa okna,
 *  - separate - czy niezbdne jest otwieranie nowego okna?
 *  - theme, ... - tre.
 */
void print_window(const char *target, int separate, const char *theme, ...)
{
	va_list ap;
	char *tmp;
	
	if (!target)
		target = "__current";

	va_start(ap, theme);
	tmp = va_format_string(format_find(theme), ap);
	
	ui_print(target, separate, (tmp) ? tmp : "");
	
	xfree(tmp);
	va_end(ap);
}

/*
 * theme_cache_reset()
 *
 * usuwa cache'owane prompty. przydaje si przy zmianie theme'u.
 */
void theme_cache_reset()
{
	xfree(prompt_cache);
	xfree(prompt2_cache);
	xfree(error_cache);
	
	prompt_cache = prompt2_cache = error_cache = NULL;
	timestamp_cache = NULL;
}

/*
 * format_add()
 *
 * dodaje dan formatk do listy.
 *
 *  - name - nazwa,
 *  - value - warto,
 *  - replace - jeli znajdzie, to zostawia (=0) lub zamienia (=1).
 */
int format_add(const char *name, const char *value, int replace)
{
	struct format f;
	list_t l;
	int hash;

	if (!name || !value)
		return -1;

	hash = ekg_hash(name);

	if (hash == ekg_hash("no_prompt_cache") && !strcasecmp(name, "no_prompt_cache")) {
		no_prompt_cache = 1;
		return 0;
	}
	
	for (l = formats; l; l = l->next) {
		struct format *g = l->data;

		if (hash == g->name_hash && !strcasecmp(name, g->name)) {
			if (replace) {
				xfree(g->value);
				g->value = xstrdup(value);
			}

			return 0;
		}
	}

	f.name = xstrdup(name);
	f.name_hash = ekg_hash(name);
	f.value = xstrdup(value);

	return (list_add(&formats, &f, sizeof(f)) ? 0 : -1);
}

/*
 * format_remove()
 *
 * usuwa formatk o danej nazwie.
 *
 *  - name.
 */
int format_remove(const char *name)
{
	list_t l;

	if (!name)
		return -1;

	for (l = formats; l; l = l->next) {
		struct format *f = l->data;

		if (!strcasecmp(f->name, name)) {
			xfree(f->value);
			xfree(f->name);
			list_remove(&formats, f, 1);
		
			return 0;
		}
	}

        return -1;
}

/*
 * try_open() // funkcja wewntrzna
 *
 * prbuje otworzy plik, jeli jeszcze nie jest otwarty.
 *
 *  - prevfd - deskryptor z poprzedniego wywoania,
 *  - prefix - cieka,
 *  - filename - nazwa pliku.
 */
static FILE *try_open(FILE *prevfd, const char *prefix, const char *filename)
{
	char buf[PATH_MAX];
	int save_errno;
	FILE *f;

	if (prevfd)
		return prevfd;

	if (prefix)
		snprintf(buf, sizeof(buf), "%s/%s", prefix, filename);
	else
		snprintf(buf, sizeof(buf), "%s", filename);

	if ((f = fopen(buf, "r")))
		return f;

	if (prefix)
		snprintf(buf, sizeof(buf), "%s/%s.theme", prefix, filename);
	else
		snprintf(buf, sizeof(buf), "%s.theme", filename);

	save_errno = errno;
	
	if ((f = fopen(buf, "r")))
		return f;

	if (errno == ENOENT)
		errno = save_errno;

	return NULL;
}

/*
 * theme_read()
 *
 * wczytuje opis wygldu z podanego pliku. 
 *
 *  - filename - nazwa pliku z opisem,
 *  - replace - czy zastpowa istniejce wpisy.
 *
 * zwraca 0 jeli wszystko w porzdku, -1 w przypadku bdu.
 */
int theme_read(const char *filename, int replace)
{
        char *buf;
        FILE *f = NULL;

        if (!filename) {
                filename = prepare_path("default.theme", 0);
		if (!filename || !(f = fopen(filename, "r")))
			return -1;
        } else {
		char *fn = xstrdup(filename), *tmp;

		if ((tmp = strchr(fn, ',')))
			*tmp = 0;
		
		errno = ENOENT;
		f = try_open(NULL, NULL, fn);

		if (!strchr(filename, '/')) {
			f = try_open(f, prepare_path("", 0), fn);
			f = try_open(f, prepare_path("themes", 0), fn);
			f = try_open(f, DATADIR "/themes", fn);
		}

		xfree(fn);

		if (!f)
			return -1;
	}

	theme_free();
	theme_init();
	ui_event("theme_init");

	while ((buf = read_file(f))) {
		char *value, *p;

		if (buf[0] == '#') {
			xfree(buf);
			continue;
		}

		if (!(value = strchr(buf, ' '))) {
			xfree(buf);
			continue;
		}

		*value++ = 0;

		for (p = value; *p; p++) {
			if (*p == '\\') {
				if (!*(p + 1))
					break;
				if (*(p + 1) == 'n')
					*p = '\n';
				memmove(p + 1, p + 2, strlen(p + 1));
			}
		}

		if (buf[0] == '-')
			format_remove(buf + 1);
		else
			format_add(buf, value, replace);

		xfree(buf);
        }

        fclose(f);

	theme_cache_reset();

        return 0;
}

/*
 * theme_free()
 *
 * usuwa formatki z pamici.
 */
void theme_free()
{
	list_t l;

	for (l = formats; l; l = l->next) {
		struct format *f = l->data;

		xfree(f->name);
		xfree(f->value);
	}	

	list_destroy(formats, 1);
	formats = NULL;

	theme_cache_reset();
}

/*
 * theme_init()
 *
 * ustawia domylne wartoci formatek.
 */
void theme_init()
{
	theme_cache_reset();

	/* wykorzystywane w innych formatach */
	format_add("prompt", "%K:%g:%G:%n", 1);
	format_add("prompt,speech", " ", 1);
	format_add("prompt2", "%K:%c:%C:%n", 1);
	format_add("prompt2,speech", " ", 1);
	format_add("error", "%K:%r:%R:%n", 1);
	format_add("error,speech", "bd!", 1);
	format_add("timestamp", "%H:%M", 1);
	format_add("timestamp,speech", " ", 1);
	
	/* prompty dla ui-readline */
	format_add("readline_prompt", "% ", 1);
	format_add("readline_prompt_away", "/ ", 1);
	format_add("readline_prompt_invisible", ". ", 1);
	format_add("readline_prompt_query", "%1> ", 1);
	format_add("readline_prompt_win", "%1%% ", 1);
	format_add("readline_prompt_away_win", "%1/ ", 1);
	format_add("readline_prompt_invisible_win", "%1. ", 1);
	format_add("readline_prompt_query_win", "%2:%1> ", 1);
	format_add("readline_prompt_win_act", "%1 (act/%2)%% ", 1);
	format_add("readline_prompt_away_win_act", "%1 (act/%2)/ ", 1);
	format_add("readline_prompt_invisible_win_act", "%1 (act/%2). ", 1);
	format_add("readline_prompt_query_win_act", "%2:%1 (act/%3)> ", 1);
					
	format_add("readline_more", "-- Wcinij Enter by kontynuowa lub Ctrl-D by przerwa --", 1);

	/* prompty i statusy dla ui-ncurses */
	format_add("ncurses_prompt_none", "", 1);
	format_add("ncurses_prompt_query", "[%1] ", 1);
	format_add("statusbar", " %c(%w%{time}%c)%w %{?uin %c(%w%{?!nick uin}%{nick}%c/%{?away %w}%{?avail %W}%{?invisible %K}%{?notavail %k}%{uin}%c) }%c%{?window (%wwin%c/%w%{window}}%{?query %c:%W%{query}}%{?debug %c(%Cdebug}%c)%w%{?activity  %c(%wact%c/%w}%{activity}%{?activity %c)%w}%{?mail  %c(%wmail%c/%w}%{mail}%{?mail %c)}%{?more  %c(%Gmore%c)}", 1);
	format_add("header", " %{?query %c(%{?query_away %w}%{?query_avail %W}%{?query_invisible %K}%{?query_notavail %k}%{query}%{?query_descr %c/%w%{query_descr}}%c) %{?query_ip (%wip%c/%w%{query_ip}%c)}}%{?!query %c(%wekg%c/%w%{version}%c) (%w%{url}%c)}", 1);
	format_add("ncurses_timestamp", "%H:%M", 1);

	/* dla funkcji format_user() */
	format_add("known_user", "%T%1%n/%2", 1);
	format_add("known_user,speech", "%1", 1);
	format_add("unknown_user", "%T%1%n", 1);
	
	/* czsto wykorzystywane, rne, przydatne itd. */
	format_add("none", "%1\n", 1);
	format_add("generic", "%> %1\n", 1);
	format_add("generic2", "%) %1\n", 1);
	format_add("generic_error", "%! %1\n", 1);
	format_add("debug", "%n%1\n", 1);
	format_add("not_enough_params", "%! Za mao parametrw. Sprbuj %Thelp %1%n\n", 1);
	format_add("invalid_params", "%! Nieprawidowe parametry. Sprbuj %Thelp %1%n\n", 1);
	format_add("invalid_uin", "%! Nieprawidowy numer uytkownika\n", 1);
	format_add("invalid_nick", "%! Nieprawidowa nazwa uytkownika\n", 1);
	format_add("user_not_found", "%! Nie znaleziono uytkownika %T%1%n\n", 1);
	format_add("not_implemented", "%! Tej funkcji jeszcze nie ma\n", 1);
	format_add("unknown_command", "%! Nieznane polecenie: %T%1%n\n", 1);
	format_add("welcome", "%> %TEKG-%1%n (Eksperymentalny Klient Gadu-Gadu)\n%> Program jest rozprowadzany na zasadach licencji GPL v2\n%> %RPrzed uyciem wcinij F1 lub wpisz ,,help''%n\n\n", 1);
	format_add("welcome,speech", "witamy w e k g", 1);
	format_add("ekg_version", "%) EKG - Eksperymentalny Klient Gadu-Gadu (%T%1%n)\n%) libgadu-%1 (protok %2, klient %3)\n%) skompilowano: %4\n", 1);
	format_add("secure", "%Y(szyfrowane)%n ", 1);
	format_add("log_failed", "%! Nie mona zapisa do historii: %1\n", 1);

	/* mail */
	format_add("new_mail_one", "%) Masz now wiadomo email\n", 1);
	format_add("new_mail_two_four", "%) Masz %1 nowe wiadomoci email\n", 1);
	format_add("new_mail_more", "%) Masz %1 nowych wiadomoci email\n", 1);

	/* add, del */
	format_add("user_added", "%> Dopisano %T%1%n do listy kontaktw\n", 1);
	format_add("user_deleted", "%) Usunito %T%1%n z listy kontaktw\n", 1);
	format_add("user_cleared_list", "%) Wyczyszczono list kontaktw\n", 1);
	format_add("user_exists", "%! %T%1%n ju istnieje w licie kontaktw\n", 1);
	format_add("user_exists_other", "%! %T%1%n ju istnieje w licie kontaktw jako %2\n", 1);

	/* zmiany stanu */
	format_add("away", "%> Zmieniono stan na zajty\n", 1);
	format_add("away_descr", "%> Zmieniono stan na zajty: %T%1%n%2\n", 1);
	format_add("back", "%> Zmieniono stan na dostpny\n", 1);
	format_add("back_descr", "%> Zmieniono stan na dostpny: %T%1%n%2%n\n", 1);
	format_add("invisible", "%> Zmieniono stan na niewidoczny\n", 1);
	format_add("invisible_descr", "%> Zmieniono stan na niewidoczny: %T%1%n%2\n", 1);
	format_add("private_mode_is_on", "%> Tryb ,,tylko dla znajomych'' jest wczony\n", 1);
	format_add("private_mode_is_off", "%> Tryb ,,tylko dla znajomych'' jest wyczony\n", 1);
	format_add("private_mode_on", "%> Wczono tryb ,,tylko dla znajomych''\n", 1);
	format_add("private_mode_off", "%> Wyczono tryb ,,tylko dla znajomych''\n", 1);
	format_add("private_mode_invalid", "%! Nieprawidowa warto\n", 1);
	format_add("descr_too_long", "%! Dugo opisu przekracza limit. Ilo ucitych znakw: %T%1%n\n", 1);
	
	/* pomoc */
	format_add("help", "%> %T%1%n%2 - %3\n", 1);
	format_add("help_more", "%) %|%1\n", 1);
	format_add("help_alias", "%) %T%1%n jest aliasem i nie posiada opisu\n", 1);
	format_add("help_footer", "\n%> %|Wicej szczegw na temat komend zwrci %Thelp <komenda>%n. Poprzedzenie komendy znakiem %T^%n spowoduje ukrycie jej wyniku. Zamiast parametru <numer/alias> mona uy znaku %T$%n oznaczajcego aktualnego rozmwc.\n\n", 1);
	format_add("help_quick", "%> %|Przed uyciem przeczytaj ulotk. Plik %Tdocs/ULOTKA%n zawiera krtki przewodnik po zaczonej dokumentacji. Jeli go nie masz, moesz cign pakiet ze strony %Thttp://dev.null.pl/ekg/%n\n", 1);
	format_add("help_set_file_not_found", "%! Nie znaleziono opisu zmiennych (nieprawidowa instalacja)\n", 1);
	format_add("help_set_var_not_found", "%! Nie znaleziono opisu zmiennej %T%1%n\n", 1);
	format_add("help_set_header", "%> %T%1%n (%2, domylna warto: %3)\n%>\n", 1);
	format_add("help_set_body", "%> %|%1\n", 1);
	format_add("help_set_footer", "", 1);

	/* ignore, unignore, block, unblock */
	format_add("ignored_added", "%> Dodano %T%1%n do listy ignorowanych\n", 1);
	format_add("ignored_modified", "%> Zmodyfikowano poziom ignorowania %T%1%n\n", 1);
	format_add("ignored_deleted", "%) Usunito %1 z listy ignorowanych\n", 1);
	format_add("ignored_deleted_all", "%) Usunito wszystkich z listy ignorowanych\n", 1);
	format_add("ignored_exist", "%! %1 jest ju na licie ignorowanych\n", 1);
	format_add("ignored_list", "%> %1 %2\n", 1);
	format_add("ignored_list_unknown_sender", "%> Ignorowanie wiadomoci od nieznanych uytkownikw\n", 1);
	format_add("ignored_list_empty", "%! Lista ignorowanych uytkownikw jest pusta\n", 1);
	format_add("error_not_ignored", "%! %1 nie jest na licie ignorowanych\n", 1);
	format_add("blocked_added", "%> Dodano %T%1%n do listy blokowanych\n", 1);
	format_add("blocked_deleted", "%) Usunito %1 z listy blokowanych\n", 1);
	format_add("blocked_deleted_all", "%) Usunito wszystkich z listy blokowanych\n", 1);
	format_add("blocked_exist", "%! %1 jest ju na licie blokowanych\n", 1);
	format_add("blocked_list", "%> %1\n", 1);
	format_add("blocked_list_empty", "%! Lista blokowanych uytkownikw jest pusta\n", 1);
	format_add("error_not_blocked", "%! %1 nie jest na licie blokowanych\n", 1);

	/* lista kontaktw */
	format_add("list_empty", "%! Lista kontaktw jest pusta\n", 1);
	format_add("list_avail", "%> %1 %Y(dostpn%@2)%n %b%3:%4%n\n", 1);
	format_add("list_avail_descr", "%> %1 %Y(dostpn%@2: %n%5%Y)%n %b%3:%4%n\n", 1);
	format_add("list_busy", "%> %1 %G(zajt%@2)%n %b%3:%4%n\n", 1);
	format_add("list_busy_descr", "%> %1 %G(zajt%@2: %n%5%G)%n %b%3:%4%n\n", 1);
	format_add("list_not_avail", "%> %1 %r(niedostpn%@2)%n\n", 1);
	format_add("list_not_avail_descr", "%> %1 %r(niedostpn%@2: %n%5%r)%n\n", 1);
	format_add("list_invisible", "%> %1 %c(niewidoczn%@2)%n %b%3:%4%n\n", 1);
	format_add("list_invisible_descr", "%> %1 %c(niewidoczn%@2: %n%5%c)%n %b%3:%4%n\n", 1);
	format_add("list_blocked", "%> %1 %m(blokujc%@2)%n\n", 1);
	format_add("list_unknown", "%> %1\n", 1);
	format_add("modify_offline", "%> %1 nie bdzie widzie naszego stanu\n", 1);
	format_add("modify_online", "%> %1 bdzie widzie nasz stan\n", 1);
	format_add("modify_done", "%> Zmieniono wpis w licie kontaktw\n", 1);

	/* lista kontaktw z boku ekranu */
	format_add("contacts_header", "", 1);
	format_add("contacts_header_group", "%K %1%n", 1);
	format_add("contacts_avail_header", "", 1);
	format_add("contacts_avail", " %Y%1%n", 1);
	format_add("contacts_avail_descr", "%Ki%Y%1%n", 1);
	format_add("contacts_avail_descr_full", "%Ki%Y%1%n %2", 1);
	format_add("contacts_avail_footer", "", 1);
	format_add("contacts_busy_header", "", 1);
	format_add("contacts_busy", " %G%1%n", 1);
	format_add("contacts_busy_descr", "%Ki%G%1%n", 1);
	format_add("contacts_busy_descr_full", "%Ki%G%1%n %2", 1);
	format_add("contacts_busy_footer", "", 1);
	format_add("contacts_not_avail_header", "", 1);
	format_add("contacts_not_avail", " %r%1%n", 1);
	format_add("contacts_not_avail_descr", "%Ki%r%1%n", 1);
	format_add("contacts_not_avail_descr_full", "%Ki%r%1%n %2", 1);
	format_add("contacts_not_avail_footer", "", 1);
	format_add("contacts_invisible_header", "", 1);
	format_add("contacts_invisible", " %c%1%n", 1);
	format_add("contacts_invisible_descr", "%Ki%c%1%n", 1);
	format_add("contacts_invisible_descr_full", "%Ki%c%1%n %2", 1);
	format_add("contacts_invisible_footer", "", 1);
	format_add("contacts_blocking_header", "", 1);
	format_add("contacts_blocking", " %m%1%n", 1);
	format_add("contacts_blocking_footer", "", 1);
	format_add("contacts_footer", "", 1);
	format_add("contacts_footer_group", "", 1);
		
	/* egnamy si, zapisujemy konfiguracj */
	format_add("quit", "%> Papa\n", 1);
	format_add("quit_descr", "%> Papa: %T%1%n%2\n", 1);
	format_add("config_changed", "Zapisa now konfiguracj? (tak/nie) ", 1);
	format_add("saved", "%> Zapisano ustawienia\n", 1);
	format_add("error_saving", "%! Podczas zapisu ustawie wystpi bd\n", 1);

	/* przychodzce wiadomoci */
	format_add("message_header", "%g.-- %n%1 %c%2%n%4%g--- -- -%n\n", 1);
	format_add("message_conference_header", "%g.-- %g[%T%3%g] -- %n%1 %c%2%4%g--- -- -%n\n", 1);
	format_add("message_footer", "%g`----- ---- --- -- -%n\n", 1);
	format_add("message_line", "%g|%n %|%1%n\n", 1);
	format_add("message_line_width", "-8", 1);
	format_add("message_timestamp", "(%Y-%m-%d %H:%M) ", 1);
	format_add("message_timestamp_today", "(%H:%M) ", 1);
	format_add("message_timestamp_now", "", 1);

	format_add("message_header,speech", "wiadomo od %1: ", 1);
	format_add("message_conference_header,speech", "wiadomo od %1 w konferencji %3: ", 1);
	format_add("message_line,speech", "%1\n", 1);
	format_add("message_footer,speech", ".", 1);

	format_add("chat_header", "%c.-- %n%1 %c%2%n%4%c--- -- -%n\n", 1);
	format_add("chat_conference_header", "%c.-- %c[%T%3%c] -- %n%1 %c%2%4%c--- -- -%n\n", 1);
	format_add("chat_footer", "%c`----- ---- --- -- -%n\n", 1);
	format_add("chat_line", "%c|%n %|%1%n\n", 1);
	format_add("chat_line_width", "-8", 1);
	format_add("chat_timestamp", "(%Y-%m-%d %H:%M) ", 1);
	format_add("chat_timestamp_today", "(%H:%M) ", 1);
	format_add("chat_timestamp_now", "", 1);

	format_add("chat_header,speech", "wiadomo od %1: ", 1);
	format_add("chat_conference_header,speech", "wiadomo od %1 w konferencji %3: ", 1);
	format_add("chat_line,speech", "%1\n", 1);
	format_add("chat_footer,speech", ".", 1);

	format_add("sent_header", "%b.-- %n%1 %4%b--- -- -%n\n", 1);
	format_add("sent_conference_header", "%b.-- %b[%T%3%b] -- %4%n%1 %b--- -- -%n\n", 1);
	format_add("sent_footer", "%b`----- ---- --- -- -%n\n", 1);
	format_add("sent_line", "%b|%n %|%1%n\n", 1);
	format_add("sent_line_width", "-8", 1);
	format_add("sent_timestamp", "%H:%M", 1);

	format_add("sysmsg_header", "%m.-- %TWiadomo systemowa%m --- -- -%n\n", 1);
	format_add("sysmsg_line", "%m|%n %|%1%n\n", 1);
	format_add("sysmsg_line_width", "-8", 1);
	format_add("sysmsg_footer", "%m`----- ---- --- -- -%n\n", 1);	

	format_add("sysmsg_header,speech", "wiadomo systemowa:", 1);
	format_add("sysmsg_line,speech", "%1\n", 1);
	format_add("sysmsg_footer,speech", ".", 1);

	/* potwierdzenia wiadomoci */
	format_add("ack_queued", "%> Wiadomo do %1 zostanie dostarczona pniej\n", 1);
	format_add("ack_delivered", "%> Wiadomo do %1 zostaa dostarczona\n", 1);
	format_add("ack_filtered", "%! %|Wiadomo do %1 najprawdopodobniej nie zostaa dostarczona, poniewa dana osoba jest niedostpna, a serwer twierdzi, e dorczy wiadomo. Sytuacja taka ma miejsce, gdy wiadomo zostaa odrzucona przez filtry serwera (np. zawiera adres strony WWW)\n", 1);
	format_add("message_too_long", "%! Wiadomo jest zbyt duga i zostaa skrcona\n", 1);

	/* ludzie zmieniaj stan */
	format_add("status_avail", "%> %1 jest dostpn%@2\n", 1);
	format_add("status_avail_descr", "%> %1 jest dostpn%@2: %T%3%n\n", 1);
	format_add("status_busy", "%> %1 jest zajt%@2\n", 1);
	format_add("status_busy_descr", "%> %1 jest zajt%@2: %T%3%n\n", 1);
	format_add("status_not_avail", "%> %1 jest niedostpn%@2\n", 1);
	format_add("status_not_avail_descr", "%> %1 jest niedostpn%@2: %T%3%n\n", 1);
	format_add("status_invisible", "%> %1 jest niewidoczn%@2\n", 1);
	format_add("status_invisible_descr", "%> %1 jest niewidoczn%@2: %T%3%n\n", 1);

	format_add("auto_away", "%> Automagicznie zmieniono stan na zajty po %1 nieaktywnoci\n", 1);
	format_add("auto_away_descr", "%> Automagicznie zmieniono stan na zajty po %1 nieaktywnoci: %T%2%n%3\n", 1);
	format_add("auto_back", "%> Automagicznie zmieniono stan na dostpny\n", 1);
	format_add("auto_back_descr", "%> Automagicznie zmieniono stan na dostpny: %T%2%n%3\n", 1);

	/* poczenie z serwerem */
	format_add("connecting", "%> cz si z serwerem...\n", 1);
	format_add("conn_failed", "%! Poczenie nie udao si: %1\n", 1);
	format_add("conn_failed_resolving", "Nie znaleziono serwera", 1);
	format_add("conn_failed_connecting", "Nie mona poczy si z serwerem", 1);
	format_add("conn_failed_invalid", "Nieprawidowa odpowied serwera", 1);
	format_add("conn_failed_disconnected", "Serwer zerwa poczenie", 1);
	format_add("conn_failed_password", "Nieprawidowe haso", 1);
	format_add("conn_failed_404", "Bd serwera HTTP", 1);
	format_add("conn_failed_tls", "Bd negocjacji TLS", 1);
	format_add("conn_failed_memory", "Brak pamici", 1);
	format_add("conn_stopped", "%! Przerwano czenie\n", 1);
	format_add("conn_timeout", "%! Przekroczono limit czasu operacji czenia z serwerem\n", 1);
	format_add("connected", "%> Poczono\n", 1);
	format_add("connected_descr", "%> Poczono: %T%1%n\n", 1);
	format_add("disconnected", "%> Rozczono\n", 1);
	format_add("disconnected_descr", "%> Rozczono: %T%1%n\n", 1);
	format_add("already_connected", "%! Klient jest ju poczony. Wpisz %Treconnect%n aby poczy ponownie\n", 1);
	format_add("during_connect", "%! czenie trwa. Wpisz %Tdisconnect%n aby przerwa\n", 1);
	format_add("conn_broken", "%! Poczenie zostao przerwane\n", 1);
	format_add("conn_disconnected", "%! Serwer zerwa poczenie\n", 1);
	format_add("not_connected", "%! Brak poczenia z serwerem. Wpisz %Tconnect%n\n", 1);
	format_add("not_connected_msg_queued", "%! Brak poczenia z serwerem. Wiadomo bdzie wysana po poczeniu.%n\n", 1);
	format_add("invalid_local_ip", "%! Nieprawidowy adres lokalny.\n", 1);
	
	/* obsuga motyww */
	format_add("theme_loaded", "%> Wczytano motyw %T%1%n\n", 1);
	format_add("theme_default", "%> Ustawiono domylny motyw\n", 1);
	format_add("error_loading_theme", "%! Bd podczas adowania motywu: %1\n", 1);

	/* zmienne, konfiguracja */
	format_add("variable", "%> %1 = %2\n", 1);
	format_add("variable_not_found", "%! Nieznana zmienna: %T%1%n\n", 1);
	format_add("variable_invalid", "%! Nieprawidowa warto zmiennej\n", 1);
	format_add("no_config", "%! Niekompletna konfiguracja. Wpisz:\n%!   %Tset uin <numerek-gg>%n\n%!   %Tset password <haso>%n\n%!   %Tset email <adres-email>%n\n%!   %Tsave%n\n%! Nastpnie wydaj polecenie:\n%!   %Tconnect%n\n%! Jeli nie masz swojego numerka, wpisz:\n%!   %Ttoken%n\n%!   %Tregister <e-mail> <haso> <token>%n\n\n%> %|Po poczeniu, nowe okna rozmowy bd tworzone automatycznie, gdy kto przyle wiadomo. Aby przej do okna o podanym numerze naley wcisn %TAlt-numer%n lub %TEsc%n, a nastpnie cyfr. Aby rozpocz rozmow, naley uy polecenia %Tquery%n. Aby doda kogo do listy kontaktw, naley uy polecenia %Tadd%n. Wszystkie kombinacje klawiszy s opisane w pliku %TREADME%n, a list komend wywietla polecenie %Thelp%n.\n\n", 2);
	format_add("no_config,speech", "niekompletna konfiguracja. wpisz set uin, a potem numer gadu-gadu, potem set pasord, za tym swoje haso, a nastpnie set imejl (bez mylnika), a za tym swj adres imejl. wpisz sejf, eby zapisa ustawienia. wpisz konekt by si poczy. niestety rejestracja wymaga odczytania z ekranu wysanego przez serwer tokenu. po poczeniu, nowe okna rozmowy bd tworzone automatycznie, gdy kto przyle wiadomo. aby przej do okna o podanym numerze, naley wcisn alt-numer lub eskejp, a nastpnie cyfr. aby rozpocz rozmow, naley uy polecenia kery. aby doda kogo do listy kontaktw, naley uy polecenia edd. wszystkie kombinacje klawiszy s opisane w pliku ridmi, a list komend wywietla polecenie help.", 1);
	format_add("error_reading_config", "%! Nie mona odczyta pliku konfiguracyjnego: %1\n", 1);
	format_add("config_read_success", "%> Wczytano plik konfiguracyjny %T%1%n\n", 1);
        format_add("config_line_incorrect", "%! Nieprawidowa linia '%T%1%n', pomijam\n", 1);
	format_add("autosaved", "%> Automatycznie zapisano ustawienia\n", 1);
	
	/* rejestracja nowego numeru */
	format_add("register", "%> Rejestracja poprawna. Wygrany numerek: %T%1%n\n", 1);
	format_add("register_failed", "%! Bd podczas rejestracji: %1\n", 1);
	format_add("register_pending", "%! Rejestracja w toku\n", 1);
	format_add("register_timeout", "%! Przekroczono limit czasu operacji rejestrowania\n", 1);
	format_add("registered_today", "%! Ju zarejestrowano jeden numer. Nie naduywaj\n", 1);

	/* kasowanie konta uytkownika z katalogu publiczengo */
	format_add("unregister", "%> Konto %T%1%n wykasowano\n", 1);
	format_add("unregister_timeout", "%! Przekroczono limit czasu operacji usuwania konta\n", 1);
	format_add("unregister_bad_uin", "%! Niepoprawny numer: %T%1%n\n", 1);
	format_add("unregister_failed", "%! Bd podczas usuwania konta: %1\n", 1);
	
	/* przypomnienie hasa */
	format_add("remind", "%> Haso zostao wysane\n", 1);
	format_add("remind_failed", "%! Bd podczas wysyania hasa: %1\n", 1);
	format_add("remind_timeout", "%! Przekroczono limit czasu operacji wysania hasa\n", 1);
	
	/* zmiana hasa */
	format_add("passwd", "%> Haso zostao zmienione\n", 1);
	format_add("passwd_failed", "%! Bd podczas zmiany hasa: %1\n", 1);
	format_add("passwd_timeout", "%! Przekroczono limit czasu operacji zmiany hasa\n", 1);
	format_add("passwd_email", "%! Naley ustawi zmienn %Temail%n\n", 1);
	
	/* zmiana informacji w katalogu publicznym */
	format_add("change", "%> Informacje w katalogu publicznym zostay zmienione\n", 1);
	format_add("change_failed", "%! Bd podczas zmiany informacji w katalogu publicznym\n", 1);
	
	/* pobieranie tokenu */
	format_add("token", "%> Token zapisano do pliku %T%1%n\n", 1);
	format_add("token_ocr", "%> Token: %T%1%n\n", 1);
	format_add("token_body", "%1\n", 1);
	format_add("token_failed", "%! Bd pobierania tokenu: %1\n", 1);
	format_add("token_timeout", "%! Przekroczono limit czasu pobierania tokenu\n", 1);
	format_add("token_unsupported", "%! System operacyjny nie zawiera funkcji potrzebnych do obsugi tokenw\n", 1);
	format_add("token_missing", "%! Naley najpierw pobra z serwera token komend %Ttoken%n\n", 1);

	/* sesemesy */
	format_add("sms_error", "%! Bd wysyania SMS: %1\n", 1);
	format_add("sms_unknown", "%! %1 nie ma podanego numeru komrki\n", 1);
	format_add("sms_sent", "%> SMS do %T%1%n zosta wysany\n", 1);
	format_add("sms_failed", "%! SMS do %T%1%n nie zosta wysany\n", 1);
	format_add("sms_msg", "EKG: msg %1 %# >> %2", 1);
	format_add("sms_chat", "EKG: chat %1 %# >> %2", 1);
	format_add("sms_conf", "EKG: conf %1 %# >> %2", 1);

	/* wyszukiwanie uytkownikw */
	format_add("search_failed", "%! Wystpi bd podczas szukania: %1\n", 1);
	format_add("search_not_found", "%! Nie znaleziono\n", 1);
	format_add("search_no_last", "%! Brak wynikw ostatniego wyszukiwania\n", 1);
	format_add("search_no_last_nickname", "%! Brak pseudonimu w ostatnim wyszukiwaniu\n", 1);
	format_add("search_stopped", "%> Zatrzymano wyszukiwanie\n", 1);

	/* 1 uin, 2 name, 3 nick, 4 city, 5 born, 6 gender, 7 active */
	format_add("search_results_multi_active", "%Y<>%n", 1);
	format_add("search_results_multi_busy", "%G<>%n", 1);
	format_add("search_results_multi_invisible", "%c<>%n", 1);
	format_add("search_results_multi_inactive", "  ", 1);
	format_add("search_results_multi_unknown", "-", 1);
/*	format_add("search_results_multi_female", "k", 1); */
/*	format_add("search_results_multi_male", "m", 1); */
	format_add("search_results_multi", "%7 %[-7]1 %K|%n %[12]3 %K|%n %[12]2 %K|%n %[4]5 %K|%n %[12]4\n", 1);

	format_add("search_results_single_active", "%Y(dostpn%@1)%n", 1);
	format_add("search_results_single_busy", "%G(zajt%@1)%n", 1);
	format_add("search_results_single_inactive", "%r(niedostpn%@1)%n", 1);
	format_add("search_results_single_invisible", "%c(niewidoczn%@1)%n", 1);
	format_add("search_results_single_unknown", "%T-%n", 1);
/*	format_add("search_results_single_female", "%Mkobieta%n", 1); */
/*	format_add("search_results_single_male", "%Cmczyzna%n", 1); */
	format_add("search_results_single", "%) Pseudonim: %T%3%n\n%) Numerek: %T%1%n %7\n%) Imi i nazwisko: %T%2%n\n%) Miejscowo: %T%4%n\n%) Rok urodzenia: %T%5%n\n", 1);

	/* exec */
	format_add("process", "%> %(-5)1 %2\n", 1);
	format_add("no_processes", "%! Nie ma dziaajcych procesw\n", 1);
	format_add("process_exit", "%> Proces %1 (%2) zakoczy dziaanie z wynikiem %3\n", 1);
	format_add("exec", "%1\n",1);
	format_add("exec_error", "%! Bd uruchamiania procesu: %1\n", 1);
	format_add("exec_prompt", "$ %1\n", 1);

	/* szczegowe informacje o uytkowniku */
	format_add("user_info_header", "%K.--%n %T%1%n/%2 %K--- -- -%n\n", 1);
	format_add("user_info_nickname", "%K| %nPseudonim: %T%1%n\n", 1);
	format_add("user_info_name", "%K| %nImi i nazwisko: %T%1 %2%n\n", 1);
	format_add("user_info_email", "%K| %nE-mail: %T%1%n\n", 1);
	format_add("user_info_status", "%K| %nStan: %T%1%n\n", 1);
	format_add("user_info_block", "%K| %nBlokowan%@1\n", 1);
	format_add("user_info_offline", "%K| %nNie widzi stanu\n", 1);
	format_add("user_info_not_in_contacts", "%K| %nNie ma nas w swoich kontaktach\n", 1);
	format_add("user_info_firewalled", "%K| %nZnajduje si za firewall/NAT\n", 1);
	format_add("user_info_ip", "%K| %nAdres: %T%1%n\n", 1);
	format_add("user_info_mobile", "%K| %nTelefon: %T%1%n\n", 1);
	format_add("user_info_groups", "%K| %nGrupy: %T%1%n\n", 1);
	format_add("user_info_never_seen", "%K| %nNie widziano podczas tej sesji\n", 1);
	format_add("user_info_last_seen", "%K| %nOstatnio widziano: %T%1%n\n", 1);
	format_add("user_info_last_seen_time", "%Y-%m-%d %H:%M", 1);
	format_add("user_info_last_descr", "%K| %nOstatni opis: %T%1%n\n", 1);
	format_add("user_info_version", "%K| %nWersja klienta: %T%1%n\n", 1);
	format_add("user_info_voip", "%K| %nObsuguje rozmowy gosowe\n", 1);
	format_add("user_info_last_ip","%K| %nOstatni adres IP: %T%1%n\n", 1);
	format_add("user_info_footer", "%K`----- ---- --- -- -%n\n", 1);

	format_add("user_info_avail", "%Ydostpn%@1%n", 1);
	format_add("user_info_avail_descr", "%Ydostpn%@1%n (%2)", 1);
	format_add("user_info_busy", "%Gzajt%@1%n", 1);
	format_add("user_info_busy_descr", "%Gzajt%@1%n (%2)", 1);
	format_add("user_info_not_avail", "%rniedostpn%@1%n", 1);
	format_add("user_info_not_avail_descr", "%rniedostpn%@1%n (%2)", 1);
	format_add("user_info_invisible", "%cniewidoczn%@1%n", 1);
	format_add("user_info_invisible_descr", "%cniewidoczn%@1%n (%2)", 1);
	format_add("user_info_blocked", "%mblokujc%@1%n", 1);
	format_add("user_info_unknown", "%Mnieznany%n", 1);

	/* grupy */
	format_add("group_members", "%> %|Grupa %T%1%n: %2\n", 1);
	format_add("group_member_already", "%! %1 naley ju do grupy %T%2%n\n", 1);
	format_add("group_member_not_yet", "%! %1 nie naley do grupy %T%2%n\n", 1);
	format_add("group_empty", "%! Grupa %T%1%n jest pusta\n", 1);

	/* status */
	format_add("show_status_profile", "%) Profil: %T%1%n\n", 1);
	format_add("show_status_uin", "%) Numer: %T%1%n\n", 1);
	format_add("show_status_uin_nick", "%) Numer: %T%1%n (%T%2%n)\n", 1);
	format_add("show_status_status", "%) Aktualny stan: %T%1%2%n\n", 1);
	format_add("show_status_server", "%) Aktualny serwer: %T%1%n:%T%2%n\n", 1);
	format_add("show_status_server_tls", "%) Aktualny serwer: %T%1%n:%T%2%n (poczenie szyfrowane)\n", 1);
	format_add("show_status_avail", "%Ydostpny%n", 1);
	format_add("show_status_avail_descr", "%Ydostpny%n (%T%1%n%2)", 1);
	format_add("show_status_busy", "%Gzajty%n", 1);
	format_add("show_status_busy_descr", "%Gzajty%n (%T%1%n%2)", 1);
	format_add("show_status_invisible", "%cniewidoczny%n", 1);
	format_add("show_status_invisible_descr", "%cniewidoczny%n (%T%1%n%2)", 1);
	format_add("show_status_not_avail", "%rniedostpny%n", 1);
	format_add("show_status_private_on", ", tylko dla znajomych", 1);
	format_add("show_status_private_off", "", 1);
	format_add("show_status_connected_since", "%) Poczony od: %T%1%n\n", 1);
	format_add("show_status_disconnected_since", "%) Rozczony od: %T%1%n\n", 1);
	format_add("show_status_last_conn_event", "%Y-%m-%d %H:%M", 1);
	format_add("show_status_last_conn_event_today", "%H:%M", 1);
	format_add("show_status_ekg_started_since", "%) Program dziaa od: %T%1%n\n", 1);
	format_add("show_status_ekg_started", "%Y-%m-%d %H:%M", 1);
	format_add("show_status_ekg_started_today", "%H:%M", 1);
	format_add("show_status_msg_queue", "%) Ilo wiadomoci w kolejce do wysania: %T%1%n\n", 1);

	/* aliasy */
	format_add("aliases_list_empty", "%! Brak aliasw\n", 1);
	format_add("aliases_list", "%> %T%1%n: %2\n", 1);
	format_add("aliases_list_next", "%> %3  %2\n", 1);
	format_add("aliases_add", "%> Utworzono alias %T%1%n\n", 1);
	format_add("aliases_append", "%> Dodano do aliasu %T%1%n\n", 1);
	format_add("aliases_del", "%) Usunito alias %T%1%n\n", 1);
	format_add("aliases_del_all", "%) Usunito wszystkie aliasy\n", 1);
	format_add("aliases_exist", "%! Alias %T%1%n ju istnieje\n", 1);
	format_add("aliases_noexist", "%! Alias %T%1%n nie istnieje\n", 1);
	format_add("aliases_command", "%! %T%1%n jest wbudowan komend\n", 1);
	format_add("aliases_not_enough_params", "%! Alias %T%1%n wymaga wikszej iloci parametrw\n", 1);

	/* poczenia bezporednie */
	format_add("dcc_attack", "%! %|Program otrzyma zbyt wiele da bezporednich pocze, ostatnie od %1\n", 1);
	format_add("dcc_spoof", "%! %|Uytkownik %1 poda serwerowi adres %T%2%n, ale poczy si z nami z adresu %T%3%n. Moliwe, e kto prbuje si pod niego podszy. Zachowaj ostrono!\n", 1);
	format_add("dcc_limit", "%! %|Przekroczono limit bezporednich pocze i dla bezpieczestwa zostay one wyczone. Aby je wczy ponownie, naley wpisa polecenie %Tset dcc 1%n i poczy si ponownie. Limit mona zmieni za pomoc zmiennej %Tdcc_limit%n.\n", 1);
	format_add("dcc_create_error", "%! Nie mona wczy pocze bezporednich: %1\n", 1);
	format_add("dcc_error_network", "%! Bd transmisji z %1\n", 1);
	format_add("dcc_error_refused", "%! Poczenie z %1 zostao odrzucone\n", 1);
	format_add("dcc_error_unknown", "%! Nieznany bd poczenia bezporedniego\n", 1);
	format_add("dcc_error_handshake", "%! Nie mona nawiza poczenia z %1\n", 1);
	format_add("dcc_user_aint_dcc", "%! %1 nie ma wczonej obsugi pocze bezporednich\n", 1);
	format_add("dcc_timeout", "%! Przekroczono limit czasu operacji bezporedniego poczenia z %1\n", 1);
	format_add("dcc_not_supported", "%! Opcja %T%1%n nie jest jeszcze obsugiwana\n", 1);
	format_add("dcc_open_error", "%! Nie mona otworzy %T%1%n: %2\n", 1);
	format_add("dcc_show_pending_header", "%> Poczenia oczekujce:\n", 1);
	format_add("dcc_show_pending_send", "%) #%1, %2, wysyanie %T%3%n\n", 1);
	format_add("dcc_show_pending_get", "%) #%1, %2, odbir %T%3%n\n", 1);
	format_add("dcc_show_pending_voice", "%) #%1, %2, rozmowa\n", 1);
	format_add("dcc_show_active_header", "%> Poczenia aktywne:\n", 1);
	format_add("dcc_show_active_send", "%) #%1, %2, wysyanie %T%3%n, %T%4b%n z %T%5b%n (%6%%)\n", 1);
	format_add("dcc_show_active_get", "%) #%1, %2, odbir %T%3%n, %T%4b%n z %T%5b%n (%6%%)\n", 1);
	format_add("dcc_show_active_voice", "%) #%1, %2, rozmowa\n", 1);
	format_add("dcc_show_empty", "%! Brak bezporednich pocze\n", 1);
	format_add("dcc_receiving_already", "%! Plik %T%1%n od uytkownika %2 jest ju pobierany\n", 1);

	format_add("dcc_done_get", "%> Zakoczono pobieranie pliku %T%2%n od %1\n", 1);
	format_add("dcc_done_send", "%> Zakoczono wysyanie pliku %T%2%n do %1\n", 1);
	format_add("dcc_close", "%) Zamknito poczenie z %1\n", 1);

	format_add("dcc_voice_offer", "%) %1 chce rozmawia\n%) Wpisz %Tdcc voice #%2%n, by rozpocz rozmow, lub %Tdcc close #%2%n, by anulowa\n", 1);
	format_add("dcc_voice_running", "%! Mona prowadzi tylko jedn rozmow gosow na raz\n", 1);
	format_add("dcc_voice_unsupported", "%! Nie wkompilowano obsugi rozmw gosowych. Przeczytaj %Tdocs/voip.txt%n\n", 1);
	format_add("dcc_get_offer", "%) %1 przesya plik %T%2%n o rozmiarze %T%3b%n\n%) Wpisz %Tdcc get #%4%n, by go odebra, lub %Tdcc close #%4%n, by anulowa\n", 1);
	format_add("dcc_get_offer_resume", "%) Plik istnieje ju na dysku, wic mona wznowi pobieranie poleceniem %Tdcc resume #%4%n\n", 1);
	format_add("dcc_get_getting", "%) Rozpoczto pobieranie pliku %T%2%n od %1\n", 1);
	format_add("dcc_get_cant_create", "%! Nie mona utworzy pliku %T%1%n\n", 1);
	format_add("dcc_not_found", "%! Nie znaleziono poczenia %T%1%n\n", 1);
	format_add("dcc_invalid_ip", "%! Nieprawidowy adres IP\n", 1);
	format_add("dcc_user_not_avail", "%! %1 musi by aktywn%@1, by mc nawiza poczenie\n", 1);

	/* query */
	format_add("query_started", "%) Rozpoczto rozmow z %T%1%n\n", 1);
	format_add("query_started_window", "%) Wcinij %TAlt-G%n by ignorowa, %TAlt-K%n by zamkn okno\n", 1);
	format_add("query_finished", "%) Zakoczono rozmow z %T%1%n\n", 1);
	format_add("query_exist", "%! Rozmowa z %T%1%n jest ju prowadzona w okienku nr %T%2%n\n", 1);

	/* zdarzenia */
        format_add("events_list_empty", "%! Brak zdarze\n", 1);
        format_add("events_list", "%> %4, on %1 %2 %3\n", 1);
	format_add("events_list_inactive", "%> %4, on %1 %2 %3 %K(nieaktywne)%n\n", 1);
        format_add("events_add", "%> Dodano zdarzenie %T%1%n\n", 1);
        format_add("events_del", "%) Usunito zdarzenie %T%1%n\n", 1);
        format_add("events_del_all", "%) Usunito wszystkie zdarzenia\n", 1);
        format_add("events_exist", "%! Zdarzenie %T%1%n istnieje dla %2\n", 1);
        format_add("events_del_noexist", "%! Zdarzenie %T%1%n nie istnieje\n", 1);
        format_add("events_seq_not_found", "%! Sekwencja %T%1%n nie znaleziona\n", 1);
	format_add("events_seq_incorrect", "%! Nieprawidowa sekwencja\n", 1);

	/* lista kontaktw z serwera */
	format_add("userlist_put_ok", "%> List kontaktw zachowano na serwerze\n", 1);
	format_add("userlist_put_error", "%! Bd podczas wysyania listy kontaktw\n", 1);
	format_add("userlist_get_ok", "%> List kontaktw wczytano z serwera\n", 1);
	format_add("userlist_get_error", "%! Bd podczas pobierania listy kontaktw\n", 1);
	format_add("userlist_clear_ok", "%) Usunito list kontaktw z serwera\n", 1);
	format_add("userlist_clear_error", "%! Bd podczas usuwania listy kontaktw\n", 1);
	format_add("userlist_config_put_ok", "%> List kontaktw i konfiguracj zachowano na serwerze\n", 1);
	format_add("userlist_config_put_error", "%! Bd podczas wysyania listy kontaktw i konfiguracji\n", 1);
	format_add("userlist_config_get_ok", "%> List kontaktw i konfiguracj wczytano z serwera\n", 1);
	format_add("userlist_config_get_error", "%! Bd podczas pobierania listy kontaktw i konfiguracji\n", 1);
	format_add("userlist_config_clear_ok", "%) Usunito list kontaktw i konfiguracj z serwera\n", 1);
	format_add("userlist_config_clear_error", "%! Bd podczas usuwania listy kontaktw i konfiguracji\n", 1);

	/* szybka lista kontaktw pod F2 */
	format_add("quick_list", "%)%1\n", 1);
	format_add("quick_list,speech", "lista kontaktw: ", 1);
	format_add("quick_list_avail", " %Y%1%n", 1);
	format_add("quick_list_avail,speech", "%1 jest dostpny, ", 1);
	format_add("quick_list_busy", " %G%1%n", 1);
	format_add("quick_list_busy,speech", "%1 jest zajty, ", 1);
	format_add("quick_list_invisible", " %c%1%n", 1);

	/* window */
	format_add("window_add", "%) Utworzono nowe okno\n", 1);
	format_add("window_noexist", "%! Wybrane okno nie istnieje\n", 1);
	format_add("window_no_windows", "%! Nie mona zamkn ostatniego okna\n", 1);
	format_add("window_del", "%) Zamknito okno\n", 1);
	format_add("windows_max", "%! Wyczerpano limit iloci okien\n", 1);
	format_add("window_list_query", "%) %1: rozmowa z %T%2%n\n", 1);
	format_add("window_list_nothing", "%) %1: brak rozmowy\n", 1);
	format_add("window_list_floating", "%) %1: pywajce %4x%5 w %2,%3 %T%6%n\n", 1);
	format_add("window_id_query_started", "%) Rozmowa z %T%2%n rozpoczta w oknie %T%1%n\n", 1);
	format_add("window_kill_status", "%! Nie mona zamkn okna stanu\n", 1);

	/* bind */
	format_add("bind_seq_incorrect", "%! Sekwencja %T%1%n jest nieprawidowa\n", 1); 
	format_add("bind_seq_add", "%> Dodano sekwencj %T%1%n\n", 1);
	format_add("bind_seq_remove", "%) Usunito sekwencj %T%1%n\n", 1);	
	format_add("bind_seq_list", "%> %1: %T%2%n\n", 1);
	format_add("bind_seq_exist", "%! Sekwencja %T%1%n ma ju przypisan akcj\n", 1);
	format_add("bind_seq_list_empty", "%! Brak przypisanych akcji\n", 1);

	/* at */
	format_add("at_list", "%> %1, %2, %3 %K(%4)%n %5\n", 1);
	format_add("at_added", "%> Utworzono plan %T%1%n\n", 1);
	format_add("at_deleted", "%) Usunito plan %T%1%n\n", 1);
	format_add("at_deleted_all", "%) Usunito plany uytkownika\n", 1);
	format_add("at_exist", "%! Plan %T%1%n ju istnieje\n", 1);
	format_add("at_noexist", "%! Plan %T%1%n nie istnieje\n", 1);
	format_add("at_empty", "%! Brak planw\n", 1);
	format_add("at_timestamp", "%d-%m-%Y %H:%M", 1);
	format_add("at_back_to_past", "%! Gdyby mona byo cofn czas...\n", 1);

	/* timer */
	format_add("timer_list", "%> %1, %2s, %3 %K(%4)%n %T%5%n\n", 1);
	format_add("timer_added", "%> Utworzono timer %T%1%n\n", 1);
	format_add("timer_deleted", "%) Usunito timer %T%1%n\n", 1);
	format_add("timer_deleted_all", "%) Usunito timery uytkownika\n", 1);
	format_add("timer_exist", "%! Timer %T%1%n ju istnieje\n", 1);
	format_add("timer_noexist", "%! Timer %T%1%n nie istnieje\n", 1);
	format_add("timer_empty", "%! Brak timerw\n", 1);

	/* last */
	format_add("last_list_in", "%) %Y <<%n [%1] %2 %3\n", 1);
	format_add("last_list_out", "%) %G >>%n [%1] %2 %3\n", 1);
	format_add("last_list_empty", "%! Nie zalogowano wiadomoci\n", 1);
	format_add("last_list_empty_nick", "%! Nie zalogowano wiadomoci dla %T%1%n\n", 1);
	format_add("last_list_timestamp", "%d-%m-%Y %H:%M", 1);
	format_add("last_list_timestamp_today", "%H:%M", 1);
	format_add("last_clear", "%) Usunito wiadomoci\n", 1);
	format_add("last_clear_one", "%) Usunito wiadomo\n", 1);

	/* queue */
	format_add("queue_list_timestamp", "%d-%m-%Y %H:%M", 1);
	format_add("queue_list_message", "%) %G >>%n [%1] %2 %3\n", 1);
	format_add("queue_clear","%) Usunito wiadomoci z kolejki\n", 1);
	format_add("queue_clear_uin","%) Usunito wiadomoci z kolejki dla %T%1%n\n", 1);
	format_add("queue_wrong_use", "%! Komenda dziaa tylko w czasie braku poczenia z serwerem\n", 1);
	format_add("queue_empty", "%! Kolejka wiadomoci jest pusta\n", 1);
	format_add("queue_empty_uin", "%! Brak wiadomoci w kolejce dla %T%1%n\n", 1);
	format_add("queue_flush", "%> Wysano zalege wiadomoci z kolejki\n", 1);

	/* conference */
	format_add("conferences_list_empty", "%! Brak konferencji\n", 1);
	format_add("conferences_list", "%> %T%1%n: %2\n", 1);
	format_add("conferences_list_ignored", "%> %T%1%n: %2 (%yingorowana%n)\n", 1);
	format_add("conferences_add", "%> Utworzono konferencj %T%1%n\n", 1);
	format_add("conferences_not_added", "%! Nie utworzono konferencji %T%1%n\n", 1);
	format_add("conferences_del", "%) Usunito konferencj %T%1%n\n", 1);
	format_add("conferences_del_all", "%) Usunito wszystkie konferencje\n", 1);
	format_add("conferences_exist", "%! Konferencja %T%1%n ju istnieje\n", 1);
	format_add("conferences_noexist", "%! Konferencja %T%1%n nie istnieje\n", 1);
	format_add("conferences_name_error", "%! Nazwa konferencji powinna zaczyna si od %T#%n\n", 1);
	format_add("conferences_rename", "%> Nazwa konferencji zmieniona: %T%1%n --> %T%2%n\n", 1);
	format_add("conferences_ignore", "%> Konferencja %T%1%n bdzie ignorowana\n", 1);
	format_add("conferences_unignore", "%> Konferencja %T%1%n nie bdzie ignorowana\n", 1);
	format_add("conferences_joined", "%> Doczono %1 do konferencji %T%2%n\n", 1);
	format_add("conferences_already_joined", "%> %1 uczestniczy ju w konferencji %T%2%n\n", 1);
	
	/* wsplne dla usug http */
	format_add("http_failed_resolving", "Nie znaleziono serwera", 1);
	format_add("http_failed_connecting", "Nie mona poczy si z serwerem", 1);
	format_add("http_failed_reading", "Serwer zerwa poczenie", 1);
	format_add("http_failed_writing", "Serwer zerwa poczenie", 1);
	format_add("http_failed_memory", "Brak pamici", 1);
	

#ifdef HAVE_OPENSSL
	/* szyfrowanie */
	format_add("key_generating", "%> Czekaj, generuj klucze...\n", 1);
	format_add("key_generating_success", "%> Wygenerowano i zapisano klucze\n", 1);
	format_add("key_generating_error", "%! Wystpi bd podczas generowania kluczy: %1\n", 1);
	format_add("key_private_exist", "%! Posiadasz ju swoj par kluczy\n", 1);
	format_add("key_public_deleted", "%) Klucz publiczny %1 zosta usunity\n", 1);
	format_add("key_public_not_found", "%! Nie znaleziono klucza publicznego %1\n", 1);
	format_add("key_public_noexist", "%! Brak kluczy publicznych\n", 1);
	format_add("key_public_received", "%> Otrzymano klucz publiczny od %1\n", 1);
	format_add("key_public_write_failed", "%! Bd podczas zapisu klucza publicznego: %1\n", 1);
	format_add("key_send_success", "%> Wysano klucz publiczny do %1\n", 1);
	format_add("key_send_error", "%! Bd podczas wysyania klucza publicznego\n", 1);
	format_add("key_list", "%> %1 (%3)\n%) %2\n", 1);
	format_add("key_list_timestamp", "%Y-%m-%d %H:%M", 1);
#endif

#ifdef WITH_PYTHON
	/* python */
	format_add("python_list", "%> %1\n", 1);
	format_add("python_list_empty", "%! Brak zaadowanych skryptw\n", 1);
	format_add("python_removed", "%) Skrypt zosta usunity\n", 1);
	format_add("python_need_name", "%! Nie podano nazwy skryptu\n", 1);
	format_add("python_not_found", "%! Nie znaleziono skryptu %T%1%n\n", 1);
	format_add("python_wrong_location", "%! Skrypt naley umieci w katalogu %T%1%n\n", 1);
#endif
}
