/*-
 *
 * Copyright (c) 1999 Seigo Tanimura
 * Copyright (c) 2003, 2004 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * (c) 2003 Mathew Kanner
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx-midi-mk.c,v 1.11 2005/10/13 18:20:01 chibis Exp $
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif

#include <dev/sound/chip.h>
#include <dev/sound/pcm/sound.h>

#include "emu10kx.h"
#include "emu10kx-midi-mk.h"


typedef struct emu_midi_softc *sc_p;
uint32_t emu_midi_card_intr(void *p, uint32_t arg);
static devclass_t emu_midi_devclass;

static unsigned char
emu_mread(void *arg __unused, struct emu_midi_softc *sc, int reg)
{
	unsigned int d;

	d = 0;
	if (sc->card->is_emu10k2) {
		d = emu_rdptr(sc->card, 0, sc->port + reg);
	};
	if (sc->card->is_emu10k1) {
		d = emu_rd(sc->card, 0x18 + reg, 1);
	};
	return d;
}

static void
emu_mwrite(void *arg __unused, struct emu_midi_softc *sc, int reg, unsigned char b)
{

	if (sc->card->is_emu10k2) {
		emu_wrptr(sc->card, 0, sc->port + reg, b);
	};
	if (sc->card->is_emu10k1) {
		emu_wr(sc->card, 0x18 + reg, b, 1);
	};
}

static int
emu_muninit(void *arg __unused, struct emu_midi_softc *sc)
{

	mtx_lock(&sc->mtx);
	sc->mpu_intr = 0;
	mtx_unlock(&sc->mtx);

	return 0;
}

static kobj_method_t emu_mpu_methods[] = {
	KOBJMETHOD(mpufoi_read, emu_mread),
	KOBJMETHOD(mpufoi_write, emu_mwrite),
	KOBJMETHOD(mpufoi_uninit, emu_muninit),
	{0, 0}
};
DEFINE_CLASS(emu_mpu, emu_mpu_methods, 0);

uint32_t
emu_midi_card_intr(void *p, uint32_t arg)
{
	struct emu_midi_softc *sc = (struct emu_midi_softc *)p;
	if (sc->mpu_intr)
		(sc->mpu_intr) (sc->mpu);
	if (sc->mpu_intr == NULL) {
		/* failover, we should read MIDI event to unlock card after
		 * interrupt  */
		device_printf(sc->dev, "midi interrupt %08x without interrupt handler, force mread!\n", arg);
		(void)emu_mread((void *)(NULL), sc, 0);
	};
	return arg;
}

static void
emu_midi_intr(void *p)
{
	(void)emu_midi_card_intr(p, 0);
}

static int
emu_midi_probe(device_t dev)
{
	sc_p scp;
	uintptr_t func, r;

	/* The parent device has already been probed. */
	r = BUS_READ_IVAR(device_get_parent(dev), dev, 0, &func);
	if (func != SCF_MIDI)
		return (ENXIO);

	scp = device_get_softc(dev);
	bzero(scp, sizeof(*scp));
	device_set_desc(dev, "EMU10Kx MIDI Interface (midi2)");
	return (0);
}

static int
emu_midi_attach(device_t dev)
{
	sc_p scp;
	struct sndcard_func *func;
	struct emu_midiinfo *midiinfo;
	uint32_t inte_val, ipr_val;

	scp = device_get_softc(dev);
	func = device_get_ivars(dev);

	/* Fill the softc. */
	scp->dev = dev;
	midiinfo = (struct emu_midiinfo *)func->varinfo;
	scp->port = midiinfo->port;
	scp->card = midiinfo->card;

	mtx_init(&scp->mtx, "emu_mid", NULL, MTX_DEF);

/*	printf("midi-mk: sc_emu=%p, sc_midi=%p\n",scp->card,scp); */
	if (scp->card->is_emu10k2) {
		if (scp->port == A_MUDATA1) {
			/* EXTERNAL MIDI (AudigyDrive) */
			/* MIDI hw config */
			inte_val = 0;
/*			inte_val = A_INTE_MIDITXENABLE1;*/
			inte_val |= A_INTE_MIDIRXENABLE1;
			ipr_val = A_IPR_MIDITRANSBUFEMPTY1;
			ipr_val |= A_IPR_MIDIRECVBUFEMPTY1;
		} else {
			/* MIDI hw config port 2 */
			inte_val = 0;
/*			inte_val = A_INTE_MIDITXENABLE2;*/
			inte_val |= A_INTE_MIDIRXENABLE2;
			ipr_val = A_IPR_MIDITRANSBUFEMPTY2;
			ipr_val |= A_IPR_MIDIRECVBUFEMPTY2;
		}
	};
	if (sc->card->is_emu10k1) {
		/* SB Live! - only one MIDI device here */
		inte_val = 0;
/*		    inte_val = INTE_MIDITXENABLE;*/
		inte_val |= INTE_MIDIRXENABLE;
		ipr_val = IPR_MIDITRANSBUFEMPTY;
		ipr_val |= IPR_MIDIRECVBUFEMPTY;
	};
	
	if (inte_val == 0)
		return ENXIO;

	scp->ihandle = emu_intr_register(scp->card, inte_val, ipr_val, &emu_midi_card_intr, scp);
	/* Init the interface. */
	scp->mpu = mpu401_init(&emu_mpu_class, scp, emu_midi_intr, &scp->mpu_intr);
	if (scp->mpu == NULL) {
		emu_intr_unregister(scp->card, scp->ihandle);
		mtx_destroy(&scp->mtx);
		return (ENOMEM);
	}
	return (0);
}


static int
emu_midi_detach(device_t dev)
{
	sc_p scp;

	scp = device_get_softc(dev);
	mpu401_uninit(scp->mpu);
	emu_intr_unregister(scp->card, scp->ihandle);
	mtx_destroy(&scp->mtx);
	return (0);
}

static device_method_t emu_midi_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe, emu_midi_probe),
	DEVMETHOD(device_attach, emu_midi_attach),
	DEVMETHOD(device_detach, emu_midi_detach),

	{0, 0},
};

static driver_t emu_midi_driver = {
	"midi",
	emu_midi_methods,
	sizeof(struct emu_midi_softc),
};
#ifdef MIDI2_KERNEL
MODULE_DEPEND(snd_midi, sound, SOUND_MINVER, SOUND_PREFVER, SOUND_MAXVER);
#else
MODULE_DEPEND(emu_midi, midi, 1, 1, 1);
#endif
DRIVER_MODULE(emu_midi, emu10kx, emu_midi_driver, emu_midi_devclass, 0, 0);
