/*-
 *
 * Copyright (c) 1999 Seigo Tanimura
 * Copyright (c) 2003,2004 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx-midi.c,v 1.26 2005/10/13 18:20:01 chibis Exp $
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif

#include <dev/sound/midi/midi.h>
#include <dev/sound/chip.h>

#include "emu10kx.h"
#include "emu10kx-midi.h"

static devclass_t midi_devclass;

extern synthdev_info midisynth_op_desc;

/* These are the synthesizer and the midi interface information. */
static struct synth_info emu_synthinfo = {
	"MPU401 MIDI",
	0,
	SYNTH_TYPE_MIDI,
	0,
	0,
	128,
	128,
	128,
	SYNTH_CAP_INPUT,
};

static struct midi_info emu_midiinfo = {
	"EMU10Kx MIDI",
	0,
	0,
	0,
};
/*
 */

static int emu_midi_probe(device_t dev);
static int emu_midi_attach(device_t dev);

static d_ioctl_t emu_midi_ioctl;
static midi_callback_t emu_midi_callback;


typedef struct emu_midi_softc *sc_p;

/* These functions are local. */
static void emu_midi_startplay(sc_p scp);
static void emu_midi_xmit(sc_p scp);
static int emu_midi_reset(sc_p scp);

static int emu_midi_status(sc_p scp);
static int emu_midi_readdata(sc_p scp, uint8_t * value);
static int emu_midi_writedata(sc_p scp, uint32_t value);
static int emu_midi_writecmd(sc_p scp, uint32_t value);

uint32_t emu_midi_intr(void *arg, uint32_t intr);

/*
 * This is the device descriptor for the midi device.
 */
static mididev_info emu_midi_op_desc = {
	"EMU10kx midi",

	SNDCARD_MPU401,

	NULL,
	NULL,
	emu_midi_ioctl,

	emu_midi_callback,

	MIDI_BUFFSIZE,		/* Queue Length */

	0,			/* XXX This is not an *audio* device! */
};
/*
HW I/O
*/

static int 
emu_midi_hw_read(sc_p sc, int idx)
{
	int midiin;
	
	midiin = 0;
	if (sc->card->is_emu10k2)
		midiin = (int)emu_rdptr(sc->card, 0, sc->port + idx);
	if (sc->card->is_emu10k1)
		midiin = (int)emu_rd(sc->card, sc->port + idx, 1);
	return midiin;
}

static void 
emu_midi_hw_write(sc_p sc, int idx, int data)
{
	if (sc->card->is_emu10k2)
		emu_wrptr(sc->card, 0, sc->port + idx, data);
	if (sc->card->is_emu10k1)
		emu_wr(sc->card, sc->port + idx, data, 1);
}
#define TIMEOUT 16384
static
void 
emu_wcwait(struct emu_sc_info *sc, uint32_t wait)
{
	uint32_t uCount;
	uint32_t newtime = 0, curtime;

	curtime = emu_rd(sc, WC_SAMPLECOUNTER, 4);
	while (wait--) {
		uCount = 0;
		while (uCount++ < TIMEOUT) {
			newtime = emu_rd(sc, WC_SAMPLECOUNTER, 4);
			if (newtime != curtime)
				break;
		}

		if (uCount >= TIMEOUT)
			break;

		curtime = newtime;
	}
}



/*
 * Here are the main functions to interact to the user process.
 */

static int
emu_midi_probe(device_t dev)
{
	sc_p scp;
	uintptr_t func, r;

	/* The parent device has already been probed. */
	r = BUS_READ_IVAR(device_get_parent(dev), dev, 0, &func);
	if (func != SCF_MIDI)
		return (ENXIO);

	scp = device_get_softc(dev);
	bzero(scp, sizeof(*scp));

	device_set_desc(dev, "EMU10Kx MIDI Interface (NEWMIDI)");
	return (0);
}

static int
emu_midi_attach(device_t dev)
{
	sc_p scp;
	mididev_info *devinfo;
	struct sndcard_func *func;
	struct emu_midiinfo *midiinfo;
	uint32_t inte_val, ipr_val;

	scp = device_get_softc(dev);
	func = device_get_ivars(dev);


	/* Fill the softc. */
	scp->dev = dev;
	midiinfo = (struct emu_midiinfo *)func->varinfo;
	scp->port = midiinfo->port;
	scp->card = midiinfo->card;

	mtx_init(&scp->mtx, "emu_mid", NULL, MTX_DEF);
	scp->devinfo = devinfo = create_mididev_info_unit(MDT_MIDI, &emu_midi_op_desc, &midisynth_op_desc);

	midiinit(devinfo, dev);

	inte_val = 0;
	if (scp->card->is_emu10k2) {
		if (scp->port == A_MUDATA1) {
			inte_val = 0;
/*			inte_val |= A_INTE_MIDITXENABLE1 */
			inte_val |= A_INTE_MIDIRXENABLE1;
			ipr_val = A_IPR_MIDITRANSBUFEMPTY1 | A_IPR_MIDIRECVBUFEMPTY1;
		} else {
			inte_val = 0;
/*			inte_val |= A_INTE_MIDITXENABLE2 */
			inte_val |= A_INTE_MIDIRXENABLE2;
			ipr_val = A_IPR_MIDITRANSBUFEMPTY2 | A_IPR_MIDIRECVBUFEMPTY2;
		}
	};
	
	if (scp->card->is_emu10k1) {
		inte_val = 0;
/*		    inte_val |= INTE_MIDITXENABLE; */
		inte_val |= INTE_MIDIRXENABLE;
		ipr_val = IPR_MIDITRANSBUFEMPTY | IPR_MIDIRECVBUFEMPTY;
	};
	
	if (inte_val == 0)
		return ENXIO;
	scp->ihandle = emu_intr_register(scp->card, inte_val, ipr_val, &emu_midi_intr, scp);

	/* Reset the interface. */
	emu_midi_reset(scp);

	return (0);
}
#if __FreeBSD_version < 500000
static int
emu_midi_ioctl(dev_t i_dev, u_long cmd, caddr_t arg, int mode __unused, struct thread *td __unused)
#else
static int
emu_midi_ioctl(struct cdev *i_dev, u_long cmd, caddr_t arg, int mode __unused, struct thread *td __unused)
#endif
{
	sc_p scp;
	mididev_info *devinfo;
	int unit;
	struct synth_info *synthinfo;
	struct midi_info *midiinfo;

	unit = MIDIUNIT(i_dev);

	MIDI_DEBUG(printf("emu_midi_ioctl: unit %d, cmd %s.\n", unit, midi_cmdname(cmd, cmdtab_midiioctl)));

	devinfo = get_mididev_info(i_dev, &unit);
	if (devinfo == NULL) {
		MIDI_DEBUG(printf("emu_midi_ioctl: unit %d is not configured.\n", unit));
		return (ENXIO);
	}
	scp = devinfo->softc;

	switch (cmd) {
	case SNDCTL_SYNTH_INFO:
		synthinfo = (struct synth_info *)arg;
		if (synthinfo->device != unit)
			return (ENXIO);
		bcopy(&emu_synthinfo, synthinfo, sizeof(emu_synthinfo));
		synthinfo->device = unit;
		return (0);
		break;
	case SNDCTL_MIDI_INFO:
		midiinfo = (struct midi_info *)arg;
		if (midiinfo->device != unit)
			return (ENXIO);
		bcopy(&emu_midiinfo, midiinfo, sizeof(emu_midiinfo));
		midiinfo->device = unit;
		return (0);
		break;
	default:
		return (ENOSYS);
	}
	/* NOTREACHED */
	return (EINVAL);
}
uint32_t
emu_midi_intr(void *arg, uint32_t intr)
{
	sc_p scp;
	u_char c;
	mididev_info *devinfo;
	int leni;

	scp = (sc_p) arg;
	devinfo = scp->devinfo;

	mtx_lock(&devinfo->flagqueue_mtx);
	mtx_lock(&scp->mtx);

	/* Read the received data. */
	while ((emu_midi_status(scp) & MUSTAT_IRDYN) == 0) {
		/* Receive the data. */
		emu_midi_readdata(scp, &c);
		mtx_unlock(&scp->mtx);

		/* Queue into the passthrough buffer and start transmitting if
		 * we can. */
		if ((devinfo->flags & MIDI_F_PASSTHRU) != 0 && ((devinfo->flags & MIDI_F_BUSY) == 0 || (devinfo->fflags & FWRITE) == 0)) {
			midibuf_input_intr(&devinfo->midi_dbuf_passthru, &c, sizeof(c), &leni);
			devinfo->callback(devinfo, MIDI_CB_START | MIDI_CB_WR);
		}
		/* Queue if we are reading. Discard an active sensing. */
		if ((devinfo->flags & MIDI_F_READING) != 0 && c != 0xfe) {
			midibuf_input_intr(&devinfo->midi_dbuf_in, &c, sizeof(c), &leni);
		}
		mtx_lock(&scp->mtx);
	}
	mtx_unlock(&scp->mtx);

	/* Transmit out data. */
	if ((devinfo->flags & MIDI_F_WRITING) != 0 && (emu_midi_status(scp) & MUSTAT_ORDYN) == 0) {
		emu_midi_xmit(scp);
	};

	mtx_unlock(&devinfo->flagqueue_mtx);

	/* Invoke the upper layer. */
	midi_intr(devinfo);
	return intr;
}

static int
emu_midi_callback(void *di, int reason)
{
	sc_p scp;
	mididev_info *d;

	d = (mididev_info *) di;

/*	mtx_assert(&d->flagqueue_mtx, MA_OWNED); */

	if (d == NULL) {
		MIDI_DEBUG(printf("emu_midi_callback: device not configured.\n"));
		return (ENXIO);
	}
	scp = d->softc;

	switch (reason & MIDI_CB_REASON_MASK) {
	case MIDI_CB_START:
		if ((reason & MIDI_CB_RD) != 0 && (d->flags & MIDI_F_READING) == 0)
			/* Begin recording. */
			d->flags |= MIDI_F_READING;
		if ((reason & MIDI_CB_WR) != 0 && (d->flags & MIDI_F_WRITING) == 0)
			/* Start playing. */
			emu_midi_startplay(scp);
		break;
	case MIDI_CB_STOP:
	case MIDI_CB_ABORT:
		if ((reason & MIDI_CB_RD) != 0 && (d->flags & MIDI_F_READING) != 0)
			/* Stop recording. */
			d->flags &= ~MIDI_F_READING;
		if ((reason & MIDI_CB_WR) != 0 && (d->flags & MIDI_F_WRITING) != 0)
			/* Stop Playing. */
			d->flags &= ~MIDI_F_WRITING;
		break;
	}

	return (0);
}
/*
 * The functions below here are the libraries for the above ones.
 */

/*
 * Starts to play the data in the output queue.
 */
static void
emu_midi_startplay(sc_p scp)
{
	mididev_info *devinfo;

	devinfo = scp->devinfo;

/*	mtx_assert(&devinfo->flagqueue_mtx, MA_OWNED); */

	/* Can we play now? */
	if (devinfo->midi_dbuf_out.rl == 0)
		return;

	devinfo->flags |= MIDI_F_WRITING;
	emu_midi_xmit(scp);
}

static void
emu_midi_xmit(sc_p scp)
{
	register mididev_info *devinfo;
	register midi_dbuf *dbuf;
	u_char c;
	int leno;

	devinfo = scp->devinfo;

/*	mtx_assert(&devinfo->flagqueue_mtx, MA_OWNED);*/

	/* See which source to use. */
	if ((devinfo->flags & MIDI_F_PASSTHRU) == 0 || ((devinfo->flags & MIDI_F_BUSY) != 0 && (devinfo->fflags & FWRITE) != 0))
		dbuf = &devinfo->midi_dbuf_out;
	else
		dbuf = &devinfo->midi_dbuf_passthru;

	/* Transmit the data in the queue. */
	while ((devinfo->flags & MIDI_F_WRITING) != 0) {
		/* Do we have the data to transmit? */
		if (dbuf->rl == 0) {
			/* Stop playing. */
			devinfo->flags &= ~MIDI_F_WRITING;
			break;
		} else {
			mtx_lock(&scp->mtx);
			if ((emu_midi_status(scp) & MUSTAT_ORDYN) != 0) {
				mtx_unlock(&scp->mtx);
				break;
			}
			/* Send the data. */
			midibuf_output_intr(dbuf, &c, sizeof(c), &leno);
			emu_midi_writedata(scp, c);
			/* We are playing now. */
			devinfo->flags |= MIDI_F_WRITING;
			mtx_unlock(&scp->mtx);
		}
	}

}
/* Reset midi. */
static int
emu_midi_reset(sc_p scp)
{
	uint8_t resp;

	mtx_lock(&scp->mtx);

	/* Reset the midi. */
	resp = 0;
	emu_midi_writecmd(scp, MUCMD_RESET);
	emu_wcwait(scp->card, 8);
	emu_midi_writecmd(scp, MUCMD_RESET);
	emu_wcwait(scp->card, 8);
	emu_midi_writecmd(scp, MUCMD_ENTERUARTMODE);
	emu_wcwait(scp->card, 8);
	(void)emu_midi_readdata(scp, &resp);
	mtx_unlock(&scp->mtx);

	return (0);
}
/* Reads the status. */
static int
emu_midi_status(sc_p scp)
{
	return (int)emu_midi_hw_read(scp, 1);
}
/* Reads a byte of data. */
static int
emu_midi_readdata(sc_p scp, uint8_t * value)
{
	*value = emu_midi_hw_read(scp, 0);
	return (0);
}
/* Writes a byte of data. */
static int
emu_midi_writedata(sc_p scp, uint32_t value)
{
	emu_midi_hw_write(scp, 0, value);
	return (0);
}
/* Writes a command. */
static int
emu_midi_writecmd(sc_p scp, uint32_t value)
{
	emu_midi_hw_write(scp, 1, value);
	return (0);
}

static device_method_t emu_midi_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe, emu_midi_probe),
	DEVMETHOD(device_attach, emu_midi_attach),

	{0, 0},
};

static driver_t emu_midi_driver = {
	"midi",
	emu_midi_methods,
	sizeof(struct emu_midi_softc),
};
DRIVER_MODULE(emu_midi, emu10kx, emu_midi_driver, midi_devclass, 0, 0);
