/*
 * Copyright (c) 1999 Cameron Grant <gandalf@vilnya.demon.co.uk>
 * Copyright (c) 2003 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHERIN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx-pcm.c,v 1.57 2005/10/16 17:37:43 chibis Exp $
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>
#include <sys/systm.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif

#include <dev/sound/chip.h>

#include "emu10kx.h"
#include "emu10kx-pcm.h"
#include "emu10kx-mixer.h"
#include "emu10kx-opcodes.h"

#include "mixer_if.h"

/*
last changes from FreeBSD emu10k1.c version:
$FreeBSD: src/sys/dev/sound/pci/emu10k1.c,v 1.55 2005/03/01 08:58:05 imp Exp $
*/

#undef EMU_AC97DEBUG

/* -------------------------------------------------------------------- */
/* Recording formats supported: */
static uint32_t emu_rfmt_ac97[] = {
	AFMT_S16_LE,
	AFMT_STEREO | AFMT_S16_LE,
	0
};

static uint32_t emu_rfmt_mic[] = {
	AFMT_U8,
	0
};

static uint32_t emu_rfmt_efx[] = {
	AFMT_STEREO | AFMT_S16_LE,
	0
};
/* Recording capabilites */
static struct pcmchan_caps emu_reccaps[3] = {
	{8000, 48000, emu_rfmt_ac97, 0},
	{8000, 8000, emu_rfmt_mic, 0},
	{48000, 48000, emu_rfmt_efx, 0},
};
/* Playback formats */
static uint32_t emu_pfmt[] = {
	AFMT_U8,
	AFMT_STEREO | AFMT_U8,
	AFMT_S16_LE,
	AFMT_STEREO | AFMT_S16_LE,
	0
};

static uint32_t emu_pfmt_pt[] = {
	AFMT_STEREO | AFMT_S16_LE,
	AFMT_AC3,
	0
};
/* Playback capabilites */
static struct pcmchan_caps emu_playcaps = {4000, 48000, emu_pfmt, 0};
static struct pcmchan_caps emu_playcaps_pt = {48000, 48000, emu_pfmt_pt, 0};

static int emu10k1_adcspeed[8] = {48000, 44100, 32000, 24000, 22050, 16000, 11025, 8000};
/* audigy supports 12kHz. */
static int emu10k2_adcspeed[9] = {48000, 44100, 32000, 24000, 22050, 16000, 12000, 11025, 8000};



uint32_t emu_pcm_intr(void *pcm, uint32_t stat);

/* AC97 emulation for Audigy */

#define AC97LEFT(x) ((x & 0x7F00)>>8)
#define AC97RIGHT(x) (x & 0x007F)
#define AC97MUTE(x)	((x & 0x8000)>>15)
#define BIT4_TO100(x) (100-(x)*100/(0x0f))
#define BIT6_TO100(x) (100-(x)*100/(0x3f))
#define V100_TOBIT6(x) (0x3f*(100-x)/100)
#define V100_TOBIT4(x) (0x0f*(100-x)/100)
#define AC97ENCODE(x_muted,x_left,x_right) (((x_muted&1)<<15) | ((x_left&0x3f)<<8) | (x_right&0x3f))

static int
emu_ac97_read_emulation(struct emu_pcm_info *sc, int regno)
{
	int use_ac97;
	int emulated;
	int tmp;
	int left, right, mute;

	use_ac97 = 1;
	emulated = 0;

	switch (regno) {
	case AC97_MIX_MASTER:
		emulated = sc->ac97_state[AC97_MIX_MASTER];
		use_ac97 = 0;
		break;
	case AC97_MIX_PCM:
		emulated = sc->ac97_state[AC97_MIX_PCM];
		use_ac97 = 0;
		break;
	case AC97_REG_RECSEL:
		emulated = 0x0505;
		use_ac97 = 0;
		break;
	case AC97_MIX_RGAIN:
		emulated = sc->ac97_state[AC97_MIX_RGAIN];
		use_ac97 = 0;
		break;
	};

	emu_wr(sc->card, AC97ADDRESS, regno, 1);
	tmp = emu_rd(sc->card, AC97DATA, 2);

	if (use_ac97)
		emulated = tmp;

	return emulated;
}

static void
emu_ac97_write_emulation(struct emu_pcm_info *sc, int regno, uint32_t data)
{
	int write_ac97;
	int left, right;
	uint32_t emu_left, emu_right;
	int is_mute;

	write_ac97 = 1;

	/* decode AC97 command */
	left = AC97LEFT(data);
	emu_left = BIT6_TO100(left);	/* We show as  6-bit AC97
					 * hardware mixer */
	right = AC97RIGHT(data);
	emu_right = BIT6_TO100(right);
	is_mute = AC97MUTE(data);
	if (is_mute)
		emu_left = emu_right = 0;

	switch (regno) {
		/* TODO: reset emulator on AC97_RESET */
	case AC97_MIX_MASTER:
		/* keep MAIN OUT volume on AC97 muted
		 * 
		 * use emumix_set_fxvol to set MASTER MIX Playback volume */
		sc->ac97_state[AC97_MIX_MASTER] = data & (0x8000 | 0x3f3f);
		sc->card->mixer->mix.volume[MIXER_P_VOLUME_LEFT] = emu_left;
		sc->card->mixer->mix.volume[MIXER_P_VOLUME_RIGHT] = emu_right;
		emumix_set_fxvol(sc->card, sc->card->mixer->mix.gpr[GPR_P_VOLUME_LEFT], emu_left);
		emumix_set_fxvol(sc->card, sc->card->mixer->mix.gpr[GPR_P_VOLUME_RIGHT], emu_right);
#ifdef EMU_AC97DEBUG
		device_printf(sc->dev, "set MIX_MASTER: vol  %d:%d ac97 m=%d l=%x r=%x d=%04x e=%04x\n",
		    sc->card->mixer->mix.volume[MIXER_P_VOLUME_LEFT],
		    sc->card->mixer->mix.volume[MIXER_P_VOLUME_RIGHT],
		    is_mute,
		    left, right, data, AC97ENCODE(is_mute, left, right));
#endif
		if((sc->card->is_emu10k2) || (sc->card->is_ca0151))
			data = 0x8000;	/* Mute AC97 main out */
		break;
	case AC97_MIX_PCM:	/* PCM OUT VOL */
		sc->ac97_state[AC97_MIX_PCM] = data & (0x8000 | 0x3f3f);
		/* PCM volume on AC97 should stay muted, use emumix_set_fxvol
		 * to set PCM volume from FX0 If PCM volume is muted, then
		 * front PCM output is done by better 6-ch Philips chip */
		sc->card->mixer->fxbuses[0].volume[MIXER_P_VOLUME_LEFT] = emu_left;
		sc->card->mixer->fxbuses[0].volume[MIXER_P_VOLUME_RIGHT] = emu_right;
		emumix_set_fxvol(sc->card, sc->card->mixer->fxbuses[0].gpr[GPR_P_VOLUME_LEFT], emu_left);
		emumix_set_fxvol(sc->card, sc->card->mixer->fxbuses[0].gpr[GPR_P_VOLUME_RIGHT], emu_right);
#ifdef EMU_AC97DEBUG
		device_printf(sc->dev, "set MIX_PCM: vol  %d:%d ac97 m=%d l=%x r=%x d=%04x e=%04x\n",
		    sc->card->mixer->fxbuses[0].volume[MIXER_P_VOLUME_LEFT],
		    sc->card->mixer->fxbuses[0].volume[MIXER_P_VOLUME_RIGHT],
		    is_mute,
		    left, right, data, AC97ENCODE(is_mute, left, right));
#endif
		if((sc->card->is_emu10k2) || (sc->card->is_ca0151))
			data = 0x8000;	/* Mute AC97 main out */
		break;
	case AC97_REG_RECSEL:
		/* PCM recording source is set to "stereo mix" (labeled "vol"
		 * in mixer) XXX !I can't remember why! */
		data = 0x0505;
		break;
	case AC97_MIX_RGAIN:	/* RECORD GAIN */
		/* Record gain on AC97 should stay zero to get AC97 sound on
		 * AC97_[RL] connectors on EMU10K2 chip AC97 on Audigy is not
		 * directly connected to any output, only to EMU10K2 chip Use
		 * this control to set AC97 mix volume inside EMU10K2 chip */
		sc->ac97_state[AC97_MIX_RGAIN] = data & (0x8000 | 0x0f0f);
		emu_left = 100 - BIT4_TO100(left);
		emu_right = 100 - BIT4_TO100(right);
		if (is_mute)
			emu_left = emu_right = 0;
		sc->card->mixer->mix.volume[MIXER_R_VOLUME_LEFT] = emu_left;
		sc->card->mixer->mix.volume[MIXER_R_VOLUME_RIGHT] = emu_right;
		/* rec control is recrding VOLUME, not attenuation and 0  IS
		 * 0%  and 15 IS 100% */
		emumix_set_fxvol(sc->card, sc->card->mixer->mix.gpr[GPR_R_VOLUME_LEFT], emu_left);
		emumix_set_fxvol(sc->card, sc->card->mixer->mix.gpr[GPR_R_VOLUME_RIGHT], emu_right);
#ifdef EMU_AC97DEBUG
		device_printf(sc->dev, "set MIX_REC: vol  %d:%d ac97 m=%d l=%x r=%x d=%04x e=%04x\n",
		    sc->card->mixer->mix.volume[MIXER_R_VOLUME_LEFT],
		    sc->card->mixer->mix.volume[MIXER_R_VOLUME_RIGHT],
		    is_mute,
		    left, right, data, sc->ac97_state[AC97_MIX_RGAIN]);
#endif
		data = 0x0000;
		break;
	};
	if (write_ac97) {
		emu_wr(sc->card, AC97ADDRESS, regno, 1);
		emu_wr(sc->card, AC97DATA, data, 2);
	}
}
/* -------------------------------------------------------------------- */
/* ac97 codec */
/* no locking needed */

static int
emu_rdcd(kobj_t obj __unused, void *devinfo, int regno)
{
	int rd;
	struct emu_pcm_info *sc = (struct emu_pcm_info *)devinfo;

	KASSERT(sc->card != NULL, ("emu_rdcd: no soundcard"));
	if ((sc->card->is_emu10k2) || (sc->card->is_ca0151) || (sc->card->is_ca0108)) {
		return emu_ac97_read_emulation(sc, regno);
	};
	emu_wr(sc->card, AC97ADDRESS, regno, 1);
	rd = emu_rd(sc->card, AC97DATA, 2);
	return rd;
}

static int
emu_wrcd(kobj_t obj __unused, void *devinfo, int regno, uint32_t data)
{
	struct emu_pcm_info *sc_pcm = (struct emu_pcm_info *)devinfo;
	struct emu_sc_info *sc;

	KASSERT(sc_pcm->card != NULL, ("emu_rdcd: no soundcard"));
	sc = sc_pcm->card;
	if ((sc->is_emu10k2) || (sc->is_ca0151) || (sc->is_ca0108)) {
		emu_ac97_write_emulation(sc_pcm, regno, data);
		return 0;
	};
	emu_wr(sc, AC97ADDRESS, regno, 1);
	emu_wr(sc, AC97DATA, data, 2);
	return 0;
}

static kobj_method_t emu_ac97_methods[] = {
	KOBJMETHOD(ac97_read, emu_rdcd),
	KOBJMETHOD(ac97_write, emu_wrcd),
	{0, 0}
};
AC97_DECLARE(emu_ac97);
/* --above done */


/* -------------------------------------------------------------------- */
/* stuff */
static int
emu_settimer(struct emu_pcm_info *sc)
{
	struct emu_pcm_pchinfo *pch;
	int tmp, delay;
	unsigned int i;

	delay = 1024;
	for (i = 0; i < sc->card->num_fxbuses; i++) {
		pch = &sc->pch[i];
		if (pch->buffer) {
			tmp = pch->blksz / sndbuf_getbps(pch->buffer);
			if (tmp < delay)
				delay = tmp;
		}
	}
	RANGE(delay, 16, 1024);
	sc->card->timerinterval = delay;
	emu_wr(sc->card, TIMER, sc->card->timerinterval & 0x03ff, 2);

	return sc->card->timerinterval;
}

static void
emu_enastop(struct emu_pcm_info *sc, char channel, int enable)
{
	int reg;

	reg = (channel & 0x20) ? SOLEH : SOLEL;
	channel &= 0x1f;
	reg |= 1 << 24;
	reg |= channel << 16;
	emu_wrptr(sc->card, 0, reg, enable);
}


static int
emu_k1_recval(int speed)
{
	int val;

	val = 0;
	while ((val < 7) && (speed < emu10k1_adcspeed[val]))
		val++;
	return val;
}

static int
emu_k2_recval(int speed)
{
	int val;

	val = 0;
	while ((val < 8) && (speed < emu10k2_adcspeed[val]))
		val++;
	return val;
}

static uint32_t
emu_rate_to_pitch(uint32_t rate)
{
	static uint32_t logMagTable[128] = {
		0x00000, 0x02dfc, 0x05b9e, 0x088e6, 0x0b5d6, 0x0e26f, 0x10eb3, 0x13aa2,
		0x1663f, 0x1918a, 0x1bc84, 0x1e72e, 0x2118b, 0x23b9a, 0x2655d, 0x28ed5,
		0x2b803, 0x2e0e8, 0x30985, 0x331db, 0x359eb, 0x381b6, 0x3a93d, 0x3d081,
		0x3f782, 0x41e42, 0x444c1, 0x46b01, 0x49101, 0x4b6c4, 0x4dc49, 0x50191,
		0x5269e, 0x54b6f, 0x57006, 0x59463, 0x5b888, 0x5dc74, 0x60029, 0x623a7,
		0x646ee, 0x66a00, 0x68cdd, 0x6af86, 0x6d1fa, 0x6f43c, 0x7164b, 0x73829,
		0x759d4, 0x77b4f, 0x79c9a, 0x7bdb5, 0x7dea1, 0x7ff5e, 0x81fed, 0x8404e,
		0x86082, 0x88089, 0x8a064, 0x8c014, 0x8df98, 0x8fef1, 0x91e20, 0x93d26,
		0x95c01, 0x97ab4, 0x9993e, 0x9b79f, 0x9d5d9, 0x9f3ec, 0xa11d8, 0xa2f9d,
		0xa4d3c, 0xa6ab5, 0xa8808, 0xaa537, 0xac241, 0xadf26, 0xafbe7, 0xb1885,
		0xb3500, 0xb5157, 0xb6d8c, 0xb899f, 0xba58f, 0xbc15e, 0xbdd0c, 0xbf899,
		0xc1404, 0xc2f50, 0xc4a7b, 0xc6587, 0xc8073, 0xc9b3f, 0xcb5ed, 0xcd07c,
		0xceaec, 0xd053f, 0xd1f73, 0xd398a, 0xd5384, 0xd6d60, 0xd8720, 0xda0c3,
		0xdba4a, 0xdd3b4, 0xded03, 0xe0636, 0xe1f4e, 0xe384a, 0xe512c, 0xe69f3,
		0xe829f, 0xe9b31, 0xeb3a9, 0xecc08, 0xee44c, 0xefc78, 0xf148a, 0xf2c83,
		0xf4463, 0xf5c2a, 0xf73da, 0xf8b71, 0xfa2f0, 0xfba57, 0xfd1a7, 0xfe8df
	};
	static char logSlopeTable[128] = {
		0x5c, 0x5c, 0x5b, 0x5a, 0x5a, 0x59, 0x58, 0x58,
		0x57, 0x56, 0x56, 0x55, 0x55, 0x54, 0x53, 0x53,
		0x52, 0x52, 0x51, 0x51, 0x50, 0x50, 0x4f, 0x4f,
		0x4e, 0x4d, 0x4d, 0x4d, 0x4c, 0x4c, 0x4b, 0x4b,
		0x4a, 0x4a, 0x49, 0x49, 0x48, 0x48, 0x47, 0x47,
		0x47, 0x46, 0x46, 0x45, 0x45, 0x45, 0x44, 0x44,
		0x43, 0x43, 0x43, 0x42, 0x42, 0x42, 0x41, 0x41,
		0x41, 0x40, 0x40, 0x40, 0x3f, 0x3f, 0x3f, 0x3e,
		0x3e, 0x3e, 0x3d, 0x3d, 0x3d, 0x3c, 0x3c, 0x3c,
		0x3b, 0x3b, 0x3b, 0x3b, 0x3a, 0x3a, 0x3a, 0x39,
		0x39, 0x39, 0x39, 0x38, 0x38, 0x38, 0x38, 0x37,
		0x37, 0x37, 0x37, 0x36, 0x36, 0x36, 0x36, 0x35,
		0x35, 0x35, 0x35, 0x34, 0x34, 0x34, 0x34, 0x34,
		0x33, 0x33, 0x33, 0x33, 0x32, 0x32, 0x32, 0x32,
		0x32, 0x31, 0x31, 0x31, 0x31, 0x31, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f
	};
	int i;

	if (rate == 0)
		return 0;	/* Bail out if no leading "1" */
	rate *= 11185;		/* Scale 48000 to 0x20002380 */
	for (i = 31; i > 0; i--) {
		if (rate & 0x80000000) {	/* Detect leading "1" */
			return (((uint32_t) (i - 15) << 20) +
			    logMagTable[0x7f & (rate >> 24)] +
			    (0x7f & (rate >> 17)) *
			    logSlopeTable[0x7f & (rate >> 24)]);
		}
		rate <<= 1;
	}

	return 0;		/* Should never reach this point */
}

static uint32_t
emu_rate_to_linearpitch(uint32_t rate)
{
	rate = (rate << 8) / 375;
	return (rate >> 1) + (rate & 1);
}
#if 0
#define PITCH_48000 0x00004000
#define PITCH_96000 0x00008000
#define PITCH_85000 0x00007155
#define PITCH_80726 0x00006ba2
#define PITCH_67882 0x00005a82
#define PITCH_57081 0x00004c1c

static uint32_t 
emu_select_interprom(uint32_t pitch_target)
{
	if (pitch_target == PITCH_48000)
		return CCCA_INTERPROM_0;
	else if (pitch_target < PITCH_48000)
		return CCCA_INTERPROM_1;
	else if (pitch_target >= PITCH_96000)
		return CCCA_INTERPROM_0;
	else if (pitch_target >= PITCH_85000)
		return CCCA_INTERPROM_6;
	else if (pitch_target >= PITCH_80726)
		return CCCA_INTERPROM_5;
	else if (pitch_target >= PITCH_67882)
		return CCCA_INTERPROM_4;
	else if (pitch_target >= PITCH_57081)
		return CCCA_INTERPROM_3;
	else
		return CCCA_INTERPROM_2;
}
#endif

static void
emu_setmap(void *arg, bus_dma_segment_t * segs, int nseg, int error)
{
	bus_addr_t *phys = (bus_addr_t *)arg;

	*phys = error ? 0 : (bus_addr_t) segs->ds_addr;

	if (bootverbose) {
		printf("emu10kx: setmap (%lx, %lx), nseg=%d, error=%d\n",
		    (unsigned long)segs->ds_addr, (unsigned long)segs->ds_len,
		    nseg, error);
	}
}

static void *
emu_malloc(struct emu_pcm_info *pcm_sc, uint32_t sz, bus_addr_t * addr)
{
	void *dmabuf;
	struct emu_sc_info *sc;
	bus_dmamap_t map;

	*addr = 0;
	sc = pcm_sc->card;
	if (bus_dmamem_alloc(sc->parent_dmat, &dmabuf, BUS_DMA_NOWAIT, &map))
		return NULL;
	if (bus_dmamap_load(sc->parent_dmat, map, dmabuf, sz, emu_setmap, addr, 0)
	    || !*addr)
		return NULL;
	return dmabuf;
}

static void
emu_free(struct emu_pcm_info *sc, void *dmabuf)
{
	bus_dmamem_free(sc->card->parent_dmat, dmabuf, NULL);
}

static void *
emu_memalloc(struct emu_pcm_info *sc, uint32_t sz, bus_addr_t * addr)
{
	uint32_t blksz, start, idx, ofs, tmp, found;
	struct emu_mem *mem = &sc->mem;
	struct emu_memblk *blk;
	void *membuf;

	blksz = sz / EMUPAGESIZE;
	if (sz > (blksz * EMUPAGESIZE))
		blksz++;
	/* find a free block in the bitmap */
	found = 0;
	start = 1;
	while (!found && start + blksz < MAXPAGES) {
		found = 1;
		for (idx = start; idx < start + blksz; idx++)
			if (mem->bmap[idx >> 3] & (1 << (idx & 7)))
				found = 0;
		if (!found)
			start++;
	}
	if (!found)
		return NULL;
	blk = malloc(sizeof(*blk), M_DEVBUF, M_NOWAIT);
	if (blk == NULL)
		return NULL;
	bzero(blk, sizeof(*blk));
	membuf = emu_malloc(sc, sz, &blk->buf_addr);
	*addr = blk->buf_addr;
	if (membuf == NULL) {
		free(blk, M_DEVBUF);
		return NULL;
	}
	blk->buf = membuf;
	blk->pte_start = start;
	blk->pte_size = blksz;
	ofs = 0;
	for (idx = start; idx < start + blksz; idx++) {
		mem->bmap[idx >> 3] |= 1 << (idx & 7);
		tmp = (uint32_t) (u_long) ((uint8_t *) blk->buf_addr + ofs);
		mem->ptb_pages[idx] = (tmp << 1) | idx;
		ofs += EMUPAGESIZE;
	}
	SLIST_INSERT_HEAD(&mem->blocks, blk, link);
	return membuf;
}

static int
emu_memfree(struct emu_pcm_info *sc, void *membuf)
{
	uint32_t idx, tmp;
	struct emu_mem *mem = &sc->mem;
	struct emu_memblk *blk, *i;

	blk = NULL;
	SLIST_FOREACH(i, &mem->blocks, link) {
		if (i->buf == membuf)
			blk = i;
	}
	if (blk == NULL)
		return EINVAL;
	SLIST_REMOVE(&mem->blocks, blk, emu_memblk, link);
	emu_free(sc, membuf);
	tmp = (uint32_t) (sc->mem.silent_page_addr) << 1;
	for (idx = blk->pte_start; idx < blk->pte_start + blk->pte_size; idx++) {
		mem->bmap[idx >> 3] &= ~(1 << (idx & 7));
		mem->ptb_pages[idx] = tmp | idx;
	}
	free(blk, M_DEVBUF);
	return 0;
}

static int
emu_memstart(struct emu_pcm_info *sc, void *membuf)
{
	struct emu_mem *mem = &sc->mem;
	struct emu_memblk *blk, *i;

	blk = NULL;
	SLIST_FOREACH(i, &mem->blocks, link) {
		if (i->buf == membuf)
			blk = i;
	}
	if (blk == NULL)
		return -EINVAL;
	return blk->pte_start;
}



static struct emu_pcm_voice *
emu_valloc(struct emu_pcm_info *sc)
{
	struct emu_pcm_voice *v;
	int i;

	v = NULL;
	for (i = 0; i < 64 && sc->voice[i].busy; i++);
	if (i < 64) {
		v = &sc->voice[i];
		v->busy = 1;
	}
	return v;
}

static int
emu_vinit(struct emu_pcm_info *sc, struct emu_pcm_voice *m, struct emu_pcm_voice *s,
    uint32_t sz, struct snd_dbuf *b)
{
	void *vbuf;
	bus_addr_t tmp_addr;

	vbuf = emu_memalloc(sc, sz, &tmp_addr);
	if (vbuf == NULL)
		return -1;
	if (b != NULL)
		sndbuf_setup(b, vbuf, sz);
	m->start = emu_memstart(sc, vbuf) * EMUPAGESIZE;
	m->end = m->start + sz;
	m->channel = NULL;
	m->speed = 0;
	m->b16 = 0;
	m->stereo = 0;
	m->running = 0;
	m->ismaster = 1;
	m->vol = 0xff;
	m->buf = tmp_addr;
	m->slave = s;
	if (s != NULL) {
		s->start = m->start;
		s->end = m->end;
		s->channel = NULL;
		s->speed = 0;
		s->b16 = 0;
		s->stereo = 0;
		s->running = 0;
		s->ismaster = 0;
		s->vol = m->vol;
		s->buf = m->buf;
		s->slave = NULL;
	}
	return 0;
}

static void
emu_vsetup(struct emu_pcm_pchinfo *ch)
{
	struct emu_pcm_voice *v = ch->master;

	if (ch->fmt) {
		v->b16 = (ch->fmt & AFMT_16BIT) ? 1 : 0;
		v->stereo = (ch->fmt & AFMT_STEREO) ? 1 : 0;
		if (v->slave != NULL) {
			v->slave->b16 = v->b16;
			v->slave->stereo = v->stereo;
		}
	}
	if (ch->spd) {
		v->speed = ch->spd;
		if (v->slave != NULL)
			v->slave->speed = v->speed;
	}
}

static void
emu_vwrite(struct emu_pcm_info *sc, struct emu_pcm_voice *v)
{
	int s;
	int i;
	unsigned int routing[8], amounts[8];
	uint32_t sa, ea, start, val, silent_page;

	s = (v->stereo ? 1 : 0) + (v->b16 ? 1 : 0);

	sa = v->start >> s;
	ea = v->end >> s;

	/* FIXME: Get Routing/Amounts from Mixer */
	for (i = 0; i < 8; i++) {
		routing[i] = i;
		amounts[i] = 0xff;
	};
	if (v->stereo) {
		if (v->ismaster) {
			amounts[0] = amounts[2] = amounts[4] = amounts[6] = 0;
		} else {
			amounts[1] = amounts[3] = amounts[5] = amounts[7] = 0;
		};
	};
	/* END Of FIXME */
	if (v->stereo) {
		emu_wrptr(sc->card, v->vnum, CPF, CPF_STEREO_MASK);
	} else {
		emu_wrptr(sc->card, v->vnum, CPF, 0);
	}
	val = v->stereo ? 28 : 30;
	val *= v->b16 ? 1 : 2;
	start = sa + val;

	if (sc->card->is_emu10k1) {
		emu_wrptr(sc->card, v->vnum, FXRT, ((routing[3] << 12) |
		    (routing[2] << 8) |
		    (routing[1] << 4) |
		    (routing[0] << 0)) << 16);
	} else {
		emu_wrptr(sc->card, v->vnum, A_FXRT1, (routing[3] << 24) |
		    (routing[2] << 16) |
		    (routing[1] << 8) |
		    (routing[0] << 0));
		emu_wrptr(sc->card, v->vnum, A_FXRT2, (routing[7] << 24) |
		    (routing[6] << 16) |
		    (routing[5] << 8) |
		    (routing[4] << 0));
		emu_wrptr(sc->card, v->vnum, A_SENDAMOUNTS, (amounts[7] << 24) |
		    (amounts[6] << 26) |
		    (amounts[5] << 8) |
		    (amounts[4] << 0));
	};
	emu_wrptr(sc->card, v->vnum, PTRX, (amounts[3] << 8) | (amounts[2] << 0));
	emu_wrptr(sc->card, v->vnum, DSL, ea | (amounts[1] << 24));
	emu_wrptr(sc->card, v->vnum, PSST, sa | (amounts[0] << 24));
#if 0
	emu_wrptr(sc->card, v->vnum, CCCA, start |
	    emu_select_interprom(emu_rate_to_linearpitch(v->speed)) |
	    (v->b16 ? 0 : CCCA_8BITSELECT));
#else
	emu_wrptr(sc->card, v->vnum, CCCA, start | (v->b16 ? 0 : CCCA_8BITSELECT));
#endif
	emu_wrptr(sc->card, v->vnum, Z1, 0);
	emu_wrptr(sc->card, v->vnum, Z2, 0);

	silent_page = ((uint32_t) (sc->mem.silent_page_addr) << 1) | MAP_PTI_MASK;
	emu_wrptr(sc->card, v->vnum, MAPA, silent_page);
	emu_wrptr(sc->card, v->vnum, MAPB, silent_page);

	emu_wrptr(sc->card, v->vnum, CVCF, CVCF_CURRENTFILTER_MASK);
	emu_wrptr(sc->card, v->vnum, VTFT, VTFT_FILTERTARGET_MASK);
	emu_wrptr(sc->card, v->vnum, ATKHLDM, 0);
	emu_wrptr(sc->card, v->vnum, DCYSUSM, DCYSUSM_DECAYTIME_MASK);
	emu_wrptr(sc->card, v->vnum, LFOVAL1, 0x8000);
	emu_wrptr(sc->card, v->vnum, LFOVAL2, 0x8000);
	emu_wrptr(sc->card, v->vnum, FMMOD, 0);
	emu_wrptr(sc->card, v->vnum, TREMFRQ, 0);
	emu_wrptr(sc->card, v->vnum, FM2FRQ2, 0);
	emu_wrptr(sc->card, v->vnum, ENVVAL, 0x8000);

	emu_wrptr(sc->card, v->vnum, ATKHLDV, ATKHLDV_HOLDTIME_MASK | ATKHLDV_ATTACKTIME_MASK);
	emu_wrptr(sc->card, v->vnum, ENVVOL, 0x8000);

	emu_wrptr(sc->card, v->vnum, PEFE_FILTERAMOUNT, 0x7f);
	emu_wrptr(sc->card, v->vnum, PEFE_PITCHAMOUNT, 0);
	if ((v->stereo) && (v->slave != NULL))
		emu_vwrite(sc, v->slave);
}

static void
emu_vtrigger(struct emu_pcm_info *sc, struct emu_pcm_voice *v, int go)
{
	uint32_t pitch_target, initial_pitch;
	uint32_t cra, cs, ccis;
	uint32_t sample, i;

	if (go) {
		cra = 64;
		cs = v->stereo ? 4 : 2;
		ccis = v->stereo ? 28 : 30;
		ccis *= v->b16 ? 1 : 2;
		sample = v->b16 ? 0x00000000 : 0x80808080;
		for (i = 0; i < cs; i++)
			emu_wrptr(sc->card, v->vnum, CD0 + i, sample);
		emu_wrptr(sc->card, v->vnum, CCR_CACHEINVALIDSIZE, 0);
		emu_wrptr(sc->card, v->vnum, CCR_READADDRESS, cra);
		emu_wrptr(sc->card, v->vnum, CCR_CACHEINVALIDSIZE, ccis);

		emu_wrptr(sc->card, v->vnum, IFATN, 0xff00);
		emu_wrptr(sc->card, v->vnum, VTFT, 0xffffffff);
		emu_wrptr(sc->card, v->vnum, CVCF, 0xffffffff);
		emu_wrptr(sc->card, v->vnum, DCYSUSV, 0x00007f7f);
		emu_enastop(sc, v->vnum, 0);

		pitch_target = emu_rate_to_linearpitch(v->speed);
		initial_pitch = emu_rate_to_pitch(v->speed) >> 8;
		emu_wrptr(sc->card, v->vnum, PTRX_PITCHTARGET, pitch_target);
		emu_wrptr(sc->card, v->vnum, CPF_CURRENTPITCH, pitch_target);
		emu_wrptr(sc->card, v->vnum, IP, initial_pitch);
	} else {
		emu_wrptr(sc->card, v->vnum, PTRX_PITCHTARGET, 0);
		emu_wrptr(sc->card, v->vnum, CPF_CURRENTPITCH, 0);
		emu_wrptr(sc->card, v->vnum, IFATN, 0xffff);
		emu_wrptr(sc->card, v->vnum, VTFT, 0x0000ffff);
		emu_wrptr(sc->card, v->vnum, CVCF, 0x0000ffff);
		emu_wrptr(sc->card, v->vnum, IP, 0);
		emu_enastop(sc, v->vnum, 1);
	}
	if ((v->stereo) && (v->slave != NULL))
		emu_vtrigger(sc, v->slave, go);
}

static int
emu_vpos(struct emu_pcm_info *sc, struct emu_pcm_voice *v)
{
	int s, ptr;

	s = (v->b16 ? 1 : 0) + (v->stereo ? 1 : 0);
	ptr = (emu_rdptr(sc->card, v->vnum, CCCA_CURRADDR) - (v->start >> s)) << s;
	return ptr & ~0x0000001f;
}
/* playback channel interface */
static void *
emupchan_init(kobj_t obj __unused, void *devinfo, struct snd_dbuf *b, struct pcm_channel *c, int dir __unused)
{
	struct emu_pcm_info *sc = devinfo;
	struct emu_pcm_pchinfo *ch;
	void *r;

	KASSERT(dir == PCMDIR_PLAY, ("emupchan_init: bad direction"));
	KASSERT(sc->pnum <= sc->card->num_fxbuses - 1, ("emupchan_init: channel overflow"));
	KASSERT(sc->card != NULL, ("empchan_init: no soundcard"));

	ch = &(sc->pch[sc->pnum++]);
	ch->buffer = b;
	ch->parent = sc;
	ch->channel = c;
	ch->blksz = sc->card->bufsz / 2;
	ch->fmt = AFMT_U8;
	ch->spd = 8000;
	snd_mtxlock(sc->lock);
	ch->master = emu_valloc(sc);
	ch->slave = emu_valloc(sc);
	snd_mtxunlock(sc->lock);
	r = (emu_vinit(sc, ch->master, ch->slave, sc->card->bufsz, ch->buffer)) ? NULL : ch;
	return r;
}

static int
emupchan_free(kobj_t obj __unused, void *c_devinfo)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;
	int r;

	snd_mtxlock(sc->lock);
	r = emu_memfree(sc, sndbuf_getbuf(ch->buffer));
	snd_mtxunlock(sc->lock);

	return r;
}

static int
emupchan_setformat(kobj_t obj __unused, void *c_devinfo, uint32_t format)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;

	ch->fmt = format;
	return 0;
}

static int
emupchan_setspeed(kobj_t obj __unused, void *c_devinfo, uint32_t speed)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;

	ch->spd = speed;
	return ch->spd;
}

static int
emupchan_setblocksize(kobj_t obj __unused, void *c_devinfo, uint32_t blocksize)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;

	if (blocksize > ch->parent->card->bufsz / 2)
		blocksize = ch->parent->card->bufsz / 2;
	ch->blksz = blocksize;
	snd_mtxlock(sc->lock);
	emu_settimer(sc);
	snd_mtxunlock(sc->lock);
	return blocksize;
}

static int
emupchan_trigger(kobj_t obj __unused, void *c_devinfo, int go)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;

	if (go == PCMTRIG_EMLDMAWR || go == PCMTRIG_EMLDMARD)
		return 0;
	snd_mtxlock(sc->lock);
	if (go == PCMTRIG_START) {
		emu_vsetup(ch);
		emu_vwrite(sc, ch->master);
		emu_settimer(sc);
		emu_enatimer(sc->card, 1);
	}
	ch->run = (go == PCMTRIG_START) ? 1 : 0;
	emu_vtrigger(sc, ch->master, ch->run);
	snd_mtxunlock(sc->lock);
	return 0;
}

static int
emupchan_getptr(kobj_t obj __unused, void *c_devinfo)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;
	int r;

	snd_mtxlock(sc->lock);
	r = emu_vpos(sc, ch->master);
	snd_mtxunlock(sc->lock);

	return r;
}

static struct pcmchan_caps *
emupchan_getcaps(kobj_t obj __unused, void *c_devinfo)
{
	struct emu_pcm_pchinfo *ch = c_devinfo;
	if (ch->parent->card->mixer->spdif_mode)
		return &emu_playcaps_pt;
	return &emu_playcaps;
}

static kobj_method_t emupchan_methods[] = {
	KOBJMETHOD(channel_init, emupchan_init),
	KOBJMETHOD(channel_free, emupchan_free),
	KOBJMETHOD(channel_setformat, emupchan_setformat),
	KOBJMETHOD(channel_setspeed, emupchan_setspeed),
	KOBJMETHOD(channel_setblocksize, emupchan_setblocksize),
	KOBJMETHOD(channel_trigger, emupchan_trigger),
	KOBJMETHOD(channel_getptr, emupchan_getptr),
	KOBJMETHOD(channel_getcaps, emupchan_getcaps),
	{0, 0}
};
CHANNEL_DECLARE(emupchan);

/* recording channel interface */
static void *
emurchan_init(kobj_t obj __unused, void *devinfo, struct snd_dbuf *b, struct pcm_channel *c, int dir __unused)
{
	struct emu_pcm_info *sc = devinfo;
	struct emu_pcm_rchinfo *ch;

	KASSERT(dir == PCMDIR_REC, ("emurchan_init: bad direction"));
	ch = &sc->rch[sc->rnum];
	ch->buffer = b;
	ch->parent = sc;
	ch->channel = c;
	ch->blksz = sc->card->bufsz / 2;
	ch->fmt = AFMT_U8;
	ch->spd = 8000;
	ch->num = sc->rnum;
	switch (sc->rnum) {
	case 0:
		ch->idxreg = sc->card->is_emu10k1 ? ADCIDX : A_ADCIDX;
		ch->basereg = ADCBA;
		ch->sizereg = ADCBS;
		ch->setupreg = ADCCR;
		ch->irqmask = INTE_ADCBUFENABLE;
		ch->iprmask = IPR_ADCBUFFULL | IPR_ADCBUFHALFFULL;
		break;

	case 1:
		ch->idxreg = FXIDX;
		ch->basereg = FXBA;
		ch->sizereg = FXBS;
		ch->setupreg = sc->card->is_emu10k1 ? FXWC : A_FXWC1;
		ch->irqmask = INTE_EFXBUFENABLE;
		ch->iprmask = IPR_EFXBUFFULL | IPR_EFXBUFHALFFULL;
		break;

	case 2:
		ch->idxreg = MICIDX;
		ch->basereg = MICBA;
		ch->sizereg = MICBS;
		ch->setupreg = 0;
		ch->irqmask = INTE_MICBUFENABLE;
		ch->iprmask = IPR_MICBUFFULL | IPR_MICBUFHALFFULL;
		break;
	}
	sc->rnum++;
	if (sndbuf_alloc(ch->buffer, sc->card->parent_dmat, sc->card->bufsz) != 0)
		return NULL;
	else {
		snd_mtxlock(sc->lock);
#if __FreeBSD_version < 500000
		emu_wrptr(sc->card, 0, ch->basereg, vtophys(sndbuf_getbuf(ch->buffer)));
#else
		emu_wrptr(sc->card, 0, ch->basereg, sndbuf_getbufaddr(ch->buffer));
#endif
		emu_wrptr(sc->card, 0, ch->sizereg, 0);	/* off */
		snd_mtxunlock(sc->lock);
		return ch;
	}
}

static int
emurchan_setformat(kobj_t obj __unused, void *c_devinfo, uint32_t format)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;

	ch->fmt = format;
	return 0;
}

static int
emurchan_setspeed(kobj_t obj __unused, void *c_devinfo, uint32_t speed)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;

	if (ch->num == 0) {
		if (ch->parent->card->is_emu10k1) {
			speed = emu10k1_adcspeed[emu_k1_recval(speed)];
		} else {
			speed = emu10k2_adcspeed[emu_k2_recval(speed)];
		};
	};
	if (ch->num == 1)
		speed = 48000;
	if (ch->num == 2)
		speed = 8000;
	ch->spd = speed;
	return ch->spd;
}

static int
emurchan_setblocksize(kobj_t obj __unused, void *c_devinfo, uint32_t blocksize)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;

	ch->blksz = blocksize;
	return blocksize;
}
/* semantic note: must start at beginning of buffer */
static int
emurchan_trigger(kobj_t obj __unused, void *c_devinfo, int go)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;
	uint32_t val, sz;

	switch (sc->card->bufsz) {
	case 4096:
		sz = ADCBS_BUFSIZE_4096;
		break;

	case 8192:
		sz = ADCBS_BUFSIZE_8192;
		break;

	case 16384:
		sz = ADCBS_BUFSIZE_16384;
		break;

	case 32768:
		sz = ADCBS_BUFSIZE_32768;
		break;

	case 65536:
		sz = ADCBS_BUFSIZE_65536;
		break;

	default:
		sz = ADCBS_BUFSIZE_4096;
	}

	snd_mtxlock(sc->lock);
	switch (go) {
	case PCMTRIG_START:
		ch->run = 1;
		emu_wrptr(sc->card, 0, ch->sizereg, sz);
		if (ch->num == 0) {
			val = sc->card->is_emu10k1 ? ADCCR_LCHANENABLE : A_ADCCR_LCHANENABLE;
			if (ch->fmt & AFMT_STEREO)
				val |= sc->card->is_emu10k1 ? ADCCR_RCHANENABLE : A_ADCCR_RCHANENABLE;
			val |= sc->card->is_emu10k1 ? emu_k1_recval(ch->spd) : emu_k2_recval(ch->spd);
			emu_wrptr(sc->card, 0, ch->setupreg, 0);
			emu_wrptr(sc->card, 0, ch->setupreg, val);
		}
		ch->ihandle = emu_intr_register(sc->card, ch->irqmask, ch->iprmask, &emu_pcm_intr, sc);
		break;

	case PCMTRIG_STOP:
	case PCMTRIG_ABORT:
		ch->run = 0;
		emu_wrptr(sc->card, 0, ch->sizereg, 0);
		if (ch->setupreg)
			emu_wrptr(sc->card, 0, ch->setupreg, 0);
		(void)emu_intr_unregister(sc->card, ch->ihandle);
		break;

	case PCMTRIG_EMLDMAWR:
	case PCMTRIG_EMLDMARD:
	default:
		break;
	}
	snd_mtxunlock(sc->lock);

	return 0;
}

static int
emurchan_getptr(kobj_t obj __unused, void *c_devinfo)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;
	struct emu_pcm_info *sc = ch->parent;
	int r;

	snd_mtxlock(sc->lock);
	r = emu_rdptr(sc->card, 0, ch->idxreg) & 0x0000ffff;
	snd_mtxunlock(sc->lock);

	return r;
}

static struct pcmchan_caps *
emurchan_getcaps(kobj_t obj __unused, void *c_devinfo)
{
	struct emu_pcm_rchinfo *ch = c_devinfo;

	return &emu_reccaps[ch->num];
}

static kobj_method_t emurchan_methods[] = {
	KOBJMETHOD(channel_init, emurchan_init),
	KOBJMETHOD(channel_setformat, emurchan_setformat),
	KOBJMETHOD(channel_setspeed, emurchan_setspeed),
	KOBJMETHOD(channel_setblocksize, emurchan_setblocksize),
	KOBJMETHOD(channel_trigger, emurchan_trigger),
	KOBJMETHOD(channel_getptr, emurchan_getptr),
	KOBJMETHOD(channel_getcaps, emurchan_getcaps),
	{0, 0}
};
CHANNEL_DECLARE(emurchan);

/* -------------------------------------------------------------------- */
/* The interrupt handler */
/*FIXME*/
uint32_t
emu_pcm_intr(void *pcm, uint32_t stat)
{
	struct emu_pcm_info *sc = (struct emu_pcm_info *)pcm;
	uint32_t ack, i, x;
	ack = 0;
	if (stat & IPR_INTERVALTIMER) {
		ack |= IPR_INTERVALTIMER;
		x = 0;
		for (i = 0; i < sc->card->num_fxbuses; i++) {
			if (sc->pch[i].run > 1) {
				sc->pch[i].run = 0;
			}
			if (sc->pch[i].run == 1) {
				x = 1;
				chn_intr(sc->pch[i].channel);
			}
		}

		if (x == 0) {
			emu_enatimer(sc->card, 0);
		};
	}
	if (stat & (IPR_ADCBUFFULL | IPR_ADCBUFHALFFULL)) {
		ack |= stat & (IPR_ADCBUFFULL | IPR_ADCBUFHALFFULL);
		if (sc->rch[0].channel)
			chn_intr(sc->rch[0].channel);
	}
	if (stat & (IPR_EFXBUFFULL | IPR_EFXBUFHALFFULL)) {
		ack |= stat & (IPR_EFXBUFFULL | IPR_EFXBUFHALFFULL);
		if (sc->rch[1].channel)
			chn_intr(sc->rch[1].channel);
	}
	if (stat & (IPR_MICBUFFULL | IPR_MICBUFHALFFULL)) {
		ack |= stat & (IPR_MICBUFFULL | IPR_MICBUFHALFFULL);
		if (sc->rch[2].channel)
			chn_intr(sc->rch[2].channel);
	}
	return ack;
}
/* -------------------------------------------------------------------- */

static int
emu_pcm_init(struct emu_pcm_info *sc)
{
	uint32_t ch, tmp, i;

	SLIST_INIT(&sc->mem.blocks);
	sc->mem.ptb_pages = emu_malloc(sc, MAXPAGES * sizeof(uint32_t), &sc->mem.ptb_pages_addr);
	if (sc->mem.ptb_pages == NULL)
		return -1;

	sc->mem.silent_page = emu_malloc(sc, EMUPAGESIZE, &sc->mem.silent_page_addr);
	if (sc->mem.silent_page == NULL) {
		emu_free(sc, sc->mem.ptb_pages);
		return -1;
	}
	/* Clear page with silence & setup all pointers to this page */
	bzero(sc->mem.silent_page, EMUPAGESIZE);
	tmp = (uint32_t) (sc->mem.silent_page_addr) << 1;
	for (i = 0; i < MAXPAGES; i++)
		sc->mem.ptb_pages[i] = tmp | i;

	for (ch = 0; ch < NUM_G; ch++) {
		emu_wrptr(sc->card, ch, MAPA, tmp | MAP_PTI_MASK);
		emu_wrptr(sc->card, ch, MAPB, tmp | MAP_PTI_MASK);
	}
	emu_wrptr(sc->card, 0, PTB, (sc->mem.ptb_pages_addr));
	emu_wrptr(sc->card, 0, TCB, 0);	/* taken from original driver */
	emu_wrptr(sc->card, 0, TCBS, 0);	/* taken from original driver */

	for (ch = 0; ch < NUM_G; ch++) {
		sc->voice[ch].vnum = ch;
		sc->voice[ch].slave = NULL;
		sc->voice[ch].busy = 0;
		sc->voice[ch].ismaster = 0;
		sc->voice[ch].running = 0;
		sc->voice[ch].b16 = 0;
		sc->voice[ch].stereo = 0;
		sc->voice[ch].speed = 0;
		sc->voice[ch].start = 0;
		sc->voice[ch].end = 0;
		sc->voice[ch].channel = NULL;
	};
	sc->pnum = sc->rnum = 0;
	if (sc->card)
		emu_enatimer(sc->card, 1);
	return 0;
}

static int
emu_pcm_uninit(struct emu_pcm_info *sc)
{
	/* init envelope engine */
	if (!SLIST_EMPTY(&sc->mem.blocks))
		device_printf(sc->dev, "warning: memblock list not empty\n");
	emu_free(sc, sc->mem.ptb_pages);
	emu_free(sc, sc->mem.silent_page);


	return 0;
}

static int
emu_pcm_probe(device_t dev)
{
	uintptr_t func, r;

	/* The parent device has already been probed. */
	r = BUS_READ_IVAR(device_get_parent(dev), dev, 0, &func);
	if (func != SCF_PCM)
		return (ENXIO);

	device_set_desc(dev, "EMU10Kx PCM Interface");
	return (0);
}

static int
emu_pcm_attach(device_t dev)
{
	struct ac97_info *codec = NULL;
	struct emu_pcm_info *sc;
	unsigned int i, gotmic;
	char status[SND_STATUSLEN];
	uint32_t inte, ipr;

	if ((sc = malloc(sizeof(*sc), M_DEVBUF, M_WAITOK | M_ZERO)) == NULL) {
		device_printf(dev, "cannot allocate softc\n");
		return ENXIO;
	}
	bzero(sc, sizeof(*sc));


	sc->card = (struct emu_sc_info *)(device_get_softc(device_get_parent(dev)));
	if (sc->card == NULL) {
		device_printf(dev, "cannot get bridge conf\n");
		return ENXIO;
	}
	sc->lock = snd_mtxcreate(device_get_nameunit(dev), "sound softc");
	sc->dev = dev;

	inte = INTE_INTERVALTIMERENB;
	ipr = IPR_INTERVALTIMER;/* Used by playback */

	sc->ihandle = emu_intr_register(sc->card, inte, ipr, &emu_pcm_intr, sc);

	if (emu_pcm_init(sc) == -1) {
		device_printf(dev, "unable to initialize PCM part of the card\n");
		goto bad;
	}
	if (sc->card->has_ac97) {
		codec = AC97_CREATE(dev, sc, emu_ac97);
		if (codec == NULL) {
			device_printf(dev, "can't create AC97 codec!\n");
			goto bad;
		}
		sc->codec = codec;
		gotmic = (ac97_getcaps(codec) & AC97_CAP_MICCHANNEL) ? 1 : 0;
		if (mixer_init(dev, ac97_getmixerclass(), codec) == -1) {
			device_printf(dev, "can't init mixer!\n");
			goto bad;
		}
	} else {
		codec = NULL;
		device_printf(dev, "no AC97 codec on card, use emuctrl to control volume\n");
	}

	if (pcm_register(dev, sc, sc->card->num_fxbuses, 1)) {
		device_printf(dev, "can't register PCM channels!\n");
		goto bad;
	}
	for (i = 0; i < sc->card->num_fxbuses; i++)
		pcm_addchan(dev, PCMDIR_PLAY, &emupchan_class, sc);
	pcm_addchan(dev, PCMDIR_REC, &emurchan_class, sc);
	snprintf(status, SND_STATUSLEN, "on %s", device_get_nameunit(sc->card->dev));
	pcm_setstatus(dev, status);

	return 0;

bad:
	if (codec)
		ac97_destroy(codec);
	if (sc->lock)
		snd_mtxfree(sc->lock);
	free(sc, M_DEVBUF);
	return ENXIO;
}

static int
emu_pcm_detach(device_t dev)
{
	int r;
	struct emu_pcm_info *sc;

	r = pcm_unregister(dev);
	if (r)
		return r;

	sc = pcm_getdevinfo(dev);

	/* shutdown chip */
	emu_pcm_uninit(sc);

	if (sc->lock)
		snd_mtxfree(sc->lock);
	free(sc, M_DEVBUF);

	return 0;
}
/* add suspend, resume */
static device_method_t emu_pcm_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe, emu_pcm_probe),
	DEVMETHOD(device_attach, emu_pcm_attach),
	DEVMETHOD(device_detach, emu_pcm_detach),

	{0, 0}
};

static driver_t emu_pcm_driver = {
	"pcm",
	emu_pcm_methods,
	PCM_SOFTC_SIZE,
	NULL,
	0,
	NULL
};
DRIVER_MODULE(snd_emu10kx_pcm, emu10kx, emu_pcm_driver, pcm_devclass, 0, 0);
#ifdef PCM_MODVER
MODULE_DEPEND(snd_emu10kx_pcm, snd_pcm, PCM_MINVER, PCM_PREFVER, PCM_MAXVER);
#endif
#ifdef SOUND_MODVER
MODULE_DEPEND(snd_emu10kx_pcm, sound, SOUND_MINVER, SOUND_PREFVER, SOUND_MAXVER);
#endif
MODULE_VERSION(snd_emu10kx_pcm, 1);
