/*
 * Copyright (c) 1999 Cameron Grant <gandalf@vilnya.demon.co.uk>
 * Copyright (c) 2003,2004 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHERIN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx-fx.c,v 1.47 2005/10/15 08:45:51 chibis Exp $
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif


#include "emu10kx.h"
#include "emu10kx-mixer.h"

static void
emu_wrefx(struct emu_sc_info *sc, unsigned int pc, unsigned int data)
{
	emu_wrptr(sc, 0, sc->code_base + pc, data);
}


void
emu_addefxop(struct emu_sc_info *sc, unsigned int op, unsigned int z, unsigned int w, unsigned int x, unsigned int y, uint32_t * pc)
{
	if ((*pc) + 1 > sc->code_size) {
		device_printf(sc->dev, "DSP CODE OVERRUN: attept to write past code_size (pc=%d)\n", (*pc));
		return;
	};
	emu_wrefx(sc, (*pc) * 2, (x << sc->high_operand_shift) | y);
	emu_wrefx(sc, (*pc) * 2 + 1, (op << sc->opcode_shift) | (z << sc->high_operand_shift) | w);
	(*pc)++;
}

void
emu_initefx(struct emu_sc_info *sc)
{
	unsigned int i;
	unsigned int c;
	uint32_t pc;
	struct emu_connector *mix;
	struct emu_connector *conn;

/* stop DSP */
	if (sc->is_emu10k1) {
		emu_wrptr(sc, 0, DBG, DBG_SINGLE_STEP);
	} else {
		emu_wrptr(sc, 0, A_DBG, A_DBG_SINGLE_STEP);
	};



/* code size is in instructions */
	pc = 0;
	for (c = 0; c < sc->code_size; c++) {
		if (sc->is_emu10k1) {
			emu_addefxop(sc, ACC3, DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
		} else {
			emu_addefxop(sc, SKIP, DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0xf), DSP_CONST(0x0), &pc);
		}
	};

	pc = 0;

	/* Input playback volumes */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		conn = &(sc->mixer->inputs[i]);
		emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), INP(conn->offset + 0), GPR(conn->gpr[GPR_P_VOLUME_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), INP(conn->offset + 1), GPR(conn->gpr[GPR_P_VOLUME_RIGHT]), &pc);
		};
	};

	/* FX (PCM) playback volumes */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		conn = &(sc->mixer->fxbuses[i]);
		emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), FX(conn->offset + 0), GPR(conn->gpr[GPR_P_VOLUME_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), FX(conn->offset + 1), GPR(conn->gpr[GPR_P_VOLUME_RIGHT]), &pc);
		};
	};

	/* Input recording volumes */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		conn = &(sc->mixer->inputs[i]);
		emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), INP(conn->offset + 0), GPR(conn->gpr[GPR_R_VOLUME_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), INP(conn->offset + 1), GPR(conn->gpr[GPR_R_VOLUME_RIGHT]), &pc);
		};
	};

	/* FX (PCM) recording volumes */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		conn = &(sc->mixer->fxbuses[i]);
		emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), FX(conn->offset + 0), GPR(conn->gpr[GPR_R_VOLUME_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), FX(conn->offset + 1), GPR(conn->gpr[GPR_R_VOLUME_RIGHT]), &pc);
		};
	};
#if 0
	/* Clean outputs */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		conn = &(sc->mixer->outputs[i]);
		emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
		};
	};
#endif
	/* Mixer */
	/* Clean MIX_GPRs */
	mix = &(sc->mixer->mix);
	emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
	emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
	emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
	emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);

	/* Mix inputs */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		conn = &(sc->mixer->inputs[i]);
		/* for playback */
		emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_LEFT]), GPR(conn->gpr[GPR_P_CACHE_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(conn->gpr[GPR_P_CACHE_RIGHT]), &pc);
		} else {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(conn->gpr[GPR_P_CACHE_LEFT]), &pc);
		};
	};
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		conn = &(sc->mixer->inputs[i]);
		/* for recording */
		emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_LEFT]), GPR(conn->gpr[GPR_R_CACHE_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(conn->gpr[GPR_R_CACHE_RIGHT]), &pc);
		} else {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(conn->gpr[GPR_R_CACHE_LEFT]), &pc);
		};
	};

	/* Mix FX */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		conn = &(sc->mixer->fxbuses[i]);
		/* for playback */
		emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_LEFT]), GPR(conn->gpr[GPR_P_CACHE_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(conn->gpr[GPR_P_CACHE_RIGHT]), &pc);
		} else {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(conn->gpr[GPR_P_CACHE_LEFT]), &pc);
		};
	};

	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		conn = &(sc->mixer->fxbuses[i]);
		/* for recording */
		emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_LEFT]), GPR(conn->gpr[GPR_R_CACHE_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(conn->gpr[GPR_R_CACHE_RIGHT]), &pc);
		} else {
			emu_addefxop(sc, ACC3, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(conn->gpr[GPR_R_CACHE_LEFT]), &pc);
		};
	};

	/* SB Live! may need additional amplification */
	emu_addefxop(sc, MACINTS, GPR(mix->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_LEFT]), GPR(sc->mixer->amp.gpr[GPR_P_VOLUME]), &pc);
	emu_addefxop(sc, MACINTS, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(sc->mixer->amp.gpr[GPR_P_VOLUME]), &pc);
	emu_addefxop(sc, MACINTS, GPR(mix->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_LEFT]), GPR(sc->mixer->amp.gpr[GPR_P_VOLUME]), &pc);
	emu_addefxop(sc, MACINTS, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(sc->mixer->amp.gpr[GPR_P_VOLUME]), &pc);

	/* set main volume for playback and recording */
	emu_addefxop(sc, MACS, GPR(mix->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_LEFT]), GPR(mix->gpr[GPR_P_VOLUME_LEFT]), &pc);
	emu_addefxop(sc, MACS, GPR(mix->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), GPR(mix->gpr[GPR_P_VOLUME_RIGHT]), &pc);
	emu_addefxop(sc, MACS, GPR(mix->gpr[GPR_R_CACHE_LEFT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_LEFT]), GPR(mix->gpr[GPR_R_VOLUME_LEFT]), &pc);
	emu_addefxop(sc, MACS, GPR(mix->gpr[GPR_R_CACHE_RIGHT]), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), GPR(mix->gpr[GPR_R_VOLUME_RIGHT]), &pc);

/* ROUTE */

	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		conn = &(sc->mixer->outputs[i]);
		if (conn->channels_used == 2) {
			if (conn->volume[MIXER_R_VOLUME_LEFT] == (-2)) {
				/* rec src */
				emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_LEFT]), &pc);
				emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), DSP_CONST(0x0), GPR(mix->gpr[GPR_R_CACHE_RIGHT]), &pc);
			} else {
				emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_LEFT]), &pc);
				emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_RIGHT]), DSP_CONST(0x0), DSP_CONST(0x0), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), &pc);
			};
		} else {

			if (conn->volume[MIXER_R_VOLUME_LEFT] == (-2)) {
				/* mono rec src (the only one known to me is
				 * "mic rec") I think we should keep it silent
				 * - driver does not support recording from
				 * anything other than "D/A Converter" */
				emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE_LEFT]), DSP_CONST(0x0), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
			} else {
				/* mono outputs (center/sub): they should be
				 * silent if we enable digital output and not
				 * silent if we disable it. We control it with
				 * monomix.gpr[GPR_P_VOLUME]: it's set in
				 * emumix_set_mode to 0 if we go digital and
				 * to 0x3ffffff if we enable analog output. In
				 * later case we have center and sub both
				 * equal to (left+right)/2. No lowpass filter
				 * for sub for now. */
				emu_addefxop(sc, MACS, GPR(sc->mixer->monomix.gpr[GPR_P_CACHE]), DSP_CONST(0x0), GPR(sc->mixer->monomix.gpr[GPR_P_VOLUME]), GPR(mix->gpr[GPR_P_CACHE_LEFT]), &pc);
				emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE_LEFT]), GPR(sc->mixer->monomix.gpr[GPR_P_CACHE]), GPR(sc->mixer->monomix.gpr[GPR_P_VOLUME]), GPR(mix->gpr[GPR_P_CACHE_RIGHT]), &pc);
			};
		};
	};


	/* Output volumes */
#if 0
	if (sc->is_emu10k2) { /* BROKEN */
		/* First output is Digital Out, with optional AC3 PASSTHROUGH */
#define A_GPR_COND	0xD7
		conn = &(sc->mixer->outputs[0]);

		for (i = 0; i < 2; i++) {
			/* Build PT data */
			emu_addefxop(sc, MACS, GPR(sc->mixer->pt_tgprs + i), FX(0 + i), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
			emu_addefxop(sc, SKIP, A_GPR_COND, A_GPR_COND, GPR(sc->mixer->pt_gprs + 0), DSP_CONST(0x1), &pc);	/* skip next if
																 * FX_PT_LEFT <=0  */
			emu_addefxop(sc, ACC3, GPR(sc->mixer->pt_tgprs + i), DSP_CONST(0x0), DSP_CONST(0x9), GPR(sc->mixer->pt_tgprs + i), &pc);	/* tgprs = tgprs + 65536 */
			emu_addefxop(sc, ANDXOR, GPR(sc->mixer->pt_ogprs + i), GPR(sc->mixer->pt_tgprs + i), GPR(sc->mixer->pt_gprs + 1), DSP_CONST(0x00), &pc);	/* ogprs = tgprs &&
																					 * 0xffff0000 */
			/* Build PCM data */
			emu_addefxop(sc, MACS, GPR(conn->gpr[GPR_P_CACHE + i]), DSP_CONST(0x0), GPR(conn->gpr[GPR_P_CACHE + i]), GPR(conn->gpr[GPR_P_VOLUME + i]), &pc);
			/* Select PT or PCM based on pt_control */
			emu_addefxop(sc, MACS, DSP_CONST(0x0), GPR(sc->mixer->pt_control), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
			emu_addefxop(sc, SKIP, A_GPR_COND, A_GPR_COND, COND_EQ_ZERO, DSP_CONST(0x1), &pc);	/* skip next if
														 * sc->mixer->pt_control
														 * == 0 */
			emu_addefxop(sc, ACC3, GPR(conn->gpr[GPR_P_CACHE + i]), GPR(sc->mixer->pt_ogprs + i), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
			emu_addefxop(sc, ACC3, OUTP(conn->offset + i), GPR(conn->gpr[GPR_P_CACHE + i]), DSP_CONST(0x0), DSP_CONST(0x0), &pc);
		}

	};

#endif
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		conn = &(sc->mixer->outputs[i]);
		emu_addefxop(sc, MACS, OUTP(conn->offset + 0), DSP_CONST(0x0), GPR(conn->gpr[GPR_P_CACHE_LEFT]), GPR(conn->gpr[GPR_P_VOLUME_LEFT]), &pc);
		if (conn->channels_used == 2) {
			emu_addefxop(sc, MACS, OUTP(conn->offset + 1), DSP_CONST(0x0), GPR(conn->gpr[GPR_P_CACHE_RIGHT]), GPR(conn->gpr[GPR_P_VOLUME_RIGHT]), &pc);
		};
	};

	sc->routing_code_end = pc;

/* start DSP */

	if (sc->is_emu10k1) {
		emu_wrptr(sc, 0, DBG, 0);
	} else {
		emu_wrptr(sc, 0, A_DBG, 0);
	};
}
