/*
 * Copyright (c) 2003 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHERIN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
TODO
1. Use card features (like 5.1, 7.1, and AC97)
2. Don't use arrays anymore
*/

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>
#include <sys/malloc.h>
#include <sys/systm.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif

#include "emu10kx.h"
#include "emu10kx-mixer.h"
#include "emu10kx-rm.h"
#include "emu10kx-vol.h"

static int log2lin(int log_t);

void
emumix_set_mode(struct emu_sc_info *sc, int mode)
{
	uint32_t a_iocfg;
	uint32_t hcfg;
	uint32_t tmp;

	switch (mode) {
	case MODE_DIGITAL:
	case MODE_ANALOG:
		break;
	default:
		return;
	}
	/* Compute HCFG register value... Enable Audio = 1 Mute Disable Audio
	 * = 0 on SB Live and Audigy, 1 on Audigy 2 Lock Tank Memory = 1 on SB
	 * Live!, 0 on Audigy Lock Sound Memory = 0 Auto Mute = 1 Enable
	 * Digital is set by 'mode' parameter */

	hcfg = HCFG_AUDIOENABLE | HCFG_AUTOMUTE;
	a_iocfg = 0;

	if (sc->rev >= 6)
		hcfg |= HCFG_JOYENABLE;

	if (sc->is_emu10k1)
		hcfg |= HCFG_LOCKTANKCACHE_MASK;
	else
		hcfg |= HCFG_CODECFORMAT_I2S | HCFG_JOYENABLE;


	if (mode == MODE_DIGITAL) {
		if (sc->broken_digital) {
			device_printf(sc->dev, "Digital mode is reported as broken on this card,\n");
			device_printf(sc->dev, "please use emuctrl to switch card to analog mode\n");
		};
		a_iocfg |= A_IOCFG_GPOUT0;
		hcfg |= HCFG_GPOUT0;
	};

	if (mode == MODE_ANALOG)
		emumix_set_spdif_mode(sc, SPDIF_MODE_PCM);

	if (sc->is_emu10k2)
		a_iocfg |= 0x80;

	if (sc->is_ca0151)
		a_iocfg |= A_IOCFG_AMUTE;

	if (sc->is_ca0108)
		a_iocfg |= 0x60;

	emu_wr(sc, HCFG, hcfg, 4);

	if ((sc->is_emu10k2) || (sc->is_ca0151) || (sc->is_ca0108)) {
		tmp = emu_rd(sc, A_IOCFG, 2);
		tmp = a_iocfg;
		emu_wr(sc, A_IOCFG, tmp, 2);
	};

	sc->mixer->is_digital = mode;

	/* Digital output can be on the same connector with center/sub.
	 * Mute center/sub if we go digital on this cards.*/
	if ((sc->is_emu10k1) || (sc->is_emu10k2) || (sc->is_ca0151))
		emumix_set_gpr(sc, sc->mixer->monomix.gpr[GPR_P_VOLUME], (sc->mixer->is_digital ? 0 : 0x3fffffff));
	if (sc->is_ca0108)
		emumix_set_gpr(sc, sc->mixer->monomix.gpr[GPR_P_VOLUME], 0x3fffffff);
}

void
emumix_set_spdif_mode(struct emu_sc_info *sc, int mode)
{
	uint32_t spcs;

	if (sc->mixer->is_digital == MODE_ANALOG)
		return;

	switch (mode) {
	case SPDIF_MODE_PCM:
		break;
	case SPDIF_MODE_AC3:
		device_printf(sc->dev, "AC3 mode does not work and disabled\n");
		return;
	default:
		return;
	};

        /*
        * Clock accuracy    = 0 (1000ppm)
        * Sample Rate       = 2 (48kHz)
        * Audio Channel     = 1 (Left of 2)
        * Source Number     = 0 (Unspecified)
        * Generation Status = 1 (Original for Cat Code 12)
        * Cat Code          = 12 (Digital Signal Mixer)
        * Mode              = 0 (Mode0)
        * Emphasis          = 0 (None)
        * CP                = 1 (Copyright unasserted)
        * AN                = 0 (Audio data)
        * P                 = 0 (Consumer)
        */

	spcs = SPCS_CLKACCY_1000PPM | SPCS_SAMPLERATE_48 |
	    SPCS_CHANNELNUM_LEFT | SPCS_SOURCENUM_UNSPEC |
	    SPCS_GENERATIONSTATUS | 0x00001200 | 0x00000000 |
	    SPCS_EMPHASIS_NONE | SPCS_COPYRIGHT;

	if (sc->is_emu10k1) {
		mode = SPDIF_MODE_PCM;
	} else {
		if (mode == SPDIF_MODE_AC3) {
/*			device_printf(sc->dev, "Audigy AC3 passthrough enabled\n");*/
			emu_wrptr(sc, 0, GPR(sc->mixer->pt_control), 0x1);
/*			emu_wrptr(sc, 0, SPBYPASS, 0x01);  ?? */
			spcs = spcs | SPCS_NOTAUDIODATA;
		} else {
/*			device_printf(sc->dev, "Audigy AC3 passthrough disabled\n");*/
			emu_wrptr(sc, 0, GPR(sc->mixer->pt_control), 0x0);
		};
	};

	emu_wrptr(sc, 0, SPCS0, spcs);
	emu_wrptr(sc, 0, SPCS1, spcs);
	emu_wrptr(sc, 0, SPCS2, spcs);

	sc->mixer->spdif_mode = mode;
}


static int
log2lin(int log_t)
{
	int lin_t;
	int idx, lin;

	if (log_t <= 0) {
		lin_t = 0x00000000;
		return lin_t;
	};
	if (log_t >= 100) {
		lin_t = 0x7fffffff;
		return lin_t;
	};

	idx = (L2L_POINTS - 1) - log_t / (L2L_POINTS);
	lin = log_t % (L2L_POINTS);
	lin_t = l2l_df[idx] * lin + l2l_f[idx];
	return lin_t;
}


void
emumix_set_fxvol(struct emu_sc_info *sc, unsigned gpr, int32_t vol)
{
	vol = log2lin(vol);
	emumix_set_gpr(sc, gpr, vol);
}

void
emumix_set_gpr(struct emu_sc_info *sc, unsigned gpr, int32_t val)
{
	emu_wrptr(sc, 0, GPR(gpr), val);
}
/* */
#define NUM_K1_INPUTS	8
static struct emu_connector emu_k1_inputs[NUM_K1_INPUTS] = {
	 /* 0x0 */ {IN_AC97, 0, STEREO, {100, 100, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 Codec"},
	 /* 0x2 */ {IN_SPDIF_CD, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "CD S/PDIF"},
	 /* 0x4 */ {IN_ZOOM, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "ZoomedVideo"},
	 /* 0x6 */ {IN_TOSLINK, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "TOS Link"},
	 /* 0x8 */ {IN_LINE1, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Line-In (1)"},
	 /* 0xA */ {IN_COAX_SPDIF, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Coaxial S/PDIF"},
	 /* 0xC */ {IN_LINE2, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Line-In (2)"},
	 /* 0xE */ {0x0E, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x0E"}
};
#define NUM_K1_OUTPUTS	20
static struct emu_connector emu_k1_outputs[NUM_K1_OUTPUTS] = {
	 /* 0x00 */ {OUT_AC97, 0, STEREO, {100, 100, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 (Front Speakers)"},
	 /* 0x02 */ {OUT_TOSLINK, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "TOS Link"},
	 /* 0x04 */ {OUT_D_CENTER, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Center"},
	 /* 0x05 */ {OUT_D_LFE, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Subwoofer"},
	 /* 0x06 */ {OUT_HEADPHONE, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Headphones"},
	 /* 0x08 */ {OUT_REAR, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Rear Speakers"},
	 /* 0x0A */ {OUT_ADC_REC, 0, STEREO, {0, 0, -2, -2}, {-1, -1, -1, -1, -1, -1, -1, -1}, "A/D converter (STEREO REC)"},
	 /* 0x0C */ {OUT_MIC_CAP, 0, MONO, {0, -1, -2, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "MIC (MONO REC)"},
	 /* 0x0D */ {0x0D, DEBUG_MIXER, MONO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x0D"},
	 /* 0x0E */ {0x0E, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x0E"},
	 /* 0x10 */ {0x10, DEBUG_MIXER, MONO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x10"},
	 /* 0x11 */ {OUT_A_CENTER, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Center"},
	 /* 0x12 */ {OUT_A_LFE, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Subwoofer"},
	 /* 0x13 */ {0x13, DEBUG_MIXER, MONO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x13"},
	 /* 0x14 */ {0x14, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x14"},
	 /* 0x16 */ {0x16, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x16"},
	 /* 0x18 */ {0x18, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x18"},
	 /* 0x1A */ {0x1A, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1A"},
	 /* 0x1C */ {0x1C, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1C"},
	 /* 0x1E */ {0x1E, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1E"}
};
#define NUM_K2_INPUTS	8
static struct emu_connector emu_k2_inputs[NUM_K2_INPUTS] = {
	 /* 0x0 */ {A_IN_AC97, NEED_AC97, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 Codec"},
	 /* 0x2 */ {A_IN_SPDIF_CD, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "CD S/PDIF"},
	 /* 0x4 */ {A_IN_O_SPDIF, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Optical S/PDIF"},
	 /* 0x6 */ {0x06, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x06"},
	 /* 0x8 */ {A_IN_LINE2, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Line-In (2)"},
	 /* 0xA */ {A_IN_R_SPDIF, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Coaxial S/PDIF"},
	 /* 0xC */ {A_IN_AUX2, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AUX2"},
	 /* 0xE */ {0x0E, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x0E"}
};
#define NUM_K2_OUTPUTS	18
static struct emu_connector emu_k2_outputs[NUM_K2_OUTPUTS] = {
	 /* 0x00 */ {A_OUT_D_FRONT, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Front"},
	 /* 0x02 */ {A_OUT_D_CENTER, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Center"},
	 /* 0x03 */ {A_OUT_D_LFE, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Subwoofer"},
	 /* 0x04 */ {A_OUT_HPHONE, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Headphones"},
	 /* 0x06 */ {A_OUT_D_REAR, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DIGITAL: Rear"},
	 /* 0x08 */ {A_OUT_A_FRONT, 0, STEREO, {100, 100, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Front"},
	 /* 0x0A */ {A_OUT_A_CENTER, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Center"},
	 /* 0x0B */ {A_OUT_A_LFE, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Subwoofer"},
	 /* 0x0C */ {A_OUT_D_SIDE, NEED_71, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Side"},
	 /* 0x0E */ {A_OUT_A_REAR, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Rear"},
	 /* 0x10 */ {A_OUT_AC97, NEED_AC97, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 ??"},
	 /* 0x12 */ {0x12, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x12"},
	 /* 0x14 */ {0x14, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x14"},
	 /* 0x16 */ {A_OUT_ADC_REC, 0, STEREO, {0, 0, -2, -2}, {-1, -1, -1, -1, -1, -1, -1, -1}, "A/D converter (REC)"},
	 /* 0x18 */ {0x18, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x18"},
	 /* 0x1A */ {0x1A, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1A"},
	 /* 0x1C */ {0x1C, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1C"},
	 /* 0x1E */ {0x1E, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1E"}
};
#define	NUM_K1_FX	1
static struct emu_connector emu_k1_fxbus[NUM_K1_FX] = {
	 /* 0x00 */ {FX_PCM + 0x0, 0, STEREO, {100, 100, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 0"}
#if 0
	 /* 0x00 */ {FX_PCM + 0x0, 0, STEREO, {100, 100, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 0"},
	 /* 0x02 */ {FX_PCM + 0x2, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 1"},
	 /* 0x04 */ {FX_PCM + 0x4, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 2"},
	 /* 0x06 */ {FX_PCM + 0x6, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 3"},
	 /* 0x08 */ {FX_PCM + 0x8, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 4"},
	 /* 0x0A */ {FX_PCM + 0xA, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 5"},
	 /* 0x0C */ {FX_PCM + 0xC, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 6"},
	 /* 0x0E */ {FX_PCM + 0xE, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 7"}
#endif
};
#define	NUM_K2_FX	1
/* on Audigy correct FX0 volume will be set in AC97 emulation code */
static struct emu_connector emu_k2_fxbus[NUM_K2_FX] = {
	 /* 0x00 */ {FX_PCM + 0x00, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 0"}
#if 0
	 /* 0x00 */ {FX_PCM + 0x00, 0, STEREO, {100, 100, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 0"},
	 /* 0x02 */ {FX_PCM + 0x02, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 1"},
	 /* 0x04 */ {FX_PCM + 0x04, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 2"},
	 /* 0x06 */ {FX_PCM + 0x06, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 3"},
	 /* 0x08 */ {FX_PCM + 0x08, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 4"},
	 /* 0x0A */ {FX_PCM + 0x0A, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 5"},
	 /* 0x0C */ {FX_PCM + 0x0C, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 6"},
	 /* 0x0E */ {FX_PCM + 0x0E, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 7"},
	 /* 0x10 */ {FX_PCM + 0x10, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 8"},
	 /* 0x12 */ {FX_PCM + 0x12, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 9"},
	 /* 0x14 */ {FX_PCM + 0x14, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 10"},
	 /* 0x16 */ {FX_PCM + 0x16, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 11"},
	 /* 0x18 */ {FX_PCM + 0x18, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 12"},
	 /* 0x1A */ {FX_PCM + 0x1A, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 13"},
	 /* 0x1C */ {FX_PCM + 0x1C, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 14"},
	 /* 0x1E */ {FX_PCM + 0x1E, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 15"}
#endif
};

#define NUM_A2V_INPUTS	8
static struct emu_connector emu_a2v_inputs[NUM_A2V_INPUTS] = {
	 /* 0x0 */ {A_IN_AC97, NEED_AC97, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 Codec"},
	 /* 0x2 */ {A_IN_SPDIF_CD, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "CD S/PDIF"},
	 /* 0x4 */ {A_IN_O_SPDIF, 0, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "S/PDIF"},
	 /* 0x6 */ {0x06, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x06"},
	 /* 0x8 */ {A_IN_LINE2, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Line-In (2)"},
	 /* 0xA */ {A_IN_R_SPDIF, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Coaxial S/PDIF"},
	 /* 0xC */ {A_IN_AUX2, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AUX2"},
	 /* 0xE */ {0x0E, DEBUG_MIXER, STEREO, {0, 0, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x0E"}
};
#define NUM_A2V_OUTPUTS	18
static struct emu_connector emu_a2v_outputs[NUM_A2V_OUTPUTS] = {
	 /* 0x00 */ {A_OUT_D_FRONT, 0, STEREO, {100, 100, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Front"},
	 /* 0x02 */ {A_OUT_D_CENTER, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Center"},
	 /* 0x03 */ {A_OUT_D_LFE, NEED_51, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Subwoofer"},
	 /* 0x04 */ {A_OUT_HPHONE, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Headphones"},
	 /* 0x06 */ {A_OUT_D_REAR, 0, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Rear"},
	 /* 0x08 */ {A_OUT_A_FRONT, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Analog Front"},
	 /* 0x0A */ {A_OUT_A_CENTER, DEBUG_MIXER, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Analog Center"},
	 /* 0x0B */ {A_OUT_A_LFE, DEBUG_MIXER, MONO, {0, -1, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Analog Subwoofer"},
	 /* 0x0C */ {A_OUT_D_SIDE, NEED_71, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Side"},
	 /* 0x0E */ {A_OUT_A_REAR, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "Analog Rear"},
	 /* 0x10 */ {A_OUT_AC97, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "AC97 ??"},
	 /* 0x12 */ {0x12, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x12"},
	 /* 0x14 */ {0x14, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x14"},
	 /* 0x16 */ {A_OUT_ADC_REC, 0, STEREO, {0, 0, -2, -2}, {-1, -1, -1, -1, -1, -1, -1, -1}, "A/D converter (REC)"},
	 /* 0x18 */ {0x18, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x18"},
	 /* 0x1A */ {0x1A, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1A"},
	 /* 0x1C */ {0x1C, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1C"},
	 /* 0x1E */ {0x1E, DEBUG_MIXER, STEREO, {0, 0, -1, -1}, {-1, -1, -1, -1, -1, -1, -1, -1}, "0x1E"}
};
#define	NUM_A2V_FX	1
static struct emu_connector emu_a2v_fxbus[NUM_A2V_FX] = {
	 /* 0x00 */ {FX_PCM + 0x0, 0, STEREO, {100, 100, 0, 0}, {-1, -1, -1, -1, -1, -1, -1, -1}, "DSP FX BUS 0"}
};

static struct emu_connector *
emumix_inputs(struct emu_sc_info *sc)
{
	struct emu_connector *conn;

	conn = NULL;
	if (sc->is_emu10k1) {
		conn = malloc(sizeof(emu_k1_inputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k1_inputs, sizeof(emu_k1_inputs));
	};
	if ((sc->is_emu10k2) || (sc->is_ca0151)) {
		conn = malloc(sizeof(emu_k2_inputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k2_inputs, sizeof(emu_k2_inputs));
	};
	if (sc->is_ca0108) {
		conn = malloc(sizeof(emu_a2v_inputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_a2v_inputs, sizeof(emu_a2v_inputs));
	};
	return conn;
};

static struct emu_connector *
emumix_outputs(struct emu_sc_info *sc)
{
	struct emu_connector *conn;

	conn = NULL;
	if (sc->is_emu10k1) {
		conn = malloc(sizeof(emu_k1_outputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k1_outputs, sizeof(emu_k1_outputs));
	};
	if ((sc->is_emu10k2) || (sc->is_ca0151)) {
		conn = malloc(sizeof(emu_k2_outputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k2_outputs, sizeof(emu_k2_outputs));
	};
	if (sc->is_ca0108) {
		conn = malloc(sizeof(emu_a2v_outputs), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_a2v_outputs, sizeof(emu_a2v_outputs));
	};
	return conn;
};

static struct emu_connector *
emumix_fxbuses(struct emu_sc_info *sc)
{
	struct emu_connector *conn;

	conn = NULL;
	if (sc->is_emu10k1) {
		conn = malloc(sizeof(emu_k1_fxbus), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k1_fxbus, sizeof(emu_k1_fxbus));
	};
	if ((sc->is_emu10k2) || (sc->is_ca0151)) {
		conn = malloc(sizeof(emu_k2_fxbus), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_k2_fxbus, sizeof(emu_k2_fxbus));
	}
	if (sc->is_ca0108) {
		conn = malloc(sizeof(emu_a2v_fxbus), M_DEVBUF, M_WAITOK);
		if (!conn)
			return NULL;
		conn = memcpy(conn, emu_a2v_fxbus, sizeof(emu_a2v_fxbus));
	}
	return conn;
};

int
emumix_init(struct emu_sc_info *sc)
{
	unsigned int i, j;
	int gprs;
	int tmp;

	sc->mixer = malloc(sizeof(struct emu_mixer), M_DEVBUF, M_NOWAIT | M_ZERO);
	if (sc->mixer == NULL)
		return ENOMEM;

	sc->mixer->inputs = emumix_inputs(sc);
	
	tmp = 0;
	tmp = sc->is_emu10k1 ? NUM_K1_INPUTS : 0;
	tmp = sc->is_emu10k2 ? NUM_K2_INPUTS : tmp;
	tmp = sc->is_ca0151 ? NUM_K2_INPUTS : tmp;
	tmp = sc->is_ca0108 ? NUM_A2V_INPUTS : tmp;
	sc->mixer->pmixercount[MIXER_INPUT] = tmp;
	sc->mixer->rmixercount[MIXER_INPUT] = tmp;
	/* allocate GPRs for inputs */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->inputs[i].channels_used);
		for (j = 0; j < sc->mixer->inputs[i].channels_used; j++) {
			sc->mixer->inputs[i].gpr[GPR_P_VOLUME + j] = gprs + j;
		};
	};
	/* recording vol */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->inputs[i].channels_used);
		for (j = 0; j < sc->mixer->inputs[i].channels_used; j++) {
			sc->mixer->inputs[i].gpr[GPR_R_VOLUME + j] = gprs + j;
		};
	};
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->inputs[i].channels_used);
		for (j = 0; j < sc->mixer->inputs[i].channels_used; j++) {
			sc->mixer->inputs[i].gpr[GPR_P_CACHE + j] = gprs + j;
		};
	};
	/* recording bus */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->inputs[i].channels_used);
		for (j = 0; j < sc->mixer->inputs[i].channels_used; j++) {
			sc->mixer->inputs[i].gpr[GPR_R_CACHE + j] = gprs + j;
		};
	};


	sc->mixer->outputs = emumix_outputs(sc);

	tmp = 0;
	tmp = sc->is_emu10k1 ? NUM_K1_OUTPUTS : 0;
	tmp = sc->is_emu10k2 ? NUM_K2_OUTPUTS : tmp;
	tmp = sc->is_ca0151 ? NUM_K2_OUTPUTS : tmp;
	tmp = sc->is_ca0108 ? NUM_A2V_OUTPUTS : tmp;
	sc->mixer->pmixercount[MIXER_OUTPUT] = tmp;
	sc->mixer->rmixercount[MIXER_OUTPUT] = 0;
	/* allocate GPRs for out */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->outputs[i].channels_used);
		for (j = 0; j < sc->mixer->outputs[i].channels_used; j++) {
			sc->mixer->outputs[i].gpr[GPR_P_VOLUME + j] = gprs + j;
		};
	};
	/* no recording vol */
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->outputs[i].channels_used);
		for (j = 0; j < sc->mixer->outputs[i].channels_used; j++) {
			sc->mixer->outputs[i].gpr[GPR_P_CACHE + j] = gprs + j;
		};
	};
	/* no recording bus */

	sc->mixer->fxbuses = emumix_fxbuses(sc);

	tmp = 0;
	tmp = sc->is_emu10k1 ? NUM_K1_FX : 0;
	tmp = sc->is_emu10k2 ? NUM_K2_FX : tmp;
	tmp = sc->is_ca0151 ? NUM_K2_FX : tmp;
	tmp = sc->is_ca0108 ? NUM_A2V_FX : tmp;
	sc->mixer->pmixercount[MIXER_FX] = tmp;
	sc->mixer->rmixercount[MIXER_FX] = tmp;
	/* allocate GPRs for fxbus */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->fxbuses[i].channels_used);
		for (j = 0; j < sc->mixer->fxbuses[i].channels_used; j++) {
			sc->mixer->fxbuses[i].gpr[GPR_P_VOLUME + j] = gprs + j;
		};
	};
	/* recording vol */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->fxbuses[i].channels_used);
		for (j = 0; j < sc->mixer->fxbuses[i].channels_used; j++) {
			sc->mixer->fxbuses[i].gpr[GPR_R_VOLUME + j] = gprs + j;
		};
	};
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->fxbuses[i].channels_used);
		for (j = 0; j < sc->mixer->fxbuses[i].channels_used; j++) {
			sc->mixer->fxbuses[i].gpr[GPR_P_CACHE + j] = gprs + j;
		};
	};
	/* recording bus */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		gprs = emu_rm_gpr_alloc(sc->rm, sc->mixer->fxbuses[i].channels_used);
		for (j = 0; j < sc->mixer->fxbuses[i].channels_used; j++) {
			sc->mixer->fxbuses[i].gpr[GPR_R_CACHE + j] = gprs + j;
		};
	};

	sc->mixer->pmixercount[MIXER_AC97] = 0;	/* use standart mixer for now */
	sc->mixer->rmixercount[MIXER_AC97] = 0;	/* use standart mixer for now */

	/* main mix */
	/* Non-Audigy cards does not have control over MIXER_P and MIXER_R
	 * volume because on this cards FreeBSD mixer direcly connected to
	 * AC97 chip and this two volume controls are not available through
	 * emuctrl. On this cards set this controls to 100%. On Audigy cards
	 * this controls are available from FreeBSD mixer as 'main' and
	 * 'reclev'. */

	/* playback mix */
	gprs = emu_rm_gpr_alloc(sc->rm, 2);
	sc->mixer->mix.channels_used = 2;
	for (j = 0; j < 2; j++) {
		sc->mixer->mix.gpr[GPR_P_CACHE + j] = gprs + j;
	};
	/* recording mix */
	gprs = emu_rm_gpr_alloc(sc->rm, 2);
	for (j = 0; j < 2; j++) {
		sc->mixer->mix.gpr[GPR_R_CACHE + j] = gprs + j;
	};
	/* playback mix vol */
	/* on Audigy correct volume will be set from AC97 emulation code */
	gprs = emu_rm_gpr_alloc(sc->rm, 2);
	for (j = 0; j < 2; j++) {
		sc->mixer->mix.gpr[GPR_P_VOLUME + j] = gprs + j;
		sc->mixer->mix.volume[MIXER_P_VOLUME + j] = (sc->is_emu10k1) ? 100 : 0;
	};
	/* recording mix vol */
	gprs = emu_rm_gpr_alloc(sc->rm, 2);
	for (j = 0; j < 2; j++) {
		sc->mixer->mix.gpr[GPR_R_VOLUME + j] = gprs + j;
		sc->mixer->mix.volume[MIXER_R_VOLUME + j] = (sc->is_emu10k1) ? 100 : 0;
	};

	sc->mixer->monomix.gpr[GPR_P_CACHE] = emu_rm_gpr_alloc(sc->rm, 1);
	sc->mixer->monomix.gpr[GPR_P_VOLUME] = emu_rm_gpr_alloc(sc->rm, 1);

	sc->mixer->amp.gpr[GPR_P_VOLUME] = emu_rm_gpr_alloc(sc->rm, 1);
	sc->mixer->amp.volume[MIXER_P_VOLUME] = 1;

	/* AC3 passthrough */

	if (sc->is_emu10k2) {
		sc->mixer->pt_control = emu_rm_gpr_alloc(sc->rm, 1);
		sc->mixer->pt_gprs = emu_rm_gpr_alloc(sc->rm, 2);
		sc->mixer->pt_tgprs = emu_rm_gpr_alloc(sc->rm, 2);
		sc->mixer->pt_ogprs = emu_rm_gpr_alloc(sc->rm, 2);
	};

	emumix_reset(sc);
	return 0;
};

void
emumix_reset(struct emu_sc_info *sc)
{
	struct emu_connector *conn;
	struct emu_mixer *mix;
	unsigned int i, j;

	mix = sc->mixer;

	for (i = 0; i < sc->num_gprs; i++)
		emumix_set_gpr(sc, i, 0);

	/* playback vol & cache */
	for (i = 0; i < mix->pmixercount[MIXER_INPUT]; i++) {
		conn = &(mix->inputs[i]);
		for (j = 0; j < conn->channels_used; j++) {
			emumix_set_fxvol(sc, conn->gpr[GPR_P_VOLUME + j], conn->volume[MIXER_P_VOLUME + j]);
			emumix_set_gpr(sc, conn->gpr[GPR_P_CACHE + j], 0);
		};
	};
	/* recording vol & cache */
	for (i = 0; i < mix->rmixercount[MIXER_INPUT]; i++) {
		conn = &(mix->inputs[i]);
		for (j = 0; j < conn->channels_used; j++) {
			emumix_set_fxvol(sc, conn->gpr[GPR_R_VOLUME + j], conn->volume[MIXER_R_VOLUME + j]);
			emumix_set_gpr(sc, conn->gpr[GPR_R_CACHE + j], 0);
		};
	};

	/* allocate GPRs for out */
	/* playback vol & cache */
	for (i = 0; i < mix->pmixercount[MIXER_OUTPUT]; i++) {
		conn = &(mix->outputs[i]);
		for (j = 0; j < conn->channels_used; j++) {
			emumix_set_fxvol(sc, conn->gpr[GPR_P_VOLUME + j], conn->volume[MIXER_P_VOLUME + j]);
			emumix_set_gpr(sc, conn->gpr[GPR_P_CACHE + j], 0);
		};
	};
	/* no recording vol &cache */

	/* allocate GPRs for fxbus */
	/* playback vol */
	for (i = 0; i < mix->pmixercount[MIXER_FX]; i++) {
		conn = &(mix->fxbuses[i]);
		for (j = 0; j < conn->channels_used; j++) {
			emumix_set_fxvol(sc, conn->gpr[GPR_P_VOLUME + j], conn->volume[MIXER_P_VOLUME + j]);
			emumix_set_gpr(sc, conn->gpr[GPR_P_CACHE + j], 0);
		};
	};
	/* recording vol */
	for (i = 0; i < mix->rmixercount[MIXER_FX]; i++) {
		conn = &(mix->fxbuses[i]);
		for (j = 0; j < conn->channels_used; j++) {
			emumix_set_fxvol(sc, conn->gpr[GPR_R_VOLUME + j], conn->volume[MIXER_R_VOLUME + j]);
			emumix_set_gpr(sc, conn->gpr[GPR_R_CACHE + j], 0);
		};
	};
	/* main mix */

	/* playback mix */
	conn = &(mix->mix);
	for (j = 0; j < conn->channels_used; j++) {
		emumix_set_fxvol(sc, conn->gpr[GPR_P_VOLUME + j], conn->volume[MIXER_P_VOLUME + j]);
		emumix_set_gpr(sc, conn->gpr[GPR_P_CACHE + j], 0);
	};
	for (j = 0; j < conn->channels_used; j++) {
		emumix_set_fxvol(sc, conn->gpr[GPR_R_VOLUME + j], conn->volume[MIXER_R_VOLUME + j]);
		emumix_set_gpr(sc, conn->gpr[GPR_R_CACHE + j], 0);
	};

	emumix_set_gpr(sc, mix->monomix.gpr[GPR_P_VOLUME], 0);
	emumix_set_gpr(sc, mix->monomix.gpr[GPR_P_CACHE], 0);

	/* not a bug: volume is a value 1..16 used to amplify signal */
	emumix_set_gpr(sc, mix->amp.gpr[GPR_P_VOLUME], mix->amp.volume[MIXER_P_VOLUME]);

	/* AC3 passthrough */

	if (sc->is_emu10k2) {
		emumix_set_gpr(sc, mix->pt_control, 0);

		emumix_set_gpr(sc, (mix->pt_gprs) + 0, 0x00001008);	/* for <=0 condifional
									 * skip */
		emumix_set_gpr(sc, (mix->pt_gprs) + 1, 0xffff0000);

		emumix_set_gpr(sc, (mix->pt_tgprs) + 0, 0);
		emumix_set_gpr(sc, (mix->pt_tgprs) + 1, 0);

		emumix_set_gpr(sc, (mix->pt_ogprs) + 0, 0);
		emumix_set_gpr(sc, (mix->pt_ogprs) + 1, 0);
	};
};

int
emumix_uninit(struct emu_sc_info *sc)
{
	unsigned int i;

	/* deallocate GPRs for inputs */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->inputs[i].gpr[GPR_P_VOLUME], sc->mixer->inputs[i].channels_used);
	};

	/* recording vol */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->inputs[i].gpr[GPR_R_VOLUME], sc->mixer->inputs[i].channels_used);
	};
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_INPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->inputs[i].gpr[GPR_P_CACHE], sc->mixer->inputs[i].channels_used);
	};
	/* recording bus */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_INPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->inputs[i].gpr[GPR_R_CACHE], sc->mixer->inputs[i].channels_used);
	};


	/* deallocate GPRs for out */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->outputs[i].gpr[GPR_P_VOLUME], sc->mixer->outputs[i].channels_used);
	};
	/* no recording vol */
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_OUTPUT]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->outputs[i].gpr[GPR_P_CACHE], sc->mixer->outputs[i].channels_used);
	};
	/* no recording bus */

	/* deallocate GPRs for fxbus */
	/* playback vol */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->fxbuses[i].gpr[GPR_P_VOLUME], sc->mixer->fxbuses[i].channels_used);
	};
	/* recording vol */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->fxbuses[i].gpr[GPR_R_VOLUME], sc->mixer->fxbuses[i].channels_used);
	};
	/* playback bus */
	for (i = 0; i < sc->mixer->pmixercount[MIXER_FX]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->fxbuses[i].gpr[GPR_P_CACHE], sc->mixer->fxbuses[i].channels_used);
	};
	/* recording bus */
	for (i = 0; i < sc->mixer->rmixercount[MIXER_FX]; i++) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->fxbuses[i].gpr[GPR_R_CACHE], sc->mixer->fxbuses[i].channels_used);
	};

	/* no ac97 */

	/* main mix */
	/* playback mix */
	(void)emu_rm_gpr_free(sc->rm, sc->mixer->mix.gpr[GPR_P_CACHE], 2);
	/* recording mix */
	(void)emu_rm_gpr_free(sc->rm, sc->mixer->mix.gpr[GPR_R_CACHE], 2);
	/* playback mix vol */
	(void)emu_rm_gpr_free(sc->rm, sc->mixer->mix.gpr[GPR_P_VOLUME], 2);
	/* recording mix vol */
	(void)emu_rm_gpr_free(sc->rm, sc->mixer->mix.gpr[GPR_R_VOLUME], 2);

	(void)emu_rm_gpr_free(sc->rm, sc->mixer->monomix.gpr[GPR_P_CACHE], 1);
	(void)emu_rm_gpr_free(sc->rm, sc->mixer->monomix.gpr[GPR_P_VOLUME], 1);

	(void)emu_rm_gpr_free(sc->rm, sc->mixer->amp.gpr[GPR_P_VOLUME], 1);


	/* ac3 passthrough */
	if (sc->is_emu10k2) {
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->pt_control, 1);
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->pt_gprs, 2);
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->pt_tgprs, 2);
		(void)emu_rm_gpr_free(sc->rm, sc->mixer->pt_ogprs, 2);
	};

	if (sc->mixer != NULL)
		free(sc->mixer, M_DEVBUF);

	return 0;
};
