/*
 * Copyright (c) 2003 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHERIN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx-rm.c,v 1.8 2005/10/09 13:06:48 chibis Exp $
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>
#include <sys/malloc.h>

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif


#include "emu10kx.h"
#include "emu10kx-rm.h"

int 
emu_rm_init(struct emu_sc_info *sc)
{
	int i;
	int maxcount;
	struct emu_rm *rm;

	rm = malloc(sizeof(struct emu_rm), M_DEVBUF, M_NOWAIT | M_ZERO);
	if (rm == NULL) {
		return ENOMEM;
	}
	sc->rm = rm;
	rm->card = sc;
	maxcount = sc->num_gprs;
	rm->num_used = 0;
	mtx_init(&(rm->gpr_lock), "emu10k", "gpr alloc", MTX_DEF);
	rm->num_gprs = (maxcount < EMU_MAX_GPR ? maxcount : EMU_MAX_GPR);
	for (i = 0; i < rm->num_gprs; i++)
		rm->allocmap[i] = 0;
	rm->last_free_gpr = 0;

	return 0;
};

int 
emu_rm_uninit(struct emu_sc_info *sc)
{
	int i;

	mtx_lock(&(sc->rm->gpr_lock));
	for (i = 0; i < sc->rm->last_free_gpr; i++)
		if (sc->rm->allocmap[i] > 0)
			device_printf(sc->dev, "rm: gpr %d not free before uninit\n", i);
	mtx_unlock(&(sc->rm->gpr_lock));

	mtx_destroy(&(sc->rm->gpr_lock));
	free(sc->rm, M_DEVBUF);
	return 0;
};

int 
emu_rm_gpr_alloc(struct emu_rm *rm, int count)
{
	int i, j;
	int allocated_gpr;

	allocated_gpr = rm->num_gprs;
	/* try fast way first */
	mtx_lock(&(rm->gpr_lock));
	if (rm->last_free_gpr + count <= rm->num_gprs) {
		allocated_gpr = rm->last_free_gpr;
		rm->last_free_gpr += count;
		rm->allocmap[allocated_gpr] = count;
		for (i = 1; i < count; i++)
			rm->allocmap[allocated_gpr + i] = -(count - i);
	} else {
		/* longer */
		i = 0;
		allocated_gpr = rm->num_gprs;
		while (i < rm->last_free_gpr - count) {
			if (rm->allocmap[i] > 0) {
				i += rm->allocmap[i];
			} else {
				allocated_gpr = i;
				for (j = 1; j < count; j++) {
					if (rm->allocmap[i + j] != 0)
						allocated_gpr = rm->num_gprs;
				}
				if (allocated_gpr == i)
					break;
			}
		}
		if (allocated_gpr + count < rm->last_free_gpr) {
			rm->allocmap[allocated_gpr] = count;
			for (i = 1; i < count; i++)
				rm->allocmap[allocated_gpr + i] = -(count - i);

		};
	}
	if (allocated_gpr == rm->num_gprs)
		allocated_gpr = (-1);
	if (allocated_gpr >= 0)
		rm->num_used += count;
	mtx_unlock(&(rm->gpr_lock));
	return allocated_gpr;
};

int 
emu_rm_gpr_free(struct emu_rm *rm, int gpr, int count)
{
	int i;

	mtx_lock(&(rm->gpr_lock));
	if (rm->allocmap[gpr] == 0) {
		device_printf(rm->card->dev, "rm: already free gpr %04x\n", gpr);

	};
	if (rm->allocmap[gpr] < 0) {
		i = gpr - 1;
		while ((rm->allocmap[i] < 0) && (i >= 0))
			i--;
		device_printf(rm->card->dev, "rm: gpr %04x is a part of another gpr block %04x length %d\n", gpr, i, rm->allocmap[i]);
	};
	if (rm->allocmap[gpr] == count) {
		for (i = 0; i < count; i++)
			rm->allocmap[gpr + i] = 0;
		if (gpr + count == rm->last_free_gpr)
			rm->last_free_gpr = gpr;
		rm->num_used -= count;
	};
	mtx_unlock(&(rm->gpr_lock));
	return 0;
};
