/*
 * Copyright (c) 1999 Cameron Grant <cg@freebsd.org>
 * Copyright (c) 2003,2004 Yuriy Tsibizov <yuriy.tsibizov@gfk.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHERIN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: emu10kx.c,v 1.94 2005/10/17 03:56:07 chibis Exp $
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <sys/rman.h>
#include <sys/systm.h>
#include <sys/sbuf.h>
#include <sys/queue.h>

#if __FreeBSD_version < 500000
#include <pci/pcireg.h>
#include <pci/pcivar.h>
#else
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>
#endif

#include <sys/lock.h>
#if __FreeBSD_version >= 500000
#include <sys/mutex.h>
#else
#define mtx_init(LOCK, DEVDESC, LOCKDESC, MTX_TYPE)
#define mtx_lock(LOCK)
#define mtx_unlock(LOCK)
#define mtx_destroy(LOCK)
#endif

#include <machine/clock.h>	/* for DELAY */

#include <dev/sound/chip.h>
#include <dev/sound/pcm/sound.h>

#include "emu10kx.h"
#include "emu10kx-dev.h"
#include "emu10kx-rm.h"
#include "emu10kx-ir.h"
#include "alsa/p16v.h"

/* Supported cards */
static struct emu_hwinfo emu_cards[] = {
	 /* 0x0020..0x002f 4.0 EMU10K1 cards */
	{0x1102, 0x0002, 0x1102, 0x0020, "CT4850", "SBLive! Value", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x0021, "CT4620", "SBLive!", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x002f, "CT????", "SBLive! mainboard implementation", HAS_AC97 | IS_EMU10K1},

	/* (range unknown) 5.1 EMU10K1 cards */
	{0x1102, 0x0002, 0x1102, 0x100a, "CT????", "SBLive! 5.1", HAS_AC97 | HAS_51 | IS_EMU10K1},

	/* 0x80??..0x805? 4.0 EMU10K1 cards */
	{0x1102, 0x0002, 0x1102, 0x8022, "CT4780", "SBLive! Value", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8023, "CT4790", "SB PCI512", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8024, "CT4760", "SBLive!", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8025, "CT????", "SBLive! Mainboard Implementation", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8026, "CT4830", "SBLive! Value", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8027, "CT4832", "SBLive! Value", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8028, "CT4760", "SBLive! OEM version", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8031, "CT4831", "SBLive! Value", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8040, "CT4760", "SBLive!", HAS_AC97 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8051, "CT4850", "SBLive! Value", HAS_AC97 | IS_EMU10K1},

	/* 0x8061..0x???? 5.1 EMU10K1  cards */
	{0x1102, 0x0002, 0x1102, 0x8061, "SB????", "SBLive! Player 5.1", HAS_AC97 | HAS_51 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8064, "SB0100", "SBLive! 5.1", HAS_AC97 | HAS_51 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8065, "SB0220", "SBLive! 5.1 Digital", HAS_AC97 | HAS_51 | IS_EMU10K1},
	{0x1102, 0x0002, 0x1102, 0x8067, "SB????", "SBLive!", HAS_AC97 | HAS_51 | IS_EMU10K1},

	/* Generic SB Live! */
	{0x1102, 0x0002, 0x1102, 0x0000, "SB????", "SBLive! (Unknown model)", HAS_AC97 | IS_EMU10K1},

	/* 0x0041..0x0043 EMU10K2 (some kind of Audigy) cards */

	/* 0x0051..0x005C 5.1 EMU10K2 cards */
	{0x1102, 0x0004, 0x1102, 0x0051, "SB0090", "Audigy", HAS_AC97 | HAS_51 | IS_EMU10K2},
	{0x1102, 0x0004, 0x1102, 0x0052, "SB????", "Audigy ES", HAS_AC97 | HAS_71 | IS_EMU10K2 | BROKEN_DIGITAL},
	{0x1102, 0x0004, 0x1102, 0x0053, "SB0090", "Audigy Player/OEM", HAS_AC97 | HAS_51 | IS_EMU10K2},
	{0x1102, 0x0004, 0x1102, 0x0058, "SB0090", "Audigy Player/OEM", HAS_AC97 | HAS_51 | IS_EMU10K2},

	/* 0x1002..0x1009 5.1 EMU10K2+P16V (CA0151) cards */
	{0x1102, 0x0004, 0x1102, 0x1002, "SB????", "Audigy 2 Platinum", HAS_51 | IS_CA0151},
	{0x1102, 0x0004, 0x1102, 0x1005, "SB????", "Audigy 2 EX", HAS_51 | IS_CA0151},
	{0x1102, 0x0004, 0x1102, 0x1007, "SB0240", "Audigy 2 6.1", HAS_51 | IS_CA0151},

	/* 0x2001..0x2003 7.1 EMU10K2+P16V (CA0151) cards */
	{0x1102, 0x0004, 0x1102, 0x2001, "SB0350", "Audigy 2 ZS", HAS_AC97 | HAS_71 | IS_CA0151},
	{0x1102, 0x0004, 0x1102, 0x2002, "SB0350", "Audigy 2 ZS", HAS_AC97 | HAS_71 | IS_CA0151},

	/* (range unknown) 7.1 EMU10K2+P16V (CA0151) Audigy 4 cards */
	{0x1102, 0x0004, 0x1102, 0x2007, "SB0380", "Audigy 4 Pro", HAS_AC97 | HAS_71 | IS_CA0151},

	 /* Generic Audigy or Audigy 2 */
	{0x1102, 0x0004, 0x1102, 0x0000, "SB????", "Audigy (Unknown model)", HAS_AC97 | HAS_51 | IS_EMU10K2},

	 /* 0x1000..0x1001 EMU10K2+P17V (CA0108) cards */
	{0x1102, 0x0008, 0x1102, 0x1000, "SB????", "Audigy 2 LS", HAS_AC97 | HAS_51 | IS_CA0108 | DIGITAL_ONLY},
	{0x1102, 0x0008, 0x1102, 0x1001, "SB0040", "Audigy 2 Value", HAS_AC97 | HAS_71 | IS_CA0108 | DIGITAL_ONLY},
	{0x1102, 0x0008, 0x0000, 0x0000, "SB????", "Audigy 2 Value (Unknown model)", HAS_AC97 | HAS_51 | IS_CA0108},
};
/* Unsupported cards */

static struct emu_hwinfo emu_bad_cards[] = {
	/* APS cards should be possible to support */
	{0x1102, 0x0002, 0x1102, 0x4001, "EMUAPS", "E-mu APS", 0},
	{0x1102, 0x0002, 0x1102, 0x4002, "EMUAPS", "E-mu APS", 0},
	/* Similar-named ("Live!" or "Audigy") cards on different chipsets */
	{0x1102, 0x8064, 0x0000, 0x0000, "SB0100", "SBLive! 5.1 OEM", 0},
	{0x1102, 0x0006, 0x0000, 0x0000, "SB????", "DELL OEM SBLive! Value", 0},
	{0x1102, 0x0007, 0x0000, 0x0000, "SB????", "Audigy LS", 0},
};

static int emu_getcard(device_t dev);

/* -------------------------------------------------------------------- */
/* Hardware */
static uint32_t
emu_rd_nolock(struct emu_sc_info *sc, unsigned int regno, unsigned int size)
{
	KASSERT(sc != NULL, ("emu_rd: NULL sc"));
	switch (size) {
	case 1:
		return bus_space_read_1(sc->st, sc->sh, regno);
	case 2:
		return bus_space_read_2(sc->st, sc->sh, regno);
	case 4:
		return bus_space_read_4(sc->st, sc->sh, regno);
	}
	return 0xffffffff;
}

static void
emu_wr_nolock(struct emu_sc_info *sc, unsigned int regno, uint32_t data, unsigned int size)
{
	KASSERT(sc != NULL, ("emu_rd: NULL sc"));
	switch (size) {
	case 1:
		bus_space_write_1(sc->st, sc->sh, regno, data);
		break;
	case 2:
		bus_space_write_2(sc->st, sc->sh, regno, data);
		break;
	case 4:
		bus_space_write_4(sc->st, sc->sh, regno, data);
		break;
	}
	return;
}
uint32_t
emu_rdptr(struct emu_sc_info *sc, unsigned int chn, unsigned int reg)
{
	uint32_t ptr, val, mask, size, offset;

	ptr = ((reg << 16) & sc->address_mask) | (chn & PTR_CHANNELNUM_MASK);
	mtx_lock(&sc->rw);
	emu_wr_nolock(sc, PTR, ptr, 4);
	val = emu_rd_nolock(sc, DATA, 4);
	mtx_unlock(&sc->rw);
	if (reg & 0xff000000) {
		size = (reg >> 24) & 0x3f;
		offset = (reg >> 16) & 0x1f;
		mask = ((1 << size) - 1) << offset;
		val &= mask;
		val >>= offset;
	}
	return val;
}

void
emu_wrptr(struct emu_sc_info *sc, unsigned int chn, unsigned int reg, uint32_t data)
{
	uint32_t ptr, mask, size, offset;
	ptr = ((reg << 16) & sc->address_mask) | (chn & PTR_CHANNELNUM_MASK);
	mtx_lock(&sc->rw);
	emu_wr_nolock(sc, PTR, ptr, 4);
	if (reg & 0xff000000) {
		size = (reg >> 24) & 0x3f;
		offset = (reg >> 16) & 0x1f;
		mask = ((1 << size) - 1) << offset;
		data <<= offset;
		data &= mask;
		data |= emu_rd_nolock(sc, DATA, 4) & ~mask;
	}
	emu_wr_nolock(sc, DATA, data, 4);
	mtx_unlock(&sc->rw);
	return;
}

uint32_t
emu_rd_p16vptr(struct emu_sc_info *sc, uint16_t chn, uint16_t reg)
{
	uint32_t val;

	mtx_lock(&sc->rw);
	emu_wr_nolock(sc, PTR2, (reg << 16) | chn, 4);
	val = emu_rd_nolock(sc, DATA2, 4);
	mtx_unlock(&sc->rw);
	return val;
}

void
emu_wr_p16vptr(struct emu_sc_info *sc, uint16_t chn, uint16_t reg, uint32_t data)
{

	mtx_lock(&sc->rw);
	emu_wr_nolock(sc, PTR2, (reg << 16) | chn, 4);
	emu_wr_nolock(sc, DATA2, data, 4);
	mtx_unlock(&sc->rw);
	return;
}

void
emu_wr(struct emu_sc_info *sc, unsigned int regno, uint32_t data, unsigned int size)
{

	mtx_lock(&sc->rw);
	emu_wr_nolock(sc, regno, data, size);
	mtx_unlock(&sc->rw);
	return;
}
uint32_t
emu_rd(struct emu_sc_info *sc, unsigned int regno, unsigned int size)
{
	uint32_t rd;

	mtx_lock(&sc->rw);
	rd = emu_rd_nolock(sc, regno, size);
	mtx_unlock(&sc->rw);
	return rd;
}


void
emu_enable_ir(struct emu_sc_info *sc)
{
	uint32_t iocfg;

	mtx_lock(&sc->rw);
	/* enable IR MIDI */
	if (sc->is_emu10k2) {
		iocfg = emu_rd_nolock(sc, A_IOCFG, 2);
		emu_wr_nolock(sc, A_IOCFG, iocfg | A_IOCFG_GPOUT2, 2);
		DELAY(500);
		emu_wr_nolock(sc, A_IOCFG, iocfg | A_IOCFG_GPOUT1 | A_IOCFG_GPOUT2, 2);
		DELAY(500);
		emu_wr_nolock(sc, A_IOCFG, iocfg | A_IOCFG_GPOUT1, 2);
		DELAY(100);
		emu_wr_nolock(sc, A_IOCFG, iocfg, 2);
	};
	if (sc->is_emu10k1) {
		iocfg = emu_rd_nolock(sc, HCFG, 4);
		emu_wr_nolock(sc, HCFG, iocfg | HCFG_GPOUT2, 4);
		DELAY(500);
		emu_wr_nolock(sc, HCFG, iocfg | HCFG_GPOUT1 | HCFG_GPOUT2, 4);
		DELAY(100);
		emu_wr_nolock(sc, HCFG, iocfg, 4);
	};
	sc->enable_ir = 1;
	mtx_unlock(&sc->rw);
	device_printf(sc->dev, "IR MIDI events enabled.\n");
	return;
}
/* hw ends */

static devclass_t emu_devclass;

 /* Interrupt manager */
int
emu_enatimer(struct emu_sc_info *sc, int go)
{
	uint32_t x;
	KASSERT(sc != NULL, ("soundcard is null!"));
	mtx_lock(&sc->lock);
	if (go) {
		if (sc->timer++ == 0) {
			x = emu_rd(sc, INTE, 4);
			x |= INTE_INTERVALTIMERENB;
			emu_wr(sc, INTE, x, 4);
		}
	} else {
		sc->timer = 0;
		x = emu_rd(sc, INTE, 4);
		x &= ~INTE_INTERVALTIMERENB;
		emu_wr(sc, INTE, x, 4);
	}
	mtx_unlock(&sc->lock);
	return 0;
}

int
emu_intr_register(struct emu_sc_info *sc, uint32_t inte_mask, uint32_t intr_mask, uint32_t(*func) (void *softc, uint32_t irq), void *isc){
	int i;
	uint32_t x;

	mtx_lock(&sc->lock);
	for (i = 0; i < EMU_MAX_IRQ_CONSUMERS; i++) {
		if (sc->ihandler[i].inte_mask == 0) {
			sc->ihandler[i].inte_mask = inte_mask;
			sc->ihandler[i].intr_mask = intr_mask;
			sc->ihandler[i].softc = isc;
			sc->ihandler[i].irq_func = func;
			x = emu_rd(sc, INTE, 4);
			x |= inte_mask;
			emu_wr(sc, INTE, x, 4);
			mtx_unlock(&sc->lock);
			return i;
		}
	}
	mtx_unlock(&sc->lock);
	return -1;
}

int
emu_intr_unregister(struct emu_sc_info *sc, int hnumber)
{
	int i = hnumber;
	uint32_t x;
	mtx_lock(&sc->lock);
	if (sc->ihandler[i].inte_mask != 0) {
		x = emu_rd(sc, INTE, 4);
		x &= ~sc->ihandler[i].inte_mask;
		emu_wr(sc, INTE, x, 4);

		sc->ihandler[i].inte_mask = 0;
		sc->ihandler[i].intr_mask = 0;
		sc->ihandler[i].softc = NULL;
		sc->ihandler[i].irq_func = NULL;

		mtx_unlock(&sc->lock);
		return hnumber;
	}
	mtx_unlock(&sc->lock);
	return -1;
}
/* The interrupt handler */
void
emu_intr(void *p)
{
	struct emu_sc_info *sc = (struct emu_sc_info *)p;
	uint32_t stat, ack;
	int i;

	mtx_lock(&sc->lock);
	while (1) {
		stat = emu_rd(sc, IPR, 4);
		ack = 0;
		if (stat == 0)
			break;
		emu_wr(sc, IPR, stat, 4);
		mtx_unlock(&sc->lock);
		for (i = 0; i < EMU_MAX_IRQ_CONSUMERS; i++) {
			if ((((sc->ihandler[i].intr_mask) & stat) != 0) &&
			    (((void *)sc->ihandler[i].irq_func) != NULL)) {
				ack |= sc->ihandler[i].irq_func(sc->ihandler[i].softc, stat);
			}
		};
		mtx_lock(&sc->lock);
	};
	mtx_unlock(&sc->lock);
}


/* Probe and attach the card */
int
emu_init(struct emu_sc_info *sc)
{
	uint32_t ch, tmp;
	uint32_t spdif_sr;
	uint32_t ac97slot;
	int def_mode;

	/* disable audio and lock cache */
	emu_wr(sc, HCFG, HCFG_LOCKSOUNDCACHE | HCFG_LOCKTANKCACHE_MASK | HCFG_MUTEBUTTONENABLE, 4);

	/* reset recording buffers */
	emu_wrptr(sc, 0, MICBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, MICBA, 0);
	emu_wrptr(sc, 0, FXBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, FXBA, 0);
	emu_wrptr(sc, 0, ADCBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, ADCBA, 0);

	/* disable channel interrupt */
	emu_wr(sc, INTE, INTE_INTERVALTIMERENB | INTE_SAMPLERATETRACKER | INTE_PCIERRORENABLE, 4);
	emu_wrptr(sc, 0, CLIEL, 0);
	emu_wrptr(sc, 0, CLIEH, 0);
	emu_wrptr(sc, 0, SOLEL, 0);
	emu_wrptr(sc, 0, SOLEH, 0);

	/* init phys inputs and outputs */
	ac97slot = 0;
	if (sc->has_51)
		ac97slot = AC97SLOT_CNTR | AC97SLOT_LFE;
	if (sc->has_71)
		ac97slot = AC97SLOT_CNTR | AC97SLOT_LFE | AC97SLOT_REARLEFT | AC97SLOT_REARRIGHT;
	if (sc->is_emu10k2)
		ac97slot |= 0x40;
	emu_wrptr(sc, 0, AC97SLOT, ac97slot);

	if (sc->is_emu10k2) /* XXX for later cards? */
		emu_wrptr(sc, 0, SPBYPASS, 0xf00);	/* What will happen if
							 * we write 1 here? */

	/* init envelope engine */
	for (ch = 0; ch < NUM_G; ch++) {
		emu_wrptr(sc, ch, DCYSUSV, 0);
		emu_wrptr(sc, ch, IP, 0);
		emu_wrptr(sc, ch, VTFT, 0xffff);
		emu_wrptr(sc, ch, CVCF, 0xffff);
		emu_wrptr(sc, ch, PTRX, 0);
		emu_wrptr(sc, ch, CPF, 0);
		emu_wrptr(sc, ch, CCR, 0);

		emu_wrptr(sc, ch, PSST, 0);
		emu_wrptr(sc, ch, DSL, 0x10);
		emu_wrptr(sc, ch, CCCA, 0);
		emu_wrptr(sc, ch, Z1, 0);
		emu_wrptr(sc, ch, Z2, 0);
		emu_wrptr(sc, ch, FXRT, 0xd01c0000);

		emu_wrptr(sc, ch, ATKHLDM, 0);
		emu_wrptr(sc, ch, DCYSUSM, 0);
		emu_wrptr(sc, ch, IFATN, 0xffff);
		emu_wrptr(sc, ch, PEFE, 0);
		emu_wrptr(sc, ch, FMMOD, 0);
		emu_wrptr(sc, ch, TREMFRQ, 24);	/* 1 Hz */
		emu_wrptr(sc, ch, FM2FRQ2, 24);	/* 1 Hz */
		emu_wrptr(sc, ch, TEMPENV, 0);

		/*** these are last so OFF prevents writing ***/
		emu_wrptr(sc, ch, LFOVAL2, 0);
		emu_wrptr(sc, ch, LFOVAL1, 0);
		emu_wrptr(sc, ch, ATKHLDV, 0);
		emu_wrptr(sc, ch, ENVVOL, 0);
		emu_wrptr(sc, ch, ENVVAL, 0);

		if ((sc->is_emu10k2) || (sc->is_ca0108) || (sc->is_ca0151)) {
			emu_wrptr(sc, ch, 0x4c, 0x0);
			emu_wrptr(sc, ch, 0x4d, 0x0);
			emu_wrptr(sc, ch, 0x4e, 0x0);
			emu_wrptr(sc, ch, 0x4f, 0x0);
			emu_wrptr(sc, ch, A_FXRT1, 0x3f3f3f3f);
			emu_wrptr(sc, ch, A_FXRT2, 0x3f3f3f3f);
			emu_wrptr(sc, ch, A_SENDAMOUNTS, 0x0);
		}
	}

	emumix_set_spdif_mode(sc, SPDIF_MODE_PCM);

	if ((sc->is_emu10k2) || (sc->is_ca0108) || (sc->is_ca0151)) {
		emu_wrptr(sc, 0, A_SPDIF_SAMPLERATE, A_SPDIF_48000);
	};

	if (sc->is_ca0151) {

		/* Setup SRCMulti_I2S SamplingRate */
		spdif_sr = emu_rdptr(sc, 0, A_SPDIF_SAMPLERATE);
		spdif_sr &= 0xfffff1ff;
		spdif_sr |= (0x2 << 9);
		emu_wrptr(sc, 0, A_SPDIF_SAMPLERATE, spdif_sr);

		/* Setup SRCSel (Enable Spdif,I2S SRCMulti) */
		emu_wr_p16vptr(sc, 0, SRCSel, 0x14);

		/* Setup SRCMulti Input Audio Enable */
		emu_wr_p16vptr(sc, 0, SRCMULTI_ENABLE, 0xFF00FF00);
	};

	if (sc->is_ca0108) {

		/* Setup SRCMulti_I2S SamplingRate */
		spdif_sr = emu_rdptr(sc, 0, A_SPDIF_SAMPLERATE);
		spdif_sr &= 0xfffff1ff;
		spdif_sr |= (0x2 << 9);
		emu_wrptr(sc, 0, A_SPDIF_SAMPLERATE, spdif_sr);

		/* Setup SRCSel (Enable Spdif,I2S SRCMulti) */
		emu_wr_p16vptr(sc, 0, SRCSel, 0x14);

		/* Setup SRCMulti Input Audio Enable */
		emu_wr_p16vptr(sc, 0, V_SRCMULTI_ENABLE, 0xFF000000);

		/* Setup SPDIF Out Audio Enable */
		/* The Audigy 2 Value has a separate SPDIF out, so no need for
		 * a mixer switch */
		emu_wr_p16vptr(sc, 0, V_SRC48_ENABLE, 0xFF000000);

		tmp = emu_rd(sc, A_IOCFG, 2) & ~0x8;
		emu_wr(sc, A_IOCFG, tmp, 2);
	};

	emu_initefx(sc);

	def_mode = MODE_ANALOG;
	if ((sc->is_emu10k2) || (sc->is_ca0108) || (sc->is_ca0151))
		def_mode = MODE_DIGITAL;
	if (((sc->is_emu10k2) || (sc->is_ca0108) || (sc->is_ca0151)) && (sc->broken_digital)) {
		device_printf(sc->dev, "Audigy card initialized in analog mode.\n");
		def_mode = MODE_ANALOG;
	};
	emumix_set_mode(sc, def_mode);

	if (bootverbose) {
		tmp = emu_rd(sc, HCFG, 4);
		device_printf(sc->dev, "Card Configuration (   0x%08x )\n", tmp);
		device_printf(sc->dev, "Card Configuration ( & 0xff000000 ) : %s%s%s%s%s%s%s%s\n",
		    (tmp & 0x80000000 ? "[Legacy MPIC] " : ""),
		    (tmp & 0x40000000 ? "[0x40] " : ""),
		    (tmp & 0x20000000 ? "[0x20] " : ""),
		    (tmp & 0x10000000 ? "[0x10] " : ""),
		    (tmp & 0x08000000 ? "[0x08] " : ""),
		    (tmp & 0x04000000 ? "[0x04] " : ""),
		    (tmp & 0x02000000 ? "[0x02] " : ""),
		    (tmp & 0x01000000 ? "[0x01]" : " "));
		device_printf(sc->dev, "Card Configuration ( & 0x00ff0000 ) : %s%s%s%s%s%s%s%s\n",
		    (tmp & 0x00800000 ? "[0x80] " : ""),
		    (tmp & 0x00400000 ? "[0x40] " : ""),
		    (tmp & 0x00200000 ? "[Legacy INT] " : ""),
		    (tmp & 0x00100000 ? "[0x10] " : ""),
		    (tmp & 0x00080000 ? "[0x08] " : ""),
		    (tmp & 0x00040000 ? "[Codec4] " : ""),
		    (tmp & 0x00020000 ? "[Codec2] " : ""),
		    (tmp & 0x00010000 ? "[I2S Codec]" : " "));
		device_printf(sc->dev, "Card Configuration ( & 0x0000ff00 ) : %s%s%s%s%s%s%s%s\n",
		    (tmp & 0x00008000 ? "[0x80] " : ""),
		    (tmp & 0x00004000 ? "[GPINPUT0] " : ""),
		    (tmp & 0x00002000 ? "[GPINPUT1] " : ""),
		    (tmp & 0x00001000 ? "[GPOUT0] " : ""),
		    (tmp & 0x00000800 ? "[GPOUT1] " : ""),
		    (tmp & 0x00000400 ? "[GPOUT2] " : ""),
		    (tmp & 0x00000200 ? "[Joystick] " : ""),
		    (tmp & 0x00000100 ? "[0x01]" : " "));
		device_printf(sc->dev, "Card Configuration ( & 0x000000ff ) : %s%s%s%s%s%s%s%s\n",
		    (tmp & 0x00000080 ? "[0x80] " : ""),
		    (tmp & 0x00000040 ? "[0x40] " : ""),
		    (tmp & 0x00000020 ? "[0x20] " : ""),
		    (tmp & 0x00000010 ? "[AUTOMUTE] " : ""),
		    (tmp & 0x00000008 ? "[LOCKSOUNDCACHE] " : ""),
		    (tmp & 0x00000004 ? "[LOCKTANKCACHE] " : ""),
		    (tmp & 0x00000002 ? "[MUTEBUTTONENABLE] " : ""),
		    (tmp & 0x00000001 ? "[AUDIOENABLE]" : " "));

		if ((sc->is_emu10k2) || (sc->is_ca0108) || (sc->is_ca0151)) {
			tmp = emu_rd(sc, A_IOCFG, 2);
			device_printf(sc->dev, "Audigy Card Configuration (    0x%04x )\n", tmp);
			device_printf(sc->dev, "Audigy Card Configuration (  & 0xff00 )");
			printf(" : %s%s%s%s%s%s%s%s\n",
			    (tmp & 0x8000 ? "[Rear Speakers] " : ""),
			    (tmp & 0x4000 ? "[Front Speakers] " : ""),
			    (tmp & 0x2000 ? "[0x20] " : ""),
			    (tmp & 0x1000 ? "[0x10] " : ""),
			    (tmp & 0x0800 ? "[0x08] " : ""),
			    (tmp & 0x0400 ? "[0x04] " : ""),
			    (tmp & 0x0200 ? "[0x02] " : ""),
			    (tmp & 0x0100 ? "[AudigyDrive Phones]" : " "));
			device_printf(sc->dev, "Audigy Card Configuration (  & 0x00ff )");
			printf(" : %s%s%s%s%s%s%s%s\n",
			    (tmp & 0x0080 ? "[0x80] " : ""),
			    (tmp & 0x0040 ? "[Mute AnalogOut] " : ""),
			    (tmp & 0x0020 ? "[0x20] " : ""),
			    (tmp & 0x0010 ? "[0x10] " : ""),
			    (tmp & 0x0008 ? "[0x08] " : ""),
			    (tmp & 0x0004 ? "[GPOUT0] " : ""),
			    (tmp & 0x0002 ? "[GPOUT1] " : ""),
			    (tmp & 0x0001 ? "[GPOUT2]" : " "));
		};		/* is_emu10k2 or ca* */
	};			/* bootverbose */

	return 0;
}

static int
emu_uninit(struct emu_sc_info *sc)
{
	uint32_t ch;

	emu_wr(sc, INTE, 0, 4);
	for (ch = 0; ch < NUM_G; ch++)
		emu_wrptr(sc, ch, DCYSUSV, 0);
	for (ch = 0; ch < NUM_G; ch++) {
		emu_wrptr(sc, ch, VTFT, 0);
		emu_wrptr(sc, ch, CVCF, 0);
		emu_wrptr(sc, ch, PTRX, 0);
		emu_wrptr(sc, ch, CPF, 0);
	}

	/* disable audio and lock cache */
	emu_wr(sc, HCFG, HCFG_LOCKSOUNDCACHE | HCFG_LOCKTANKCACHE_MASK | HCFG_MUTEBUTTONENABLE, 4);

	emu_wrptr(sc, 0, PTB, 0);
	/* reset recording buffers */
	emu_wrptr(sc, 0, MICBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, MICBA, 0);
	emu_wrptr(sc, 0, FXBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, FXBA, 0);
	emu_wrptr(sc, 0, FXWC, 0);
	emu_wrptr(sc, 0, ADCBS, ADCBS_BUFSIZE_NONE);
	emu_wrptr(sc, 0, ADCBA, 0);
	emu_wrptr(sc, 0, TCB, 0);
	emu_wrptr(sc, 0, TCBS, 0);

	/* disable channel interrupt */
	emu_wrptr(sc, 0, CLIEL, 0);
	emu_wrptr(sc, 0, CLIEH, 0);
	emu_wrptr(sc, 0, SOLEL, 0);
	emu_wrptr(sc, 0, SOLEH, 0);

	return 0;
}
/* IVAR manager */
static int
emu_read_ivar(device_t bus __unused, device_t dev, int ivar_index, uintptr_t * result)
{
	struct sndcard_func *func = device_get_ivars(dev);

	switch (ivar_index) {
	case 0:
		*result = func->func;
		break;

	default:
		return ENOENT;
	}

	return 0;
}

static int
emu_write_ivar(device_t bus __unused, device_t dev __unused,
    int ivar_index, uintptr_t value __unused)
{

	switch (ivar_index) {
		case 0:
		return EINVAL;

	default:
		return (ENOENT);
	}
}

static int
emu_pci_probe(device_t dev)
{
	struct sbuf *s;
	int thiscard = 0;
	uint16_t vendor;

	vendor = pci_read_config(dev, PCIR_DEVVENDOR, /* bytes */ 2);
	if (vendor != 0x1102)
		return ENXIO;	/* Not Creative */

	thiscard = emu_getcard(dev);
	if (thiscard < 0)
		return ENXIO;

	s = sbuf_new(NULL, NULL, 4096, 0);
	if (s == NULL)
		return ENOMEM;
	sbuf_printf(s, "Creative %s [%s]", emu_cards[thiscard].desc, emu_cards[thiscard].SBcode);
	sbuf_finish(s);

	device_set_desc_copy(dev, sbuf_data(s));
#ifdef BUS_PROBE_DEFAULT
	return BUS_PROBE_DEFAULT;
#else
        return 0;
#endif
}

static int
emu_getcard(device_t dev)
{
	uint16_t device;
	uint16_t subdevice;
	int n_cards = 0;
	int thiscard = (-1);
	int i;

	device = pci_read_config(dev, PCIR_DEVICE, /* bytes */ 2);
	subdevice = pci_read_config(dev, PCIR_SUBDEV_0, /* bytes */ 2);

	n_cards = sizeof(emu_cards) / sizeof(struct emu_hwinfo);
	for (i = 0; i < n_cards; i++) {
		if (device == emu_cards[i].device) {
			if (subdevice == emu_cards[i].subdevice) {
				thiscard = i;
				break;
			}
			if (0x0000 == emu_cards[i].subdevice) {
				thiscard = i;
				/* don't break, we can get more specific card
				 * later in the list */
			}
		}
	};

	n_cards = sizeof(emu_bad_cards) / sizeof(struct emu_hwinfo);
	for (i = 0; i < n_cards; i++) {
		if (device == emu_bad_cards[i].device) {
			if (subdevice == emu_bad_cards[i].subdevice) {
				thiscard = (-1);
				break;
			}
			if (0x0000 == emu_bad_cards[i].subdevice) {
				thiscard = (-1);
				break;	/* we avoid all this cards */
			}
		}
	};
	return thiscard;
}

static int
emu_pci_attach(device_t dev)
{
	struct sndcard_func *func;
	struct emu_sc_info *sc;
#ifdef	BUILD_MIDI
	struct emu_midiinfo *midiinfo[3];
#endif
	uint32_t data;
	int i;
	int device_flags;
	char status[255];
	int error = ENXIO;

	sc = device_get_softc(dev);

	/* Fill in the softc. */
	mtx_init(&sc->lock, "emu10kx", "bridge conf", MTX_DEF);
	mtx_init(&sc->rw, "emu10kx", "atomic op", MTX_DEF);
	sc->dev = dev;
	sc->type = pci_get_devid(dev);
	sc->rev = pci_get_revid(dev);
	sc->enable_ir = 0;
	sc->enable_debug = 0;
	sc->has_ac97 = 0;
	sc->has_51 = 0;
	sc->has_71 = 0;
	sc->broken_digital = 0;
	sc->is_emu10k1 = 0;
	sc->is_emu10k2 = 0;
	sc->is_ca0151 = 0;
	sc->is_ca0108 = 0;

	device_flags = emu_cards[emu_getcard(dev)].flags;
	if (device_flags & HAS_51)
		sc->has_51 = 1;
	if (device_flags & HAS_71) {
		sc->has_51 = 1;
		sc->has_71 = 1;
	};
	if (device_flags & IS_EMU10K1)
		sc->is_emu10k1 = 1;
	if (device_flags & IS_EMU10K2)
		sc->is_emu10k2 = 1;
	if (device_flags & IS_CA0151)
		sc->is_ca0151 = 1;
	if (device_flags & IS_CA0108)
		sc->is_ca0108 = 1;
	if ((sc->is_emu10k2) && (sc->rev == 4)) {
		sc->is_emu10k2 = 0;
		sc->is_ca0151 = 1;	/* for unknown Audigy 2 cards */
	};

	if ((sc->is_emu10k1 + sc->is_emu10k2 + sc->is_ca0151 + sc->is_ca0108) != 1) {
		device_printf(sc->dev, "Unable to detect HW chipset\n");
		goto bad;
	};

	if (device_flags & BROKEN_DIGITAL)
		sc->broken_digital = 1;
	if (device_flags & HAS_AC97)
		sc->has_ac97 = 1;

	sc->opcode_shift = 0;
	if ((sc->is_emu10k2) || (sc->is_ca0151) || (sc->is_ca0108)) {
		sc->opcode_shift = 24;
		sc->high_operand_shift = 12;
		sc->code_base = AUDIGY_CODEBASE;
		sc->code_size = 0x800 / 2;	/* 0x600-0xdff,  2048 words,
						 * 1024 instructions */
		sc->gpr_base = A_FXGPREGBASE;
		sc->num_gprs = 0x200;
		sc->input_base = 0x40;
		sc->output_base = 0x60;
		sc->dsp_zero = 0xc0;
		sc->mchannel_fx = 8;
		sc->num_fxbuses = 16;
		sc->num_inputs = 8;
		sc->num_outputs = 16;
		sc->address_mask = A_PTR_ADDRESS_MASK;
	};
	if (sc->is_emu10k1) {
		sc->opcode_shift = 20;
		sc->high_operand_shift = 10;
		sc->code_base = MICROCODEBASE;
		sc->code_size = 0x400 / 2;	/* 0x400-0x7ff,  1024 words,
						 * 512 instructions */
		sc->gpr_base = FXGPREGBASE;
		sc->num_gprs = 0x100;
		sc->input_base = 0x10;
		sc->output_base = 0x20;
		sc->dsp_zero = 0x40;
		sc->mchannel_fx = 0;
		sc->num_fxbuses = 8;
		sc->num_inputs = 8;
		sc->num_outputs = 16;
		sc->address_mask = PTR_ADDRESS_MASK;
	};
	
	if (sc->opcode_shift == 0)
		goto bad;

	data = pci_read_config(dev, PCIR_COMMAND, 2);
	data |= (PCIM_CMD_PORTEN | PCIM_CMD_BUSMASTEREN);
	pci_write_config(dev, PCIR_COMMAND, data, 2);
	data = pci_read_config(dev, PCIR_COMMAND, 2);

#ifdef PCIR_BARS
	i = PCIR_BAR(0);
#else
	i = PCIR_MAPS;
#endif
#if __FreeBSD_version > 502109
	sc->reg = bus_alloc_resource_any(dev, SYS_RES_IOPORT, &i, RF_ACTIVE);
#else
	sc->reg = bus_alloc_resource(dev, SYS_RES_IOPORT, &i, 0, ~0, 1, RF_ACTIVE);
#endif
	if (sc->reg == NULL) {
		device_printf(dev, "unable to map register space\n");
		goto bad;
	}
	sc->st = rman_get_bustag(sc->reg);
	sc->sh = rman_get_bushandle(sc->reg);

/* FIXME: PCM was HERE */
/*
	sc->bufsz = pcm_getbuffersize(dev, 4096, EMU_DEFAULT_BUFSZ, 65536);
*/
	sc->bufsz = EMU_DEFAULT_BUFSZ;
#if __FreeBSD_version > 501102
	if (bus_dma_tag_create( /* parent */ NULL, /* alignment */ 2, /* boundary */ 0,
	     /* lowaddr */ 1 << 31,	/* can only access 0-2gb */
	     /* highaddr */ BUS_SPACE_MAXADDR,
	     /* filter */ NULL, /* filterarg */ NULL,
	     /* maxsize */ sc->bufsz, /* nsegments */ 1, /* maxsegz */ 0x3ffff,
	     /* flags */ 0, /* lockfunc */ busdma_lock_mutex,
	     /* lockarg */ &Giant, &sc->parent_dmat) != 0) {
#else
	if (bus_dma_tag_create( /* parent */ NULL, /* alignment */ 2, /* boundary */ 0,
	     /* lowaddr */ 1 << 31,	/* can only access 0-2gb */
	     /* highaddr */ BUS_SPACE_MAXADDR,
	     /* filter */ NULL, /* filterarg */ NULL,
	     /* maxsize */ sc->bufsz, /* nsegments */ 1, /* maxsegz */ 0x3ffff,
	     /* flags */ 0, &sc->parent_dmat) != 0) {
#endif
		device_printf(dev, "unable to create dma tag\n");
		goto bad;
	}
	i = 0;
#if __FreeBSD_version > 502109
	sc->irq = bus_alloc_resource_any(dev, SYS_RES_IRQ, &i, RF_ACTIVE | RF_SHAREABLE);
#else
	sc->irq = bus_alloc_resource(dev, SYS_RES_IRQ, &i, 0, ~0, 1, RF_ACTIVE | RF_SHAREABLE);
#endif
	if ((!(sc->irq)) || bus_setup_intr(dev, sc->irq, INTR_MPSAFE | INTR_TYPE_AV, emu_intr, sc, &sc->ih)) {
		device_printf(dev, "unable to map interrupt\n");
		goto bad;
	}
	if (emu_rm_init(sc) != 0) {
		device_printf(dev, "unable to create resource manager\n");
		goto bad;
	};

	if (emumix_init(sc) != 0) {
		device_printf(dev, "unable to create private mixer\n");
		goto bad;
	};

	if (emu_init(sc) == -1) {
		device_printf(dev, "unable to initialize the card\n");
		goto bad;
	}
	if (emu10kx_dev_init(sc) == ENXIO) {
		device_printf(dev, "unable to create control device\n");
		goto bad;
	}
	snprintf(status, 255, "rev %d at io 0x%lx irq %ld", sc->rev, rman_get_start(sc->reg), rman_get_start(sc->irq));
	emu_enatimer(sc, 0);
	/* PCM Audio */
	func = malloc(sizeof(struct sndcard_func), M_DEVBUF, M_NOWAIT | M_ZERO);
	if (func == NULL) {
		error = ENOMEM;
		goto bad;
	}
	func->func = SCF_PCM;
	sc->pcm = device_add_child(dev, "pcm", -1);
	device_set_ivars(sc->pcm, func);
#ifdef BUILD_MIDI
	/* Midi Interface 1 */
	if ((sc->is_emu10k1) || (sc->is_emu10k2)) {
		func = malloc(sizeof(struct sndcard_func), M_DEVBUF, M_NOWAIT | M_ZERO);
		if (func == NULL) {
			error = ENOMEM;
			goto bad;
		}
		midiinfo[0] = malloc(sizeof(struct emu_midiinfo), M_DEVBUF, M_NOWAIT | M_ZERO);
		if (midiinfo[0] == NULL) {
			error = ENOMEM;
			goto bad;
		}
		midiinfo[0]->card = sc;
		if (sc->is_emu10k2) {
			midiinfo[0]->port = A_MUDATA1;
			midiinfo[0]->portnr = 1;
		};
		if (sc->is_emu10k1) {
			midiinfo[0]->port = MUDATA;
			midiinfo[0]->portnr = 1;
		};
		func->func = SCF_MIDI;
		func->varinfo = midiinfo[0];
		sc->midi[0] = device_add_child(dev, "midi", -1);
		device_set_ivars(sc->midi[0], func);
	};
	/* Midi Interface 2 */
	if (sc->is_emu10k2) {
		func = malloc(sizeof(struct sndcard_func), M_DEVBUF, M_NOWAIT | M_ZERO);
		if (func == NULL) {
			error = ENOMEM;
			goto bad;
		}
		midiinfo[1] = malloc(sizeof(struct emu_midiinfo), M_DEVBUF, M_NOWAIT | M_ZERO);
		if (midiinfo[1] == NULL) {
			error = ENOMEM;
			goto bad;
		}
		midiinfo[1]->card = sc;

		midiinfo[1]->port = A_MUDATA2;
		midiinfo[1]->portnr = 2;

		func->func = SCF_MIDI;
		func->varinfo = midiinfo[1];
		sc->midi[1] = device_add_child(dev, "midi", -1);
		device_set_ivars(sc->midi[1], func);
	};
#endif

	return bus_generic_attach(dev);

bad:
	if (sc->cdev)
		emu10kx_dev_uninit(sc);
	if (sc->mixer)
		emumix_uninit(sc);
	if (sc->rm)
		emu_rm_uninit(sc);
#ifdef PCIR_BARS
	if (sc->reg)
		bus_release_resource(dev, SYS_RES_IOPORT, PCIR_BAR(0), sc->reg);
#else
	if (sc->reg)
		bus_release_resource(dev, SYS_RES_IOPORT, PCIR_MAPS, sc->reg);
#endif
	if (sc->ih)
		bus_teardown_intr(dev, sc->irq, sc->ih);
	if (sc->irq)
		bus_release_resource(dev, SYS_RES_IRQ, 0, sc->irq);
	if (sc->parent_dmat)
		bus_dma_tag_destroy(sc->parent_dmat);
	mtx_destroy(&sc->lock);
	mtx_destroy(&sc->rw);
	return error;
}

static int
emu_pci_detach(device_t dev)
{
	struct emu_sc_info *sc;
	int devcount, i;
	device_t *childlist;
	int r = 0;

	sc = device_get_softc(dev);

	if (sc->pcm)
		r = device_delete_child(dev, sc->pcm);
	if (r)
		return r;

#ifdef BUILD_MIDI
	if (sc->midi[0])
		r = device_delete_child(dev, sc->midi[0]);
	if (r)
		return r;
	if (sc->midi[1])
		r = device_delete_child(dev, sc->midi[1]);
	if (r)
		return r;
#endif
	(void)device_get_children(dev, &childlist, &devcount);
	for (i = 0; i < devcount - 1; i++) {
		device_printf(dev, "removing stale child %d (unit %d)\n", i, device_get_unit(childlist[i]));
		device_delete_child(dev, childlist[i]);
	}
	free(childlist, M_TEMP);

	/* shutdown chip */
	emu_uninit(sc);
	emumix_uninit(sc);
	r = emu10kx_dev_uninit(sc);
	if (r)
		return r;
	emu_rm_uninit(sc);
#ifdef PCIR_BARS
	if (sc->reg)
		bus_release_resource(dev, SYS_RES_IOPORT, PCIR_BAR(0), sc->reg);
#else
	if (sc->reg)
		bus_release_resource(dev, SYS_RES_IOPORT, PCIR_MAPS, sc->reg);
#endif
	bus_teardown_intr(dev, sc->irq, sc->ih);
	bus_release_resource(dev, SYS_RES_IRQ, 0, sc->irq);
	bus_dma_tag_destroy(sc->parent_dmat);
	mtx_destroy(&sc->lock);
	mtx_destroy(&sc->rw);
	return bus_generic_detach(dev);
}
/* add suspend, resume */
static device_method_t emu_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe, emu_pci_probe),
	DEVMETHOD(device_attach, emu_pci_attach),
	DEVMETHOD(device_detach, emu_pci_detach),
	/* Bus methods */
	DEVMETHOD(bus_read_ivar, emu_read_ivar),
	DEVMETHOD(bus_write_ivar, emu_write_ivar),

	{0, 0}
};


static driver_t emu_driver = {
	"emu10kx",
	emu_methods,
	sizeof(struct emu_sc_info),
	NULL,
	0,
	NULL
};

static int
emu_modevent(module_t mod __unused, int cmd, void *data __unused)
{
	int err = 0;

	switch (cmd) {
	case MOD_LOAD:
		break;		/* Success */

	case MOD_UNLOAD:
	case MOD_SHUTDOWN:

		/* XXX  Should check state of pcm & midi subdevices here !!! */

		break;		/* Success */

	default:
		err = EINVAL;
		break;
	}

	return (err);

};


DRIVER_MODULE(snd_emu10kx, pci, emu_driver, emu_devclass, emu_modevent, NULL);
MODULE_VERSION(snd_emu10kx, 1);
