/*-
 * Copyright 2003 Yuriy Tsibizov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer
 *    in this position and unchanged.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioccom.h>

#include "../emu10kx.h"
#include "../emu10kx-ioctl.h"
#include "../emu10kx-opcodes.h"

#define MIXER_FX	0
#define MIXER_INPUT	1
#define MIXER_OUTPUT 2
#define MIXER_AC97 3
#define MIXER_MAIN 4

#define LIST_OUTPUT 1
#define SAVE_OUTPUT 2


static void
usage(void)
{
	printf("EMU10Kx Private mixer / Control\n");
	printf("Usage:\n");
	printf("emuctrl [-f device] amp [amplification]\n");
	printf("emuctrl [-f device] debug [on|off]\n");
	printf("emuctrl [-f device] dump [voice]\n");
	printf("emuctrl [-f device] rget volumeId\n");
	printf("emuctrl [-f device] ir [enable]\n");
	printf("emuctrl [-f device] list [in|out|fx|mix]\n");
	printf("emuctrl [-f device] mode [analog|digital]\n");
	printf("emuctrl [-f device] poke [reg|regptr|code|p16v] ...\n");
	printf("emuctrl [-f device] reset\n");
	printf("emuctrl [-f device] rget volumeId\n");
	printf("emuctrl [-f device] rset volumeId left[:right]\n");
	printf("emuctrl [-f device] save\n");
	printf("emuctrl [-f device] set volumeId left[:right]\n");
	printf("emuctrl [-f device] spdif [pcm|pass]\n");
	return;
}

static void
list(int emu10k, int show_list, int mode)
{
	int i;
	struct emu10kx_mixer_ctl ctl;
	struct emu10kx_mixer_page page;
	int showpage;
	const char *pageid[] = {"fx", "in", "out", "ac97", "mix"};

	i = 0;
	showpage = 0;

	if (mode == LIST_OUTPUT)
		printf("Id\tDescription          \t  Vol\tRecVol\n");
	for (showpage = 0; showpage < 5; showpage++)
		if (show_list & (1 << showpage)) {
			page.page = showpage;
			if (ioctl(emu10k, E10GETPAGEINFO, &page) < 0) {
				fprintf(stderr, "E10GETPAGEINFO failed.\n");
			} else {
				if (mode == LIST_OUTPUT)
					printf("%s\n", page.desc);
				ctl.page = showpage;
				for (i = 0; i < page.playback_controls; i++) {
					ctl.connector = i;
					ctl.page = showpage;
					if (ioctl(emu10k, E10GETMIXER, &ctl) < 0) {
/*
						printf("%s%d is debug mode volume control\n", pageid[showpage], i);
*/
					} else {
						switch (mode) {
						case LIST_OUTPUT:
							printf("%s%d\t%-20s\t%3d", pageid[showpage], i,
							    ctl.desc,
							    ctl.volume[0]);
							if (ctl.volume[1] >= 0)
								printf(":%d", ctl.volume[1]);
							if (ctl.volume[2] >= 0)
								printf("\t(%d", ctl.volume[2]);
							if (ctl.volume[3] >= 0)
								printf(":%d", ctl.volume[3]);
							if (ctl.volume[2] >= 0)
								printf(")");
							printf("\n");
							break;
						case SAVE_OUTPUT:
							printf("set %s%d %d",
							    pageid[showpage], i,
							    ctl.volume[0]);
							if (ctl.volume[1] >= 0)
								printf(":%d", ctl.volume[1]);
							printf("\n");
							if (ctl.volume[2] >= 0)
								printf("rset %s%d %d",
								    pageid[showpage], i,
								    ctl.volume[2]);
							if (ctl.volume[3] >= 0)
								printf(":%d", ctl.volume[3]);
							if (ctl.volume[2] >= 0)
								printf("\n");
							break;
						};
					};
				};
			};
		};
}

static int
parse_id(char *id, int *page, int *nr)
{
	int res;
	int start = 0;
	char str[4];
	int val;

	res = sscanf(id, "%3s", str);
	if (res < 1)
		return (-1);
	if (strncmp(str, "fx", 2) == 0) {
		*page = MIXER_FX;
		start = 2;
	}
	if (strncmp(str, "in", 2) == 0) {
		*page = MIXER_INPUT;
		start = 2;
	}
	if (strncmp(str, "out", 3) == 0) {
		*page = MIXER_OUTPUT;
		start = 3;
	}
	if (strncmp(str, "ac97", 4) == 0) {
		*page = MIXER_AC97;
		start = 4;
	}
	if (strncmp(str, "mix", 3) == 0) {
		*page = MIXER_MAIN;
		start = 3;
	}
	res = sscanf((id + start), "%4d", &val);
	if (res < 1)
		return (-1);
	*nr = val;
	return 0;
}


char ops[0x10][8] = {
	"MACS",
	"MACS1",
	"MACW",
	"MACW1",
	"MACINTS",
	"MACINTW",
	"ACC3",
	"MACMV",
	"ANDXOR",
	"TSTNEG",
	"LIMIT",
	"LIMIT1",
	"LOG",
	"EXP",
	"INTERP",
	"SKIP"
};
char voice_reg[0x7f][19] = {
	 /* 0x00 */ "CPF",
	 /* 0x01 */ "PTRX",
	 /* 0x02 */ "CVCF",
	 /* 0x03 */ "VTFT",
	 /* 0x04 */ "Z2",
	 /* 0x05 */ "Z1",
	 /* 0x06 */ "PSST",
	 /* 0x07 */ "DSL",
	 /* 0x08 */ "CCCA",
	 /* 0x09 */ "CCR",
	 /* 0x0a */ "REG_0A",
	 /* 0x0b */ "FXRT",
	 /* 0x0c */ "MAPA",
	 /* 0x0d */ "MAPB",
	 /* 0x0e */ "REG_0E",
	 /* 0x0f */ "REG_0F",
	 /* 0x10 */ "ENVVOL",
	 /* 0x11 */ "ATKHLDV",
	 /* 0x12 */ "DCYSUSV",
	 /* 0x13 */ "LFOVAL1",
	 /* 0x14 */ "ENVVAL",
	 /* 0x15 */ "ATKHLDM",
	 /* 0x16 */ "DCYSUSM",
	 /* 0x17 */ "LFOVAL2",
	 /* 0x18 */ "IP",
	 /* 0x19 */ "IFATN",
	 /* 0x1a */ "PEFE",
	 /* 0x1b */ "FMMOD",
	 /* 0x1c */ "TREMFRQ",
	 /* 0x1d */ "FM2FRQ2",
	 /* 0x1e */ "TEMPENV",
	 /* 0x1f */ "REG_1F",
	 /* 0x20 */ "CD0",
	 /* 0x21 */ "CD1",
	 /* 0x22 */ "CD2",
	 /* 0x23 */ "CD3",
	 /* 0x24 */ "CD4",
	 /* 0x25 */ "CD5",
	 /* 0x26 */ "CD6",
	 /* 0x27 */ "CD7",
	 /* 0x28 */ "CD8",
	 /* 0x29 */ "CD9",
	 /* 0x2a */ "CDA",
	 /* 0x2b */ "CDB",
	 /* 0x2c */ "CDC",
	 /* 0x2d */ "CDD",
	 /* 0x2e */ "CDE",
	 /* 0x2f */ "CDF",
	 /* 0x30 */ "REG_30",
	 /* 0x31 */ "REG_31",
	 /* 0x32 */ "REG_32",
	 /* 0x33 */ "REG_33",
	 /* 0x34 */ "REG_34",
	 /* 0x35 */ "REG_35",
	 /* 0x36 */ "REG_36",
	 /* 0x37 */ "REG_37",
	 /* 0x38 */ "REG_38",
	 /* 0x39 */ "REG_39",
	 /* 0x3a */ "REG_3A",
	 /* 0x3b */ "REG_3B",
	 /* 0x3c */ "REG_3C",
	 /* 0x3d */ "REG_3D",
	 /* 0x3e */ "REG_3E",
	 /* 0x3f */ "REG_3F",
	 /* 0x40 */ "PTB",
	 /* 0x41 */ "TCB",
	 /* 0x42 */ "ADCCR",
	 /* 0x43 */ "FXWC",
	 /* 0x44 */ "TCBS",
	 /* 0x45 */ "MICBA",
	 /* 0x46 */ "ADCBA",
	 /* 0x47 */ "FXBA",
	 /* 0x48 */ "REG_48",
	 /* 0x49 */ "MICBS",
	 /* 0x4a */ "ADCBS",
	 /* 0x4b */ "FXBS",
	 /* 0x4c */ "REG_4C",
	 /* 0x4d */ "REG_4D",
	 /* 0x4e */ "REG_4E",
	 /* 0x4f */ "REG_4F",
	 /* 0x50 */ "CDCS",
	 /* 0x51 */ "GPSCS",
	 /* 0x52 */ "DBG",
	 /* 0x53 */ "A_DBG",
	 /* 0x54 */ "SPCS0",
	 /* 0x55 */ "SPCS1",
	 /* 0x56 */ "SPCS2",
	 /* 0x57 */ "REG_57",
	 /* 0x58 */ "CLIEL",
	 /* 0x59 */ "CLIEH",
	 /* 0x5a */ "CLIPL",
	 /* 0x5b */ "CLIPH",
	 /* 0x5c */ "SOLEL",
	 /* 0x5d */ "SOLEH",
	 /* 0x5e */ "SPBYPASS",
	 /* 0x5f */ "AC97SLOT",
	 /* 0x60 */ "CDSRCS",
	 /* 0x61 */ "GPSRCS",
	 /* 0x62 */ "ZVSRCS",
	 /* 0x63 */ "MICIDX (A_ADCIDX)",
	 /* 0x64 */ "ADCIDX",
	 /* 0x65 */ "FXIDX",
	 /* 0x66 */ "REG_66",
	 /* 0x67 */ "REG_67",
	 /* 0x68 */ "REG_68",
	 /* 0x69 */ "REG_69",
	 /* 0x6a */ "REG_6A",
	 /* 0x6b */ "REG_6B",
	 /* 0x6c */ "REG_6C",
	 /* 0x6d */ "REG_6D",
	 /* 0x6e */ "REG_6E",
	 /* 0x6f */ "REG_6F",
	 /* 0x70 */ "A_MUDATA1",
	 /* 0x71 */ "A_MUCMD1",
	 /* 0x72 */ "A_MUDATA2",
	 /* 0x73 */ "A_MUCMD2",
	 /* 0x74 */ "A_FXWC1",
	 /* 0x75 */ "A_FXWC2",
	 /* 0x76 */ "A_SPDIF_SAMPLERATE",
	 /* 0x77 */ "REG_77",
	 /* 0x78 */ "REG_78",
	 /* 0x79 */ "REG_79",
	 /* 0x7a */ "REG_7a",
	 /* 0x7b */ "REG_7b",
	 /* 0x7c */ "A_FXRT2",
	 /* 0x7d */ "A_SENDAMOUNTS",
	 /* 0x7e */ "A_FXRT1"
};

int
main(int argc, char *argv[])
{
	char *name;
	char ch;
	int emu10k;
	int res = 0;
	int left, right;
	int show_list = 0;
	int is_digital = 0;
	int spdif_mode = 0;
	int enable_ir = 0;
	int debug = 0;
	int addr, code;
	int p, n;
	struct emu10kx_mixer_ctl *ctl;
	struct emu10kx_peek *dump;
	struct emu10kx_params *params;
	unsigned int i, j;
	uint32_t op, x, y, z, w;
	char str[64];
	int need_usage = 1;
	int amp = 0;

	ctl = malloc(sizeof(struct emu10kx_mixer_ctl));
	if ((void *)ctl == NULL)
		return (-2);

	name = strdup("/dev/emu10kx0");

	while ((ch = getopt(argc, argv, "f:")) != -1)
		switch (ch) {
		case 'f':
			name = strdup(optarg);
			break;
		default:
			need_usage = 1;
		}
	argc -= (optind - 1);
	argv += (optind - 1);

	if ((emu10k = open(name, O_RDWR)) < 0) {
		err(1, "%s", name);
		free(name);
		return (0);
	};

	free(name);

	if (argc < 2) {
		usage();
		return 0;
	};

	if (strncmp(argv[1], "list", 4) == 0) {
		need_usage = 0;
		show_list = 1 + 2 + 4 + 0 * 8 + 16;
		if (argc > 2) {
			res = sscanf(argv[2], "%4s", str);
			if (strncmp(str, "fx", 2) == 0)
				show_list = 1;
			if (strncmp(str, "in", 2) == 0)
				show_list = 2;
			if (strncmp(str, "out", 3) == 0)
				show_list = 4;
			if (strncmp(str, "ac97", 4) == 0)
				show_list = 8;
			if (strncmp(str, "mix", 3) == 0)
				show_list = 16;
		};
		list(emu10k, show_list, LIST_OUTPUT);
		return 0;
	};
	if (strncmp(argv[1], "save", 4) == 0) {
		need_usage = 0;
		debug = -1;
		if (ioctl(emu10k, E10DEBUG, &debug) < 0) {
			fprintf(stderr, "E10DEBUG failed.\n");
			exit(1);
		};
		printf("debug %s\n", debug ? "on" : "off");

		show_list = 1 + 2 + 4 + 0 * 8 + 16;
		list(emu10k, show_list, SAVE_OUTPUT);
		amp = 0;
		if (ioctl(emu10k, E10MIXERAMPLIFY, &amp) < 0) {
			fprintf(stderr, "E10MIXERAMPLIFY failed.\n");
			exit(1);
		};
		printf("amp %d\n", amp);
		if (ioctl(emu10k, E10GETMODE, &is_digital) < 0) {
			fprintf(stderr, "E10GETMODE failed.\n");
			exit(1);
		};
		printf("mode %s\n", is_digital ? "digital" : "analog");
		if (ioctl(emu10k, E10GETSPDIFMODE, &spdif_mode) < 0) {
			fprintf(stderr, "E10GETSPDIFMODE failed.\n");
			exit(1);
		};
		printf("spdif %s\n", spdif_mode ? "pass" : "pcm");
		enable_ir = 0;
		if (ioctl(emu10k, E10ENABLEIR, &enable_ir) < 0) {
			fprintf(stderr, "E10ENABLEIR failed.\n");
			exit(1);
		};
		if (enable_ir)
			printf("ir enable\n");
		printf("reset\n");

	};

	if (strncmp(argv[1], "mode", 4) == 0) {
		need_usage = 0;
		if (argc > 2) {
			res = sscanf(argv[2], "%7s", str);
			is_digital = 2;
			if (strncmp(str, "analog", 6) == 0)
				is_digital = 0;
			if (strncmp(str, "digital", 7) == 0)
				is_digital = 1;
			if (is_digital == 2) {
				usage();
				printf("Wrong card mode: %s\n", str);
				exit(1);
			};
			if (ioctl(emu10k, E10SETMODE, &is_digital) < 0) {
				fprintf(stderr, "E10SETMODE failed.\n");
				exit(1);
			};
		} else {
			if (ioctl(emu10k, E10GETMODE, &is_digital) < 0) {
				fprintf(stderr, "E10GETMODE failed.\n");
				exit(1);
			};
			printf("Card is in %s mode\n", is_digital ? "Digital" : "Analog");
		};
		return 0;
	};

	if (strncmp(argv[1], "reset", 5) == 0) {
		need_usage = 0;
		is_digital = 0;
		if (ioctl(emu10k, E10MIXERRESET, &is_digital) < 0) {
			fprintf(stderr, "E10MIXERRESET failed.\n");
			exit(1);
		};
		return 0;
	};

	if (strncmp(argv[1], "spdif", 5) == 0) {
		need_usage = 0;
		if (argc > 2) {
			res = sscanf(argv[2], "%4s", str);
			spdif_mode = 2;
			if (strncmp(str, "pcm", 3) == 0)
				spdif_mode = 0;
			if (strncmp(str, "pass", 4) == 0)
				spdif_mode = 1;
			if (spdif_mode == 2) {
				usage();
				printf("Wrong S/PDIF mode: %s\n", str);
				exit(1);
			};
			if (ioctl(emu10k, E10SETSPDIFMODE, &spdif_mode) < 0) {
				fprintf(stderr, "E10SETSPDIFMODE failed.\n");
				exit(1);
			};
		} else {
			if (ioctl(emu10k, E10GETSPDIFMODE, &spdif_mode) < 0) {
				fprintf(stderr, "E10GETSPDIFMODE failed.\n");
				exit(1);
			};
			printf("S/PDIF output is %s\n", spdif_mode ? "S/PDIF passthrough" : "PCM stereo");
		};
		return 0;
	};

	if (strncmp(argv[1], "ir", 4) == 0) {
		need_usage = 0;
		if (argc > 2) {
			res = sscanf(argv[2], "%6s", str);
			if (strncmp(str, "enable", 6) == 0) {
				enable_ir = 1;
				if (ioctl(emu10k, E10ENABLEIR, &enable_ir) < 0) {
					fprintf(stderr, "E10ENABLEIR failed.\n");
					exit(1);
				};
			};
		} else {
			enable_ir = 0;
			if (ioctl(emu10k, E10ENABLEIR, &enable_ir) < 0) {
				fprintf(stderr, "E10ENABLEIR failed.\n");
				exit(1);
			};
			printf("IR reciever MIDI events %s\n", enable_ir ? "enabled" : "disabled");

		};
		return 0;
	};

/* setvol */
	if (strncmp(argv[1], "set", 3) == 0) {
		need_usage = 0;
		if (argc < 4) {
			usage();
			return 0;
		};
		res = parse_id(argv[2], &p, &n);
		if (res < 0) {
			usage();
			return 0;
		};
		ctl->page = p;
		ctl->connector = n;

		res = sscanf(argv[3], "%d:%d", &left, &right);
		if (res < 1) {
			usage();
			return 0;
		};
		if (res == 1)
			right = left;
		ctl->volume[0] = left;
		ctl->volume[1] = right;
		ctl->volume[2] = (-1);
		ctl->volume[3] = (-1);
		if (ioctl(emu10k, E10SETMIXER, ctl) < 0) {
			fprintf(stderr, "E10SETMIXER failed.\n");
			exit(1);
		};
	};


	if (strncmp(argv[1], "rset", 4) == 0) {
		need_usage = 0;
		if (argc < 4) {
			usage();
			return 0;
		};
		res = parse_id(argv[2], &p, &n);
		if (res < 0) {
			usage();
			return 0;
		};
		ctl->page = p;
		ctl->connector = n;

		res = sscanf(argv[3], "%d:%d", &left, &right);
		if (res < 1) {
			usage();
			return 0;
		};
		if (res == 1)
			right = left;
		ctl->volume[2] = left;
		ctl->volume[3] = right;
		ctl->volume[0] = (-1);
		ctl->volume[1] = (-1);
		if (ioctl(emu10k, E10SETMIXER, ctl) < 0) {
			fprintf(stderr, "E10SETMIXER failed.\n");
			exit(1);
		};
	};
	if (strncmp(argv[1], "amp", 3) == 0) {
		need_usage = 0;
		if (argc < 2) {
			usage();
			return 0;
		};
		if (argc == 3) {
			res = sscanf(argv[2], "%d", &amp);
			if (res < 1) {
				usage();
				return 0;
			};
			if (ioctl(emu10k, E10MIXERAMPLIFY, &amp) < 0) {
				fprintf(stderr, "E10MIXERAMPLIFY failed.\n");
				exit(1);
			};
		} else {
			amp = 0;
			if (ioctl(emu10k, E10MIXERAMPLIFY, &amp) < 0) {
				fprintf(stderr, "E10MIXERAMPLIFY failed.\n");
				exit(1);
			};
			printf("Current amplification is %d\n", amp);
		};
	};
	if (strncmp(argv[1], "debug", 4) == 0) {
		need_usage = 0;
		if (argc > 2) {
			res = sscanf(argv[2], "%3s", str);
			debug = -1;
			if (strncmp(str, "on", 2) == 0)
				debug = 1;
			if (strncmp(str, "off", 3) == 0)
				debug = 0;
			if (debug == -1) {
				usage();
				printf("Wrong debug state: %s\n", str);
				exit(1);
			};
			if (ioctl(emu10k, E10DEBUG, &debug) < 0) {
				fprintf(stderr, "E10DEBUG failed.\n");
				exit(1);
			};
			if (debug)
				printf("Card is in debug mode\n");
		};
		if (argc == 1) {
			debug = -1;
			if (ioctl(emu10k, E10DEBUG, &debug) < 0) {
				fprintf(stderr, "E10DEBUG failed.\n");
				exit(1);
			};
			printf("Card is%s in debug mode\n", debug ? " " : " not");
		}
		return 0;
	};

	if (strncmp(argv[1], "poke", 4) == 0) {
		need_usage = 0;

		dump = malloc(sizeof(struct emu10kx_peek));
		if ((void *)dump == NULL)
			return (-2);

		if (argc < 3) {
			usage();
			return 0;
		};

		if (strncmp(argv[2], "reg", 3) == 0) {
			/* poke DSP register (emu_wr) not implemented */
			return 0;
		};

		if (strncmp(argv[2], "regptr", 6) == 0) {
			/* poke DSP register by pointer (wrptr) not
			 * implemented */
			return 0;
		};
		if (strncmp(argv[2], "code", 4) == 0) {
			/* poke DSP code not implemented */
			return 0;
		};
		if (strncmp(argv[2], "efx", 3) == 0) {
			/* poke DSP fx register not implemented */
			return 0;
		};

		if (strncmp(argv[2], "p16v", 4) == 0) {
			/* poke P16V  register by pointer */

			if (argc < 5) {
				usage();
				return 0;
			};

			res = sscanf(argv[3], "%x", &addr);
			if (res < 1) {
				usage();
				return 0;
			};
			res = sscanf(argv[4], "%x", &code);
			if (res < 1) {
				usage();
				return 0;
			};

			dump->addr = addr;
			dump->voice = 0;
			dump->code[0] = 0;
			dump->code[1] = 0;
			printf("Working on addr 0x%02x\n", dump->addr);

			if (ioctl(emu10k, E10PEEKP16V, dump) < 0) {
				fprintf(stderr, "E10PEEKP16V failed.\n");
				exit(1);
			};
			printf("Was: 0x%08x\n", dump->code[0]);

			dump->addr = addr;
			dump->voice = 0;
			dump->code[0] = code;
			dump->code[1] = 0;
			printf("Try: 0x%08x\n", dump->code[0]);

			if (ioctl(emu10k, E10POKEP16V, dump) < 0) {
				fprintf(stderr, "E10POKEP16V failed.\n");
				exit(1);
			};

			dump->addr = addr;
			dump->voice = 0;
			dump->code[0] = 0;
			dump->code[1] = 0;

			if (ioctl(emu10k, E10PEEKP16V, dump) < 0) {
				fprintf(stderr, "E10PEEKP16V failed.\n");
				exit(1);
			};

			printf("Got: 0x%08x\n", dump->code[0]);
		};		/* p16v */
	};			/* poke */


	if (strncmp(argv[1], "dump", 4) == 0) {
		need_usage = 0;

		params = malloc(sizeof(struct emu10kx_params));
		if ((void *)params == NULL)
			return (-2);

		if (ioctl(emu10k, E10PARAMS, params) < 0) {
			fprintf(stderr, "E10PARAMS failed.\n");
			exit(1);
		};

		printf("Card Configuration (   0x%08x )\n", params->iocfg);
		printf("Card Configuration ( & 0xff000000 ) : %s%s%s%s%s%s%s%s\n",
		    (params->iocfg & 0x80000000 ? "[Legacy MPIC] " : ""),
		    (params->iocfg & 0x40000000 ? "[0x40] " : ""),
		    (params->iocfg & 0x20000000 ? "[0x20] " : ""),
		    (params->iocfg & 0x10000000 ? "[0x10] " : ""),
		    (params->iocfg & 0x08000000 ? "[0x08] " : ""),
		    (params->iocfg & 0x04000000 ? "[0x04] " : ""),
		    (params->iocfg & 0x02000000 ? "[0x02] " : ""),
		    (params->iocfg & 0x01000000 ? "[0x01]" : " "));
		printf("Card Configuration ( & 0x00ff0000 ) : %s%s%s%s%s%s%s%s\n",
		    (params->iocfg & 0x00800000 ? "[0x80] " : ""),
		    (params->iocfg & 0x00400000 ? "[0x40] " : ""),
		    (params->iocfg & 0x00200000 ? "[Legacy INT] " : ""),
		    (params->iocfg & 0x00100000 ? "[0x10] " : ""),
		    (params->iocfg & 0x00080000 ? "[0x08] " : ""),
		    (params->iocfg & 0x00040000 ? "[Codec4] " : ""),
		    (params->iocfg & 0x00020000 ? "[Codec2] " : ""),
		    (params->iocfg & 0x00010000 ? "[I2S Codec]" : " "));
		printf("Card Configuration ( & 0x0000ff00 ) : %s%s%s%s%s%s%s%s\n",
		    (params->iocfg & 0x00008000 ? "[0x80] " : ""),
		    (params->iocfg & 0x00004000 ? "[GPINPUT0] " : ""),
		    (params->iocfg & 0x00002000 ? "[GPINPUT1] " : ""),
		    (params->iocfg & 0x00001000 ? "[GPOUT0] " : ""),
		    (params->iocfg & 0x00000800 ? "[GPOUT1] " : ""),
		    (params->iocfg & 0x00000400 ? "[GPOUT2] " : ""),
		    (params->iocfg & 0x00000200 ? "[Joystick] " : ""),
		    (params->iocfg & 0x00000100 ? "[0x01]" : " "));
		printf("Card Configuration ( & 0x000000ff ) : %s%s%s%s%s%s%s%s\n",
		    (params->iocfg & 0x00000080 ? "[0x80] " : ""),
		    (params->iocfg & 0x00000040 ? "[0x40] " : ""),
		    (params->iocfg & 0x00000020 ? "[0x20] " : ""),
		    (params->iocfg & 0x00000010 ? "[AUTOMUTE] " : ""),
		    (params->iocfg & 0x00000008 ? "[LOCKSOUNDCACHE] " : ""),
		    (params->iocfg & 0x00000004 ? "[LOCKTANKCACHE] " : ""),
		    (params->iocfg & 0x00000002 ? "[MUTEBUTTONENABLE] " : ""),
		    (params->iocfg & 0x00000001 ? "[AUDIOENABLE]" : " "));

		if ((params->is_emu10k2) || (params->is_ca0151) || (params->is_ca0108)) {
			printf("Audigy Card Configuration (    0x%04x )\n", params->a_iocfg);
			printf("Audigy Card Configuration (  & 0xff00 )");
			printf(" : %s%s%s%s%s%s%s%s\n",
			    (params->a_iocfg & 0x8000 ? "[Rear Speakers] " : ""),
			    (params->a_iocfg & 0x4000 ? "[Front Speakers] " : ""),
			    (params->a_iocfg & 0x2000 ? "[0x20] " : ""),
			    (params->a_iocfg & 0x1000 ? "[0x10] " : ""),
			    (params->a_iocfg & 0x0800 ? "[0x08] " : ""),
			    (params->a_iocfg & 0x0400 ? "[0x04] " : ""),
			    (params->a_iocfg & 0x0200 ? "[0x02] " : ""),
			    (params->a_iocfg & 0x0100 ? "[AudigyDrive Phones]" : " "));
			printf("Audigy Card Configuration (  & 0x00ff )");
			printf(" : %s%s%s%s%s%s%s%s\n",
			    (params->a_iocfg & 0x0080 ? "[0x80] " : ""),
			    (params->a_iocfg & 0x0040 ? "[Mute AnalogOut] " : ""),
			    (params->a_iocfg & 0x0020 ? "[0x20] " : ""),
			    (params->a_iocfg & 0x0010 ? "[0x10] " : ""),
			    (params->a_iocfg & 0x0008 ? "[0x08] " : ""),
			    (params->a_iocfg & 0x0004 ? "[GPOUT0] " : ""),
			    (params->a_iocfg & 0x0002 ? "[GPOUT1] " : ""),
			    (params->a_iocfg & 0x0001 ? "[GPOUT2]" : " "));
		};		/* is audigy+ */

		dump = malloc(sizeof(struct emu10kx_peek));
		if ((void *)dump == NULL)
			return (-2);

		printf("DSP code dump (%d instructions)\n", params->code_size);
		for (i = 0; i < params->code_size; i++) {
			dump->addr = i;
			dump->code[0] = 0x0;
			dump->code[1] = 0x0;
			if (ioctl(emu10k, E10PEEKDSP, dump) < 0) {
				fprintf(stderr, "E10PEEK failed.\n");
				exit(1);
			};
			if (params->is_emu10k1) {
				x = (dump->code[0] & 0x000ffc00) >> 10;
				y = (dump->code[0] & 0x000003ff);
				op = (dump->code[1] & 0x00f00000) >> 20;
				z = (dump->code[1] & 0x000ffc00) >> 10;
				w = (dump->code[1] & 0x000003ff);
				printf("%03d\t%s\t%03x\t%03x\t%03x\t%03x\t\t(%03x)\n", i, ops[op], z, w, x, y, params->code_base + i * 2);
			} else {
				x = (dump->code[0] & 0x00fff000) >> 12;
				y = (dump->code[0] & 0x00000fff);
				op = (dump->code[1] & 0xff000000) >> 24;
				z = (dump->code[1] & 0x00fff000) >> 12;
				w = (dump->code[1] & 0x00000fff);
				printf("%03d\t%s\t%03x\t%03x\t%03x\t%03x\t\t(%03x)\n", i, ops[op], z, w, x, y, params->code_base + i * 2);
			};
		};
		printf("DSP Register dump\n");
		for (i = 0; i < params->num_gprs; i++) {
			dump->addr = i;
			dump->code[0] = 0x0;
			dump->code[1] = 0x0;
			if (ioctl(emu10k, E10PEEKGPR, dump) < 0) {
				fprintf(stderr, "E10PEEK failed.\n");
				exit(1);
			};
			printf("%03d\t%08x\t\t(%03x)\n", i, dump->code[0], i + params->gpr_base);
		};
		if ((params->is_ca0108) || (params->is_ca0151)) {
			if (params->is_ca0151)
				printf("Audigy 2 ");
			if (params->is_ca0108)
				printf("Audigy 2 Value ");
			printf("P16V Register dump\n");
			for (i = 0; i < 0x7f; i++) {
				dump->addr = i;
				dump->voice = 0;
				dump->code[0] = 0x0;
				dump->code[1] = 0x0;
				if (ioctl(emu10k, E10PEEKP16V, dump) < 0) {
					fprintf(stderr, "E10PEEKP16V failed.\n");
					exit(1);
				};
				printf("%02x\t%08x\n", i, dump->code[0]);
			};
		};

		if (argc > 2) {
			res = sscanf(argv[2], "%5s", str);
			if (strncmp(str, "voice", 5) == 0) {
				printf("Voice Registers dump\n");
				for (i = 0; i < params->num_g; i++) {
					dump->voice = i;
					printf("Voice %d\n", i);
					for (j = 0; j < 0x7f; j++) {
						dump->addr = j;
						dump->code[0] = 0x0;
						dump->code[1] = 0x0;
						if (ioctl(emu10k, E10PEEKVOICE, dump) < 0) {
							fprintf(stderr, "E10PEEK failed.\n");
							exit(1);
						};
						printf("%02x\t%18s\t%08x\n", j, voice_reg[j], dump->code[0]);
					};
				};
			};
		};
	};			/* dump */


	close(emu10k);
	free(ctl);
	if (need_usage)
		usage();
	return 0;
}
