/*
 * $Id: options.c,v 1.1.1.1 2006/07/08 16:31:47 dave Exp $
 */
#include "options.h"

OPTIONS	options = {
	/* Interface Options */
	NULL,		/* Interface */
	0,		/* Promisc */

	/* Trace Options */
	DEFAULT_INITIAL_HOP,	/* Initial hop */
	DEFAULT_MAX_HOP,	/* maximum hops */
	3,		/* Number of probes */
	3000,		/* Timeout */

	/* Packet construction options */
	{0},		/* Source IP Address */
	0,		/* Source Port */
	0,		/* IP Flags */
	TH_SYN,		/* TCP Flags */
	NULL,		/* Data */
	0,		/* Data Length */
	0,		/* Bad checksums */
	0,		/* TCP sequence number */
	DEFAULT_TCP_WIN,/* TCP window size */
	DEFAULT_IP_FRAGOFF,	/* IP fragment offset */

	/* Output options */
	llNORMAL,	/* Log level */
	0,		/* Numeric output */

	/* Internal Stuff */
	0,		/* Specified options */
	{0},		/* Target IP */
};

typedef struct __flag_info {
	char	*name;
	int	*dest;
	int	value;
} FLAG_INFO;

FLAG_INFO	flag_info[] = {
	{"RF",	&options.ip_flags,  IP_RF},
	{"DF",	&options.ip_flags,  IP_DF},
	{"MF",	&options.ip_flags,  IP_MF},
	{"FIN",	&options.tcp_flags, TH_FIN},
	{"SYN",	&options.tcp_flags, TH_SYN},
	{"RST",	&options.tcp_flags, TH_RST},
	{"PSH",	&options.tcp_flags, TH_PUSH},
	{"ACK",	&options.tcp_flags, TH_ACK},
	{"URG",	&options.tcp_flags, TH_URG},
	{"ECE",	&options.tcp_flags, TH_ECE},
	{"CWR",	&options.tcp_flags, TH_CWR},
	{NULL, NULL, 0},
};

char	short_options[] =
		"p:CF:"
		"i:c"
		"I:T:U:P:"
		"r:t:1:h:m:"
		"A:s:f:d:D:R:bq:w:O:"
		"vBn";

#ifdef HAVE_GETOPT_LONG
struct	option	long_options[] = {
	/* Profile Options */
	{"profile",	1, 0, 'p'},
	{"clear",	0, 0, 'C'},
	{"config",	1, 0, 'F'},

	/* Interface Options */
	{"interface",	1, 0, 'i'},
	{"promisc",	0, 0, 'c'},

	/* Trace Type Options */
	{"icmp",	1, 0, 'I'},
	{"tcp",		1, 0, 'T'},
	{"udp",		1, 0, 'U'},
	{"protocol",	1, 0, 'P'},

	/* Trace Options */
	{"probes",	1, 0, 'r'},
	{"timeout",	1, 0, 't'},
	{"initial",	1, 0, '1'},
	{"hop",		1, 0, 'h'},
	{"maximum",	1, 0, 'm'},

	/* Packet Construction Options */
	{"address",	1, 0, 'A'},
	{"source",	1, 0, 's'},
	{"flags",	1, 0, 'f'},
	{"data",	1, 0, 'd'},
	{"data-file",	1, 0, 'D'},
	{"random",	1, 0, 'R'},
	{"badcksum",	0, 0, 'b'},
	{"seq",		1, 0, 'q'},
	{"window",	1, 0, 'w'},
	{"fragoff",	1, 0, 'O'},

	/* Output Options */
	{"verbose",	0, 0, 'v'},
	{"debug",	0, 0, 'B'},
	{"numeric",	0, 0, 'n'},

	/* List Terminator */
	{NULL,		0, 0, 0}
};
#endif /* HAVE_GETOPT_LONG */

void
usage (char *name) {
	fprintf(stderr, "%s - version %s (libpcap=%s)\n",
	    PACKAGE_NAME, PACKAGE_VERSION, pcap_version);
	fprintf(stderr, "Usage: %s [ options ] target [...]\n", name);
	fprintf(stderr, "  -p <profile>   Load trace options from profile\n");
	fprintf(stderr, "  -C             Clear probes list\n");
	fprintf(stderr, "  -F <file>      Specify profiles file\n");

	fprintf(stderr, "  -i <interface> Specify interface\n");
	fprintf(stderr, "  -c             Promiscuous mode\n");

	fprintf(stderr, "  -I <type>      ICMP probe\n");
	fprintf(stderr, "  -T <port>      TCP probe\n");
	fprintf(stderr, "  -U <port>      UDP probe\n");
	fprintf(stderr, "  -P <protocol>  Protocol probe\n");

	fprintf(stderr, "  -1 <hop>       Initial hop\n");
	fprintf(stderr, "  -h <hop>       Specifiy a specific hop\n");
	fprintf(stderr, "  -m <hop>       Specify maximum hop\n");
	fprintf(stderr, "  -r <count>     Number of probes per hop\n");
	fprintf(stderr, "  -t <timeout>   Probe timeout\n");

	fprintf(stderr, "  -A <address>   Source address\n");
	fprintf(stderr, "  -s <port>      Source port\n");
	fprintf(stderr, "  -f <flags>     IP/TCP flags\n");
	fprintf(stderr, "  -d <data>      Packet data\n");
	fprintf(stderr, "  -D <file>      Load packet data from file\n");
	fprintf(stderr, "  -R <length>    Random packet data\n");
	fprintf(stderr, "  -b             Send packets with bad checksums\n");
	fprintf(stderr, "  -q <seq>       TCP sequence number\n");
	fprintf(stderr, "  -w <window>    TCP window size\n");
	fprintf(stderr, "  -O <fragoff>   IP fragment offset\n");

	fprintf(stderr, "  -v             Increase out verbosity\n");
	fprintf(stderr, "  -B             Enable debugging output\n");
	fprintf(stderr, "  -n             Numeric output (disable name resolution)\n");
}

void read_datafile (char *raw_filename) {
	char		filename[PATH_MAX+1];
	FILE		*datafile;

	if (raw_filename[0] == SHARE_DIR_CHAR && raw_filename[1] != 0x00)
		snprintf(filename, PATH_MAX, DATADIR "/" PACKAGE_NAME "/%s",
		  &raw_filename[1]);
	else
		strncpy(filename, raw_filename, PATH_MAX);

	logmsg(llDEBUG, "Reading datafile %s\n", filename);
	if ((options.data = malloc(MAX_PKT_DATA_LEN)) == NULL)
		errx(EX_OSERR, "Unable to allocate data buffer!");

	if ((datafile = fopen(filename, "r")) == NULL)
		err(EX_IOERR, "Unable to open data file!");
	options.datalen = fread(options.data, 1, MAX_PKT_DATA_LEN, datafile);
	if (ferror(datafile))
		err(EX_IOERR, "Unable to read data!");
	if (fclose(datafile) != 0)
		err(EX_IOERR, "Unable to close data file!");
}

void
process_cmdline(int argc, char *argv[]) {
	char		*cp;
	int		i;
	struct servent	*servent;
	struct protoent	*protoent;

	if (argc == 1) {
		usage(argv[0]);
		exit(EX_USAGE);
	}

	if (options.loglevel > llDEBUG) {
		printf("Options: |");
		for (i = 0; i < argc; i++) {
			printf("%s|", argv[i]);
		}
		printf("\n");
	}

#ifdef HAVE_GETOPT_LONG
	opterr = 1;
	while ((i = getopt_long(argc, argv, short_options, long_options,
	    NULL)) != -1) {
#else
	while ((i = getopt(argc, argv, short_options)) != -1) {
#endif /* HAVE_GETOPT_LONG */
		switch(i) {

		/* Profile Options */
		case 'p':
			if (!load_profile(optarg))
				errx(EX_USAGE, "Unable to locate profile [%s]",
				    optarg);
			break;
		case 'C':
			logmsg(llDEBUG, "Clearing probes list\n");
			clear_probes();
			break;
		case 'F':
			profile_loc[0] = strdup(optarg);
			profile_loc[1] = NULL;
			logmsg(llDEBUG, "Profiles file set to %s\n",
			    profile_loc[0]);
			break;

		/* Interface Options */
		case 'i':
			options.interface = strdup(optarg);
			logmsg(llDEBUG, "Interface = %s\n", options.interface);
			break;
		case 'c':
			options.promisc = 1;
			logmsg(llDEBUG, "Promiscuous mode enabled\n");
			break;

		/* Trace Type Options */
		case 'I':
			for (i = 0; optarg[i] != 0; i++) {
				switch(toupper(optarg[i])) {
				case 'E':
				case 'P':
					add_probe(ptICMP, icmpECHO);
					break;
				case 'T':
				case 'S':
					add_probe(ptICMP, icmpTIMESTAMP);
					break;
				case 'A':
				case 'N':
				case 'M':
					add_probe(ptICMP, icmpADDRESSMASK);
					break;
				case 'I':
					add_probe(ptICMP, icmpINFO);
					break;
				case ':':
				case ',':
					/* Skip over delimiters */
					break;
				default:
					errx(EX_USAGE, "Invalid ICMP probe type!");
					/* NOTREACHED */
				}
			}
			break;
		case 'T':
			cp = strtok(optarg, PROBE_DELIMS);
			while (cp != NULL) {
				if ((i = atoi(cp)) == 0) {
					if ((servent = getservbyname(cp,
					    "tcp")) == NULL)
						errx(EX_USAGE,
						    "Invalid TCP service!");
					add_probe(ptTCP,
					    htons(servent->s_port));
				} else
					add_probe(ptTCP, i);
				cp = strtok(NULL, PROBE_DELIMS);
			}
			break;
		case 'U':
			cp = strtok(optarg, PROBE_DELIMS);
			while (cp != NULL) {
				if ((i = atoi(cp)) == 0) {
					if ((servent = getservbyname(cp,
					    "udp")) == NULL)
						errx(EX_USAGE,
						    "Invalid UDP service!");
					add_probe(ptUDP,
					    htons(servent->s_port));
				} else
					add_probe(ptUDP, i);
				cp = strtok(NULL, PROBE_DELIMS);
			}
			break;
		case 'P':
			cp = strtok(optarg, PROBE_DELIMS);
			while (cp != NULL) {
				if ((i = atoi(cp)) == 0) {
					if ((protoent = getprotobyname(cp))
					    == NULL)
						errx(EX_USAGE,
						    "Invalid protocol");
					add_probe(ptPROTO, protoent->p_proto);
				} else
					add_probe(ptPROTO, i);
				cp = strtok(NULL, PROBE_DELIMS);
			}
			break;

		/* Trace options */
		case '1':
			if ((options.initial_hop = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid initial hop value!");
			logmsg(llDEBUG, "Initial hop = %d\n",
			    options.initial_hop);
			break;
		case 'h':
			if ((options.initial_hop = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid hop value!");
			options.max_hop = options.initial_hop;
			logmsg(llDEBUG, "Examining hop %d\n",
			    options.initial_hop);
			break;
		case 'm':
			if ((options.max_hop = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid maximum hop value!");
			logmsg(llDEBUG, "Maximum hop = %d\n", options.max_hop);
			break;
		case 'r':
			if ((options.probes = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid probe count value!");
			logmsg(llDEBUG, "Probe count = %d\n", options.probes);
			break;
		case 't':
			if ((options.timeout = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid timeout value!");
			logmsg(llDEBUG, "Timeout = %d\n", options.timeout);
			break;

		/* Packet Construction Options */
		case 'A':
			if (!inet_aton(optarg, &options.source_ip))
				errx(EX_USAGE, "Invalid source address specified!");
			options.specified |= SPEC_SRC_ADDR;
			logmsg(llDEBUG, "Source IP address = %s\n",
			    inet_ntoa(options.source_ip));
			break;
		case 's':
			if (sscanf(optarg, "%d", &options.source_port) != 1)
				errx(EX_USAGE, "Invalid source port!");
			options.specified |= SPEC_SRC_PORT;
			logmsg(llDEBUG, "Source port = %d\n", options.source_port);
			break;
		case 'f':
			options.ip_flags = 0;
			options.tcp_flags = 0;
			cp = strtok(optarg, ",");
			while (cp != NULL) {
				for (i = 0; flag_info[i].name != NULL; i++)
					if (strcasecmp(flag_info[i].name, cp)
					    == 0)
						break;
				if (flag_info[i].name == NULL)
					errx(EX_USAGE,
					    "Unrecognised flag! (%s)", cp);
				*flag_info[i].dest |= flag_info[i].value;
				cp = strtok(NULL, ",");
			}
			logmsg(llDEBUG, "IP Flags = 0x%04x, TCP Flags = 0x%02x",
			    options.ip_flags, options.tcp_flags);
			break;
		case 'd':
			if ((options.data = malloc(strlen(optarg))) == NULL)
				err(EX_OSERR, "Unable to allocate data buffer!");
			options.datalen = expand_string(options.data, optarg);
			logmsg(llDEBUG, "Packet data: %d bytes\n",
			    options.datalen);
			break;
		case 'D':
			read_datafile(optarg);
			logmsg(llDEBUG, "Packet data: %d bytes\n",
			    options.datalen);
			break;
		case 'R':
			if ((options.datalen = atoi(optarg)) == 0)
				errx(EX_USAGE, "Invalid data size!");
			if ((options.data = malloc(options.datalen)) == NULL)
				err(EX_OSERR, "Unable to allocate data buffer!");
			for (i = 0; i < options.datalen; i++)
				options.data[i] = rand() % 256;
			break;
		case 'b':
			options.badcksum = 1;
			logmsg(llDEBUG, "Bad checksums enabled\n");
			break;
		case 'q':
			if (sscanf(optarg, "%ld", &options.tcp_seq) != 1)
				errx(EX_USAGE, "Invalid TCP sequence number!");
			options.specified |= SPEC_TCP_SEQ;
			logmsg(llDEBUG,"TCP sequence number = %ld",
			    options.tcp_seq);
			break;
		case 'w':
			if (sscanf(optarg, "%d", &options.tcp_win) != 1)
				errx(EX_USAGE, "Invalid TCP window size!");
			logmsg(llDEBUG, "TCP window size = %d\n", options.tcp_win);
			break;
		case 'O':
			if (sscanf(optarg, "%d", &options.ip_fragoff) != 1)
				errx(EX_USAGE, "Invalid IP fragment offset!");
			logmsg(llDEBUG, "IP fragment offset = %d\n", options.ip_fragoff);
			break;

		/* Output Options */
		case 'v':
			if (++options.loglevel > llINSANE) {
				options.loglevel = llINSANE;
				logmsg(llVERBOSE, "Set log level to Insane "
				    "(Too much of a good thing)\n");
			} else
				logmsg(llVERBOSE, "Set log level to %s\n",
				    loglevel_desc[options.loglevel]);
			break;
		case 'B':
			options.loglevel = llDEBUG;
			logmsg(llDEBUG, "Set log level to %s\n",
			    loglevel_desc[options.loglevel]);
			break;
		case 'n':
			options.numeric = 1;
			logmsg(llDEBUG, "Numeric output enabled\n");
			break;

		default:
			exit(EX_USAGE);
			/* NOTREACHED */

		}
	}

	/*
	 * Sanity check options
	 */

	if (options.initial_hop >= options.max_hop) {
		if (options.max_hop != DEFAULT_MAX_HOP)
			warnx("Warning: Initial hop <= maximum hop!");
		else {
			options.max_hop = options.initial_hop +
			    DEFAULT_MAX_HOP;
			logmsg(llNORMAL, "Warning: Setting maximum hop to %d\n",
			    options.max_hop);
		}
	}

}
