/*
 * $Id: packetfu.c,v 1.1.1.1 2006/07/08 16:31:47 dave Exp $
 */
#include "packetfu.h"

intf_t		*intf = NULL;
ip_t		*ip = NULL;
pcap_t		*pcap = NULL;
int		datalink_len = 0;
char		*pcap_iface = NULL;

unsigned char	intf_buffer[1024];
struct intf_entry *intf_info = (struct intf_entry *)intf_buffer;

#define RESET_INTF_INFO \
	memset(intf_info, 0, sizeof(*intf_info)); \
	intf_info->intf_len = sizeof(intf_buffer);

char *
target_iface(struct in_addr addr) {
	struct addr daddr;

	daddr.addr_type = ADDR_TYPE_IP;
	daddr.addr_bits = 32;
	daddr.addr_ip = addr.s_addr;

	RESET_INTF_INFO;
	if (intf_get_dst(intf, intf_info, &daddr) < 0)
		err(EX_OSERR, "intf_get_dst(%s/%d) failed",
		    __FILE__, __LINE__);
	return(intf_info->intf_name);
}

void
pcap_init(char *ifname) {
	char errbuf[PCAP_ERRBUF_SIZE];
	int datalink;
	bpf_u_int32 localnet, netmask;
	struct bpf_program filter_code;
	char pcap_filter[1024];

	/* Close the current pcap handle if there is one */
	if (pcap != NULL)
		pcap_close(pcap);

	/* Get IP Address Of Interface */
	RESET_INTF_INFO;
	strncpy(intf_info->intf_name, ifname, sizeof(intf_info->intf_name));
	if (intf_get(intf, intf_info) < 0)
		err(EX_IOERR, "intf_get(%s:%d) failed", __FILE__, __LINE__);
	memcpy(&options.source_ip, &intf_info->intf_addr.addr_ip,
	    sizeof(options.source_ip));

	/* Create new pcap handle */
	if ((pcap = pcap_open_live(ifname, PCAP_SNAPLEN,
	    options.promisc, PCAP_TIMEOUT, errbuf)) == NULL)
		errx(EX_IOERR, "%s", errbuf);

	/* Calculate datalink header length */
	if ((datalink = pcap_datalink(pcap)) < 0)
		errx(EX_IOERR, "pcap_datalink() failed! (%s)", pcap_geterr(pcap));
	switch(datalink) {
	case DLT_RAW:    datalink_len = 0;  break;
	case DLT_NULL:   datalink_len = 4;  break;
	case DLT_EN10MB: datalink_len = 14; break;
	case DLT_SLIP:   datalink_len = 24; break;
	case DLT_PPP:    datalink_len = 24; break;
	default:
		errx(EX_IOERR, "Unknown data link type! (%d)", datalink);
	}

	/* Get netmask (for pca_compile call later on) */
	if (pcap_lookupnet(ifname, &localnet, &netmask, NULL) < 0)
		errx(EX_IOERR, "pcap_lookupnet failed! (%s)", pcap_geterr(pcap));

	/* Compile our pcap filter */
	snprintf(pcap_filter, sizeof(pcap_filter),
	    PCAP_FILTER, inet_ntoa(options.source_ip));
	if (pcap_compile(pcap, &filter_code, pcap_filter, 1, netmask) < 0)
		errx(EX_IOERR, "pcap_compile failed! (%s)", pcap_geterr(pcap));

	/* Install out pcap filter */
	if (pcap_setfilter(pcap, &filter_code) < 0)
		errx(EX_IOERR, "pcap_setfilter failed! (%s)", pcap_geterr(pcap));

	/* Cleanup */
	pcap_freecode(&filter_code);
}

void
network_startup(void) {
	if ((intf = intf_open()) == NULL)
		err(EX_OSERR, "intf_open(%s:%d) failed", __FILE__, __LINE__);
	if ((ip = ip_open()) == NULL)
		err(EX_OSERR, "ip_open(%s:%d) failed", __FILE__, __LINE__);
}

void
send_packet(PROBE *probe, STATE *state) {
	unsigned char buffer[IP_LEN_MAX];
	struct ip_hdr   *iph;
	struct tcp_hdr  *tcp;
	struct udp_hdr  *udp;
	struct etrace_icmp_pkt *icmp;
	void		*payload;
	int		len;

	memset(buffer, 0, sizeof(buffer));
	iph = (struct ip_hdr *)buffer;
	iph->ip_v = 4;
	iph->ip_hl = 5;
	iph->ip_tos = 0;
	iph->ip_len = 0;
	iph->ip_id = htons(state->ip_id);
	iph->ip_off = htons(options.ip_flags | options.ip_fragoff);
	iph->ip_ttl = state->ttl;
	iph->ip_p = 0;
	iph->ip_sum = 0;
	iph->ip_src = options.source_ip.s_addr;
	iph->ip_dst = state->dest_ip.s_addr;
	switch(probe->type) {
	case ptICMP:
		iph->ip_p = IP_PROTO_ICMP;
		icmp = (struct etrace_icmp_pkt *)(buffer+IP_HDR_LEN);
		payload = icmp+ICMP_HDR_LEN;
		len = IP_HDR_LEN + ICMP_HDR_LEN + 4;
		switch(probe->data) {
		/* FIXME: Only inc length if options.datalen = 0 */
		case icmpECHO:
			icmp->type = ICMP_ECHO;
			break;
		case icmpTIMESTAMP:
			icmp->type = ICMP_TSTAMP;
			len += 12;
			break;
		case icmpADDRESSMASK:
			icmp->type = ICMP_MASK;
			len += 4;
			break;
		case icmpINFO:
			icmp->type = ICMP_INFO;
			break;
		default:
			errx(EX_SOFTWARE, "Bad ICMP type (%d)", probe->data);
			/* NOTREACHED */
		}
		icmp->code = 0;
		icmp->checksum = 0;
		icmp->id = htons(state->ip_id);
		icmp->seq = htons(state->tcp_seq % 0xffff);
		break;
	case ptTCP:
		iph->ip_p = IP_PROTO_TCP;
		tcp = (struct tcp_hdr *)(buffer+IP_HDR_LEN);
		payload = tcp+TCP_HDR_LEN;
		len = IP_HDR_LEN + TCP_HDR_LEN;
		tcp->th_sport = htons(state->src_port);
		tcp->th_dport = htons(probe->data);
		tcp->th_seq = htons(state->tcp_seq);
		tcp->th_ack = htons(state->tcp_ack);
		tcp->th_x2 = 0;
		tcp->th_off = 5; 		/* FIXME: Is this correct?? */
		tcp->th_flags = options.tcp_flags;
		tcp->th_win = htons(options.tcp_win);
		tcp->th_sum = 0;
		tcp->th_urp = 0;
		break;
	case ptUDP:
		iph->ip_p = IP_PROTO_UDP;
		udp = (struct udp_hdr *)(buffer+IP_HDR_LEN);
		payload = udp+UDP_HDR_LEN;
		len = IP_HDR_LEN + UDP_HDR_LEN;
		udp->uh_sport = htons(state->src_port);
		udp->uh_dport = htons(probe->data);
		udp->uh_ulen = htons(UDP_HDR_LEN + options.datalen);
		udp->uh_sum = 0;
		break;
	case ptPROTO:
		iph->ip_p = probe->data;
		payload = iph+IP_HDR_LEN;
		len = IP_HDR_LEN;
		break;
	default:
		errx(EX_SOFTWARE, "Invalid probe type! (%d)", probe->type);
		/* NOTREACHED */
	}

	/* Append Payload */
	if (options.datalen > 0) {
		memcpy(payload, options.data, options.datalen);
		len += options.datalen;
	}

	/* Calculate Checksums */
	iph->ip_len = htons(len);
	ip_checksum(buffer, len);

	/* Send Packet */
	if (ip_send(ip, buffer, len) != len)
		err(EX_IOERR, "ip_send() failed");
}

int
process_packet(PROBE *probe, STATE *state, const u_char *packet, int pktlen) {
	struct ip_hdr   *iph;
	struct tcp_hdr  *tcp;
	struct udp_hdr  *udp;
	struct etrace_icmp_pkt *icmp;

	iph = (struct ip_hdr *)(packet + datalink_len);
	switch(iph->ip_p) {
	case IP_PROTO_ICMP:
		icmp = (struct etrace_icmp_pkt *)(packet + (iph->ip_hl << 2) + datalink_len);
		switch (icmp->type) {
		case ICMP_TIMEXCEED:
			if (htons(((struct ip_hdr *)&(icmp->payload))->ip_id)
			    != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Time Exceeded "
				    "with incorrect IP ID!\n");
				return(0);
			}
			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "TTL Exceeded");

			state->ttl++;
			return(1);

		case ICMP_ECHOREPLY:
			if (probe->type != ptICMP || probe->data != icmpECHO)
				return(0);
			if (ntohs(icmp->id) != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Echo Reply "
				    "with incorrect ICMP ID!\n");
				return(0);
			}

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "ICMP Echo");

			state->complete = 1;
			return(1);

		case ICMP_TSTAMPREPLY:
			if (probe->type != ptICMP ||
			    probe->data != icmpTIMESTAMP)
				return(0);
			if (ntohs(icmp->id) != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Timestamp Reply "
				    "with incorrect ICMP ID!\n");
				return(0);
			}

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "ICMP Timestamp");

			state->complete = 1;
			return(1);

		case ICMP_MASKREPLY:
			if (probe->type != ptICMP ||
			    probe->data != icmpADDRESSMASK)
				return(0);
			if (icmp->id != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Address Mask reply "
				    "with incorrect ICMP ID!\n");
				return(0);
			}

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "ICMP Address Mask");

			state->complete = 1;
			return(1);

		case ICMP_INFOREPLY:
			if (probe->type != ptICMP ||
			    probe->data != icmpINFO)
				return(0);
			if (icmp->id != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Info "
				    "with incorrect ICMP ID!\n");
				return(0);
			}

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "ICMP Info");

			state->complete = 1;
			return(1);

		case ICMP_UNREACH:
			if (probe->type != ptUDP ||
			    icmp->code != ICMP_UNREACH_PORT)
				goto icmp_error;

			if (htons(((struct ip_hdr *)&(icmp->payload))->ip_id)
			    != state->ip_id) {
				logmsg(llDEBUG, "Received ICMP Port Unreachable "
				    "with incorrect IP ID!\n");
				return(0);
			}

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "Port Unreachable");
			state->complete = 1;
			return(1);

		case ICMP_REDIRECT:
			/* FIXME: Do somthing intelligent with redirects */

		default:
		    icmp_error:
			if (htons(((struct ip_hdr *)&(icmp->payload))->ip_id)
			    == state->ip_id) {
				display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
				    "ICMP %s", icmp_description(icmp->type, icmp->code));
			} else {
				logmsg(llDEBUG, "Unknown ICMP packet from %s: %s\n",
				    inet_ntoa(*(struct in_addr *)&iph->ip_src),
				    icmp_description(icmp->type, icmp->code));
			}
			break;
		}
		break;
	case IP_PROTO_TCP:
		if (probe->type != ptTCP)
			return(0);

		tcp = (struct tcp_hdr *)(packet + (iph->ip_hl << 2) + datalink_len);
		if (iph->ip_src == state->dest_ip.s_addr &&
		    ntohs(tcp->th_sport) == probe->data &&
		    ntohs(tcp->th_dport) == state->src_port) {
			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "%s%s%s%s%s%s",
			    tcp->th_flags & TH_FIN  ? "F" : "",
			    tcp->th_flags & TH_SYN  ? "S" : "",
			    tcp->th_flags & TH_RST  ? "R" : "",
			    tcp->th_flags & TH_PUSH ? "P" : "",
			    tcp->th_flags & TH_ACK  ? "A" : "",
			    tcp->th_flags & TH_URG  ? "U" : "");

			state->complete = 1;
			return(1);
		}
		break;
	case IP_PROTO_UDP:
		if (probe->type != ptUDP)
			return(0);

		udp = (struct udp_hdr *)(packet + (iph->ip_hl << 2) + datalink_len);
		if (iph->ip_src == options.target.s_addr &&
		    ntohs(udp->uh_sport) == probe->data &&
		    ntohs(udp->uh_dport) == state->src_port) {

			display_hop(state->ttl, &iph->ip_src, iph->ip_ttl,
			    "UDP Packet");

			state->complete = 1;
			return(1);
		}
		break;
	}

	return(0);
}

void
network_shutdown(void) {
	if (pcap != NULL)
		pcap_close(pcap);
	if (ip != NULL)
		ip_close(ip);
	if (intf != NULL)
		intf_close(intf);
}
