/*
 * $Id: trace.c,v 1.1.1.1 2006/07/08 16:31:47 dave Exp $
 */
#include "trace.h"
#include "probes.h"
#include "packetfu.h"

static	sigjmp_buf	jmpbuf;
static	int		canjump;

/*
 * Signal handler for packet read timeout
 */
void sig_alarm (int sig) {
	if (canjump == 1)
		siglongjmp(jmpbuf, 1);
}

void
do_trace(char *target) {
	struct itimerval	tout, timeroff;
	struct hostent		*host;
	struct pcap_pkthdr	hdr;
	const  u_char		*rbuf = NULL;
	static PROBE		*probe;
	static STATE		state;
	int			i;
	char			*iface;

	logmsg(llDEBUG, "Initialising trace\n");
	if (signal(SIGALRM, sig_alarm) == SIG_ERR)
		err(EX_OSERR, "Unable to install signal handler!");

	memset(&timeroff, 0, sizeof(timeroff));
	memset(&state, 0, sizeof(state));

	if ((i = inet_aton(target, &options.target)) != 1) {
		if (i < 0)
			err(EX_OSERR, "inet_aton() failed");
		logmsg(llDEBUG, "Resolving hostname\n");
		if ((host = gethostbyname(target)) == NULL)
			errx(EX_NOHOST, hstrerror(h_errno));
		memcpy(&options.target, (struct in_addr *)host->h_addr,
		    sizeof(options.target));
	}
	state.dest_ip = options.target;

	if (!(iface = options.interface))
		iface = target_iface(options.target);
	if (pcap == NULL || strcmp(pcap_iface, iface) != 0) {
		free(pcap_iface);
		pcap_iface = strdup(iface);
		pcap_init(pcap_iface);
	}

	logmsg(llNORMAL, "Starting %s trace to %s\n",
	    probes.first == probes.last ?
		probe_desc(probes.first) :  "Multi-Probe",
	    inet_ntoa(options.target));

	state.ttl = options.initial_hop;
	probe = probes.first;

	while (!state.complete && state.ttl <= options.max_hop) {
		state.attempt = 1;

		if (sigsetjmp(jmpbuf, 1)) {

			if ((probe = probe->next) == NULL) {
				probe = probes.first;
				if (++state.attempt > options.probes) {
					display_hop(state.ttl, NULL, -1, "Timeout");
					state.ttl++;
					continue;
				}
			}
		}
		canjump = 1;

		state.ip_id = rand() % 0xffff;

		if ((options.specified & SPEC_SRC_PORT) != 0)
			state.src_port = options.source_port;
		else
			state.src_port = 1024 + (rand() % (0xffff - 1024));

		if ((options.specified & SPEC_TCP_SEQ) != 0)
			state.tcp_seq = options.tcp_seq;
		else
			state.tcp_seq = rand() % 0xffffffff;

		state.tcp_ack = rand() % 0xffffffff;

		memset(&tout, 0, sizeof(tout));
		tout.it_value.tv_sec = (options.timeout / 1000);
		tout.it_value.tv_usec = (options.timeout % 1000) * 1000;

		send_packet(probe, &state);


		while(1) {
			if (tout.it_value.tv_sec == 0 && tout.it_value.tv_usec == 0) {
				logmsg(llDEBUG, "Manually sending SIGALRM\n");
				if (kill(getpid(), SIGALRM) != 0)
					errx(EX_OSERR, "Unable to send SIGALRM");
			}


			logmsg(llINSANE, "Enabling timer (%d.%d)\n",
			    tout.it_value.tv_sec, tout.it_value.tv_usec);
			if (setitimer(ITIMER_REAL, &tout, NULL) != 0)
				err(EX_OSERR, "Unable to set timer!");


			while ((rbuf = pcap_next(pcap, &hdr)) == NULL);
			logmsg(llDEBUG, "Received a packet (%d bytes)\n",
			    hdr.caplen - datalink_len);
			if (hdr.caplen != hdr.len)
				warnx("Only captured %d bytes of a %d byte packet!",
				    hdr.caplen, hdr.len);

			if (setitimer(ITIMER_REAL, &timeroff, &tout) != 0)
				err(EX_OSERR, "Unable to pause timer!");
			logmsg(llINSANE, "Timer paused (%d.%d)\n",
			    tout.it_value.tv_sec, tout.it_value.tv_usec);

			if (process_packet(probe, &state, rbuf, hdr.caplen))
				break;
		}

		canjump = 0;
		logmsg(llINSANE, "packet read loop exited normally\n");

		if (setitimer(ITIMER_REAL, &timeroff, NULL) != 0)
			err(EX_OSERR, "Unable to disable timer!");
	}

}
