## Client-side HTTP for GET, POST, and HEAD commands.
#
# RCS: @(#) $Id: http.tcl,v 1.43 2002/10/03 13:34:32 dkf Exp $
# $Id: MyHttp.tcl,v 1.32.2.3 2004/03/09 18:28:23 [Xp-AvR] Exp $

namespace eval ::Http {

  variable http

  array set http {
    -accept */*
    -proxyhost {}
    -proxyport {}
    -useragent {Tcl http client package 2.3}
    -proxyfilter ProxyRequired
  }

  variable urlTypes
  array set urlTypes {
    http {80 ::socket}
  }

  proc Finish { token {errormsg ""} {skipCB 0} } {
    variable $token
    upvar 0 $token state
    global errorInfo errorCode

    if {[string length $errormsg] != 0} then {
      set state(error) [list $errormsg $errorInfo $errorCode]
      set state(status) error
    }
    catch {close $state(sock)}
    catch {after cancel $state(after)}
    if {[info exists state(-command)] && !$skipCB} then {
      if {[catch {eval $state(-command) {$token}} err]} then {
        if {[string length $errormsg] == 0} then {
          set state(error) [list $err $errorInfo $errorCode]
          set state(status) error
        }
      }
      if {[info exist state(-command)]} then {
        unset state(-command)
      }
    }
  } ;# Finish

  proc Reset { token {why reset} } {
    variable $token
    upvar 0 $token state
    set state(status) $why

    catch {fileevent $state(sock) readable {}}
    catch {fileevent $state(sock) writable {}}
    Finish $token
    if {[info exists state(error)]} then {
      set errorlist $state(error)
      unset state
      eval error $errorlist
    }
  } ;# Finish

  proc GetUrl { url args } {
    variable http
    variable urlTypes

    if {![info exists http(uid)]} then {
      set http(uid) 0
    }
    set token [namespace current]::[incr http(uid)]
    variable $token
    upvar 0 $token state
    Reset $token

    array set state {
      -blocksize      8192
      -queryblocksize 8192
      -validate       0
      -headers 	      {}
      -timeout 	      0
      -type           application
      -queryprogress  {}
      state	      header
      meta	      {}
      currentsize     0
      totalsize	      0
      querylength     0
      queryoffset     0
      type            application
      body            {}
      status	      ""
      http            ""
    }
    set options {-blocksize -channel -command -handler -headers -progress -validate -timeout -type}
    set usage [join $options ", "]
    regsub -all -- - $options {} options
    set pat ^-([join $options |])$
    foreach {flag value} $args {
      if {[regexp $pat $flag]} then {
        if {[info exists state($flag)] && [string is integer -strict $state($flag)] && ![string is integer -strict $value]} then {
          unset $token
          return -code error "Bad value for $flag ($value), must be integer"
        }
        set state($flag) $value
      } else {
        unset $token
        return -code error "Unknown option $flag, can be: $usage"
      }
    }

    if {![regexp -nocase {^(([^:]*)://)?([^/:]+)(:([0-9]+))?(/.*)?$} $url x prefix proto host y port srvurl]} then {
      unset $token
      return -code error "Unsupported URL: $url"
    }
    if {[string length $proto] == 0} then {
     set proto http
     set url ${proto}://$url
    }
    if {![info exists urlTypes($proto)]} then {
      unset $token
      return -code error "Unsupported URL type \"$proto\""
    }
    set defport [lindex $urlTypes($proto) 0]
    set defcmd [lindex $urlTypes($proto) 1]

    if {[string length $port] == 0} then {
      set port $defport
    }
    if {[string length $srvurl] == 0} then {
      set srvurl /
    }
    if {[string length $proto] == 0} then {
      set url http://$url
    }
    set state(url) $url
    if {![catch {$http(-proxyfilter) $host} proxy]} then {
     set phost [lindex $proxy 0]
     set pport [lindex $proxy 1]
    }

    if {$state(-timeout) > 0} then {
      set state(after) [after $state(-timeout) [list Http::Reset $token timeout]]
      set async -async
    } else {
      set async ""
    }

    if {[info exists phost] && [string length $phost]} then {
      set srvurl $url
      set conStat [catch {eval $defcmd $async {$phost $pport}} s]
    } else {
      set conStat [catch {eval $defcmd $async {$host $port}} s]
    }
    if {$conStat} then {
     Finish $token "" 1
     Cleanup $token
     return -code error $s
    }
    set state(sock) $s

    if {$state(-timeout) > 0} then {
      fileevent $s writable [list ::Http::Connect $token]
      Http::Wait $token
      if {[string equal $state(status) "error"]} then {
        set err [lindex $state(error) 0]
        Cleanup $token
        return -code error $err
      } elseif {![string equal $state(status) "connect"]} then {
        return $token
      }
      set state(status) ""
    }

    fconfigure $s -translation {auto crlf} -buffersize $state(-blocksize)

    catch {fconfigure $s -blocking off}
    set how GET
    if {$state(-validate)} then {
       set how HEAD
    }

    if {[catch {
      puts $s "$how $srvurl HTTP/1.0"
      puts $s "Accept: $http(-accept)"
      puts $s "Host: $host"
      puts $s "User-Agent: $http(-useragent)"
      foreach {key value} $state(-headers) {
        regsub -all \[\n\r\]  $value {} value
        set key [string trim $key]
        if {[string equal $key "Content-Length"]} then {
          set contDone 1
          set state(querylength) $value
        }
        if {[string length $key]} then {
          puts $s "$key: $value"
        }
      }

      puts $s ""
      flush $s
      fileevent $s readable [list ::Http::Event $token]

      if {![info exists state(-command)]} then {
        Wait $token
        if {[string equal $state(status) "error"]} then {
          return -code error [lindex $state(error) 0]
        }
      }
    } err]} then {
      if {[string equal $state(status) "error"]} then {
        Finish $token $err 1
      }
      Cleanup $token
      return -code error $err
    }

    return $token
  } ;# GetUrl

  proc Data {token} {
    variable $token
    upvar 0 $token state
    return $state(body)
  } ;# Data

  proc Ncode {token} {
    variable $token
    upvar 0 $token state
    if {[regexp {[0-9]{3}} $state(http) numeric_code]} then {
      return $numeric_code
    } else {
      return $state(http)
    }
  } ;# Ncode

  proc Cleanup {token} {
    variable $token
    upvar 0 $token state
    if {[info exist state]} then {
      unset state
    }
  } ;# Cleanup

  proc Size {token} {
    variable $token
    upvar 0 $token state
    return $state(currentsize)
  } ;# Size

  proc Connect {token} {
    variable $token
    upvar 0 $token state
    global errorInfo errorCode
    if {[eof $state(sock)] || [string length [fconfigure $state(sock) -error]]} then {
      Finish $token "connect failed [fconfigure $state(sock) -error]" 1
    } else {
      set state(status) connect
      fileevent $state(sock) writable {}
    }
    return
  } ;# Connect

  proc Event {token} {
    variable $token
    upvar 0 $token state
    set s $state(sock)

    if {[eof $s]} then {
      Eof $token
      return
    }
    if {[string equal $state(state) "header"]} then {
      if {[catch {gets $s line} n]} then {
        Finish $token $n
      } elseif {$n == 0} then {
        set state(state) body
        if {![regexp -nocase ^text $state(type)]} then {
          fconfigure $s -translation binary
          if {[info exists state(-channel)]} then {
            fconfigure $state(-channel) -translation binary
          }
        }
      } elseif {$n > 0} then {
        if {[regexp -nocase {^content-type:(.+)$} $line x type]} then {
          if {(([info exist ::libraries::FileType]) && ($::libraries::FileType != ""))} then {
            set state(type) $::libraries::FileType
          } else {
            set state(type) [string trim $type]
          }
        }
        if {[regexp -nocase {^content-length:(.+)$} $line x length]} then {
          set state(totalsize) [string trim $length]
        }
        if {[regexp -nocase {^([^:]+):(.+)$} $line x key value]} then {
          lappend state(meta) $key [string trim $value]
        } elseif {[regexp ^HTTP $line]} then {
          set state(http) $line
        }
      }
    } else {
      if {[catch {
        if {[info exists state(-handler)]} then {
          set n [eval $state(-handler) {$s $token}]
        } else {
          set block [read $s $state(-blocksize)]
          set n [string length $block]
          if {$n >= 0} then {
            append state(body) $block
          }
        }
        if {$n >= 0} then {
          incr state(currentsize) $n
        }
      } err]} then {
        Finish $token $err
      } else {
        if {[info exists state(-progress)]} then {
          eval $state(-progress) {$token $state(totalsize) $state(currentsize)}
        }
      }
    }
  } ;# Event

  proc Eof {token} {
    variable $token
    upvar 0 $token state
    if {[string equal $state(state) "header"]} then {
      set state(status) eof
    } else {
      set state(status) ok
    }
    set state(state) eof
    Finish $token
  }

  proc Wait {token} {
    variable $token
    upvar 0 $token state

    if {![info exists state(status)] || [string length $state(status)] == 0} then {
      vwait $token\(status)
    }

    return $state(status)
  } ;# Wait

  proc ProxyRequired {host} {
    variable http
    if {[info exists http(-proxyhost)] && [string length $http(-proxyhost)]} then {
      if {![info exists http(-proxyport)] || ![string length $http(-proxyport)]} then {
        set http(-proxyport) 8080
      }
      return [list $http(-proxyhost) $http(-proxyport)]
    } else {
     return {}
    }
  } ;# ProxyRequired

} ;# namespace eval ::Http

