/*
 * uf_features.c -- part of share.mod
 *
 * $Id: uf_features.c,v 1.13 (1.0.3) 2004/05/07 18:55:15 wcc Exp $
 */

typedef struct uff_list_struct {
  struct uff_list_struct *next; /* Pointer to next entry                */
  struct uff_list_struct *prev; /* Pointer to previous entry            */
  uff_table_t *entry;
} uff_list_t;

typedef struct {
  uff_list_t *start;
  uff_list_t *end;
} uff_head_t;

static uff_head_t uff_list;
static char uff_sbuf[512];

static void uff_init(void)
{
  EvangelineBzero(&uff_list, sizeof(uff_head_t));
}

static int uff_expmem(void)
{
  uff_list_t *ul;
  int tot = 0;

  for (ul = uff_list.start; ul; ul = ul->next)
    tot += sizeof(uff_list_t);
  return tot;
}

/* Search for a feature in the uff feature list that matches a supplied
 * feature flag. Returns a pointer to the entry in the list or NULL if
 * no feature uses the flag. */
static uff_list_t *uff_findentry_byflag(int flag)
{
  uff_list_t *ul;

  for (ul = uff_list.start; ul; ul = ul->next)
    if (ul->entry->flag & flag)
      return ul;
  return NULL;
}

/* Search for a feature in the uff feature list. Returns a pointer to the
 * entry in the list or NULL if no such feature exists. */
static uff_list_t *uff_findentry_byname(char *feature)
{
  uff_list_t *ul;

  for (ul = uff_list.start; ul; ul = ul->next)
    if (!strcmp(ul->entry->feature, feature))
      return ul;
  return NULL;
}

/* Insert entry into sorted list. */
static void uff_insert_entry(uff_list_t *nul)
{
  uff_list_t *ul, *lul = NULL;

  ul = uff_list.start;
  while (ul && ul->entry->priority < nul->entry->priority) {
    lul = ul;
    ul = ul->next;
  }

  nul->prev = NULL;
  nul->next = NULL;
  if (lul) {
    if (lul->next)
      lul->next->prev = nul;
    nul->next = lul->next;
    nul->prev = lul;
    lul->next = nul;
  } else if (ul) {
    uff_list.start->prev = nul;
    nul->next = uff_list.start;
    uff_list.start = nul;
  } else
    uff_list.start = nul;
  if (!nul->next)
    uff_list.end = nul;
}

/* Remove entry from sorted list. */
static void uff_remove_entry(uff_list_t *ul)
{
  if (!ul->next)
    uff_list.end = ul->prev;
  else
    ul->next->prev = ul->prev;
  if (!ul->prev)
    uff_list.start = ul->next;
  else
    ul->prev->next = ul->next;
}

/* Add a single feature to the list. */
static void uff_addfeature(uff_table_t *ut)
{
  uff_list_t *ul;

  if (uff_findentry_byname(ut->feature)) {
    putlog(LOG_MISC, "*", "(!) share: same feature name used twice: %s",
           ut->feature);
    return;
  }
  ul = uff_findentry_byflag(ut->flag);
  if (ul) {
    putlog(LOG_MISC, "*", "(!) share: feature flag %d used twice by %s and %s",
           ut->flag, ut->feature, ul->entry->feature);
    return;
  }
  ul = nmalloc(sizeof(uff_list_t));
  ul->entry = ut;
  uff_insert_entry(ul);
}

/* Add a complete table to the list. */
static void uff_addtable(uff_table_t *ut)
{
  if (!ut)
    return;
  for (; ut->feature; ++ut)
    uff_addfeature(ut);
}

/* Remove a single feature from the list. */
static int uff_delfeature(uff_table_t *ut)
{
  uff_list_t *ul;

  for (ul = uff_list.start; ul; ul = ul->next)
    if (!strcmp(ul->entry->feature, ut->feature)) {
      uff_remove_entry(ul);
      nfree(ul);
      return 1;
    }
  return 0;
}

/* Remove a complete table from the list. */
static void uff_deltable(uff_table_t *ut)
{
  if (!ut)
    return;
  for (; ut->feature; ++ut)
    (int) uff_delfeature(ut);
}

/* Parse the given features string, set internal flags apropriately and
 * eventually respond with all features we will use. */
static void uf_features_parse(int idx, char *par)
{
  char *buf, *s, *p;
  uff_list_t *ul;

  uff_sbuf[0] = 0;
  p = s = buf = nmalloc(strlen(par) + 1);
  strcpy(buf, par);

  dcc[idx].u.bot->uff_flags = 0;

  while ((s = strchr(s, ' ')) != NULL) {
    *s = '\0';

    ul = uff_findentry_byname(p);
    if (ul && (ul->entry->ask_func == NULL || ul->entry->ask_func(idx))) {
      dcc[idx].u.bot->uff_flags |= ul->entry->flag;
      strcat(uff_sbuf, ul->entry->feature);
      strcat(uff_sbuf, " ");
      if (!strncmp(p, "encrypt", 7)) {
        encrypt_userfile = 1;
        debug0("uf_features.c->uf_features_parse) Found 'encrypt' feature, enabling");
      }
    }
    p = ++s;
  }
  nfree(buf);

  if (uff_sbuf[0]) {
    dprintf(idx, "s feats %s\n", uff_sbuf);
  }
}

/* Return a list of features we are supporting. */
static char *uf_features_dump(int idx)
{
  uff_list_t *ul;

  uff_sbuf[0] = 0;
  for (ul = uff_list.start; ul; ul = ul->next)
    if (ul->entry->ask_func == NULL || ul->entry->ask_func(idx)) {
      strcat(uff_sbuf, ul->entry->feature);
      strcat(uff_sbuf, " ");
    }
  return uff_sbuf;
}

static int uf_features_check(int idx, char *par)
{
  char *buf, *s, *p;
  uff_list_t *ul;

  uff_sbuf[0] = 0;
  p = s = buf = nmalloc(strlen(par) + 1);
  strcpy(buf, par);

  dcc[idx].u.bot->uff_flags = 0;

  while ((s = strchr(s, ' ')) != NULL) {
    *s = '\0';

    ul = uff_findentry_byname(p);
    if (ul && (ul->entry->ask_func == NULL || ul->entry->ask_func(idx))) {
      dcc[idx].u.bot->uff_flags |= ul->entry->flag;
    } else {
      putlog(LOG_BOTS, "*", "Bot %s tried unsupported feature!", dcc[idx].nick);
      dprintf(idx, "s e Attempt to use an unsupported feature\n");
      zapfbot(idx);

      nfree(buf);
      return 0;
    }
    p = ++s;
  }
  nfree(buf);
  return 1;
}

/* Call all active feature functions, sorted by their priority. This
 * should be called when we're about to send a user file. */
static int uff_call_sending(int idx, char *user_file)
{
  uff_list_t *ul;

  for (ul = uff_list.start; ul; ul = ul->next)
    if (ul->entry && ul->entry->snd &&
        (dcc[idx].u.bot->uff_flags & ul->entry->flag))
      if (!(ul->entry->snd(idx, user_file)))
        return 0;
  return 1;
}

/* Call all active feature functions, sorted by their priority. This
 * should be called when we've received a user file and are about to
 * parse it. */
static int uff_call_receiving(int idx, char *user_file)
{
  uff_list_t *ul;

  for (ul = uff_list.end; ul; ul = ul->prev)
    if (ul->entry && ul->entry->rcv &&
        (dcc[idx].u.bot->uff_flags & ul->entry->flag))
      if (!(ul->entry->rcv(idx, user_file)))
        return 0;
  return 1;
}

/* Feature `overbots' */
static int uff_ask_override_bots(int idx)
{
  if (overr_local_bots)
    return 1;
  else
    return 0;
}

/* Feature `compress` */
#ifndef ZLIB_PROBLEM
int compress_level = 9;

static int uff_comp(int idx, char *filename)
{
  debug1("(uf_features.c->uff_comp) Compressing user file for %s. (zlib)", dcc[idx].nick);
  return compress_file(filename, compress_level);
}

static int uff_uncomp(int idx, char *filename)
{
  debug1("(uf_features.c->uff_uncomp) Uncompressing user file from %s. (zlib)", dcc[idx].nick);
  return uncompress_file(filename);
}

static int uff_ask_compress(int idx)
{
  if (share_compressed)
    return 1;
  else
    return 0;
}
#endif

/* Feature `encrypt` */
static int uff_writeenc(int idx, char *fname);
static int uff_readenc(int idx, char *fname);
static int is_encryptedfile(char *fname);

static void share_cryptKey(int idx, char *par) {
  char *ki;
  ki = newsplit(&par);
  debug2("(uf_features.c->share_cryptKey) Got shareKey from %s: %s", dcc[idx].nick, ki);
  strcpy(shareKey, ki);
}

static void share_md5sum(int idx, char *par) {
  char *ki;
  ki = newsplit(&par);
  debug2("(uf_features.c->share_md5sum) Got md5sum from %s: %s", dcc[idx].nick, ki);
  strcpy(md5sum, ki);
}

static int is_encryptedfile(char *fname) {
  FILE *file;
  char buffer[512];

  file = fopen(fname, "r");
  if (file == NULL)
    return 0;
  fgets(buffer, 511, file);
  if (!strncmp(buffer, "#Evangeline!Encrypted: Userfile", 31))
    return 1;
  else
    return 0;
}

static int uff_writeenc(int idx, char *fname) {

  debug2("(uf_features.c->uff_writeenc) Sending shareKey to %s: %s", dcc[idx].nick, shareKey);
  dprintf(idx, "s sck %s\n", shareKey);
  putlog(LOG_BOTS, "*", "Encrypting userfile for %s", dcc[idx].nick);

  return 1;
}

static int uff_readenc(int idx, char *fname) {

  if (is_encryptedfile(fname)) {
    encrypt_userfile = 1;
    putlog(LOG_BOTS, "*", "Decrypting userfile from %s", dcc[idx].nick);
  }
  return 1;
}

static int uff_ask_sharecrypt(int idx) {
  if (share_encrypted)
    return 1;
  else
    return 0;
}

static uff_table_t internal_uff_table[] = {
  {"overbots", UFF_OVERRIDE, uff_ask_override_bots, 0, NULL, NULL},
  {"invites",  UFF_INVITE,   NULL,                  0, NULL, NULL},
  {"exempts",  UFF_EXEMPT,   NULL,                  0, NULL, NULL},
#ifndef ZLIB_PROBLEM
  {"compress", UFF_COMPRESS, uff_ask_compress, 	  100, uff_comp, uff_uncomp},
#endif
  {"encrypt",  UFF_ENCRYPT,  uff_ask_sharecrypt,  100, uff_writeenc, uff_readenc},
  {NULL,       0,            NULL,                  0, NULL, NULL}
};
