/*
 * inet_ntop.c --
 *   provides inet_ntop()
 *
 * $Id: inet_ntop.c, v 1.4 2003/11/28 20:01:23 [Xp-AvR] Exp $
 */

#include "main.h"
#include "inet_ntop.h"

#if defined(USE_IPV6) && !defined(HAVE_INET_NTOP)

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>

#ifdef SPRINTF_CHAR
# define SPRINTF(x) strlen(sprintf/**/x)
#else
# define SPRINTF(x) ((size_t)sprintf x)
#endif

#define NS_INADDRSZ     4
#define NS_IN6ADDRSZ    16
#define NS_INT16SZ      2

static const char *EvangelineInet_ntop4 (const u_char *src, char *dst, socklen_t size);
static const char *EvangelineInet_ntop6 (const u_char *src, char *dst, socklen_t size);

const char *EvangelineInet_ntop(af, src, dst, size)
	int af;
	const void *src;
	char *dst;
	socklen_t size;
{
	switch (af) {
	case AF_INET:
		return (EvangelineInet_ntop4(src, dst, size));
	case AF_INET6:
		return (EvangelineInet_ntop6(src, dst, size));
	default:
		return (NULL);
	}
}

static const char *EvangelineInet_ntop4(src, dst, size)
	const u_char *src;
	char *dst;
	socklen_t size;
{
	static const char fmt[] = "%u.%u.%u.%u";
	char tmp[sizeof "255.255.255.255"];

	if (SPRINTF((tmp, fmt, src[0], src[1], src[2], src[3])) > size) {
		return (NULL);
	}
	return strcpy(dst, tmp);
}

static const char *EvangelineInet_ntop6(src, dst, size)
	const u_char *src;
	char *dst;
	socklen_t size;
{
	char tmp[sizeof "ffff:ffff:ffff:ffff:ffff:ffff:255.255.255.255"], *tp;
	struct { int base, len; } best, cur;
	u_int words[NS_IN6ADDRSZ / NS_INT16SZ];
	int i;

	EvangelineMemset(words, '\0', sizeof words);
	for (i = 0; i < NS_IN6ADDRSZ; i += 2)
		words[i / 2] = (src[i] << 8) | src[i + 1];
	best.base = -1;
	cur.base = -1;
	for (i = 0; i < (NS_IN6ADDRSZ / NS_INT16SZ); i++) {
		if (words[i] == 0) {
			if (cur.base == -1)
				cur.base = i, cur.len = 1;
			else
				cur.len++;
		} else {
			if (cur.base != -1) {
				if (best.base == -1 || cur.len > best.len)
					best = cur;
				cur.base = -1;
			}
		}
	}
	if (cur.base != -1) {
		if (best.base == -1 || cur.len > best.len)
			best = cur;
	}
	if (best.base != -1 && best.len < 2)
		best.base = -1;

	tp = tmp;
	for (i = 0; i < (NS_IN6ADDRSZ / NS_INT16SZ); i++) {
		if (best.base != -1 && i >= best.base &&
		    i < (best.base + best.len)) {
			if (i == best.base)
				*tp++ = ':';
			continue;
		}
		if (i != 0)
			*tp++ = ':';
		if (i == 6 && best.base == 0 &&
		    (best.len == 6 || (best.len == 5 && words[5] == 0xffff))) {
			if (!EvangelineInet_ntop4(src+12, tp, sizeof tmp - (tp - tmp)))
				return (NULL);
			tp += strlen(tp);
			break;
		}
		tp += SPRINTF((tp, "%x", words[i]));
	}
	if (best.base != -1 && (best.base + best.len) ==
	    (NS_IN6ADDRSZ / NS_INT16SZ))
		*tp++ = ':';
	*tp++ = '\0';

	if ((socklen_t)(tp - tmp) > size) {
		return (NULL);
	}
	return strcpy(dst, tmp);
}
#endif /* !HAVE_INET_NTOP */
