/*
 * compress.c
 *   uses the compression library libz to compress and uncompress files
 *
 * $Id: compress.c,v 1.22 (1.0.1) 2004/04/28 14:01:43 wcc Exp $
 */

#include <errno.h>
#include "main.h"

#ifndef NEEDTOUSEMYZLIB
  #ifndef ZLIB_PROBLEM
    #include <zlib.h>
  #endif
#endif

#ifdef NEEDTOUSEMYZLIB
  #ifndef ZLIB_PROBLEM
    #include "libs/zlib/zlib.h"
  #endif
#endif

#ifdef HAVE_MMAP
#  undef panic
#  include <sys/types.h>
#  include <sys/mman.h>
#  include <sys/stat.h>
#endif /* HAVE_MMAP */
#include "compress.h"

#define BUFLEN 512
#include "tclcompress.c"

void putlog EVANGELINE_VARARGS(int, arg1);
void *n_malloc(int, const char *, int);
void n_free(void *, const char *, int);
void make_rand_str(char *, int);
int is_file(const char *);
int movefile(char *, char *);

int is_compressedfile(char *filename)
{
#ifndef ZLIB_PROBLEM
  char buf1[50], buf2[50];
  FILE *fin;
  register int len1, len2, i;

  EvangelineMemset(buf1, 0, 50);
  EvangelineMemset(buf2, 0, 50);
  if (!is_file(filename))
    return COMPF_FAILED;

  fin = gzopen(filename, "rb");
  if (!fin)
    return COMPF_FAILED;
  len1 = gzread(fin, buf1, sizeof(buf1));
  if (len1 < 0)
    return COMPF_FAILED;
  if (gzclose(fin) != Z_OK)
    return COMPF_FAILED;

  fin = fopen(filename, "rb");
  if (!fin)
    return COMPF_FAILED;
  len2 = fread(buf2, 1, sizeof(buf2), fin);
  if (ferror(fin))
    return COMPF_FAILED;
  fclose(fin);

  if (len1 != len2)
    return COMPF_COMPRESSED;
  for (i = 0; i < sizeof(buf1); i++)
    if (buf1[i] != buf2[i])
      return COMPF_COMPRESSED;
  return COMPF_UNCOMPRESSED;
#else
  return 0;
#endif
}

/* Uncompresses a file `f_src' and saves it as `f_target'. */
int uncompress_to_file(char *f_src, char *f_target)
{
#ifndef ZLIB_PROBLEM
  char buf[BUFLEN];
  int len;
  FILE *fin, *fout;

  if (!is_file(f_src)) {
    putlog(LOG_MISC, "*", "Failed to uncompress file `%s': not a file.", f_src);
    return COMPF_ERROR;
  }
  fin = gzopen(f_src, "rb");
  if (!fin) {
    putlog(LOG_MISC, "*", "Failed to uncompress file `%s': gzopen failed.",
           f_src);
    return COMPF_ERROR;
  }

  fout = fopen(f_target, "wb");
  if (!fout) {
    putlog(LOG_MISC, "*", "Failed to uncompress file `%s': open failed: %s.",
           f_src, strerror(errno));
    return COMPF_ERROR;
  }

  while (1) {
    len = gzread(fin, buf, sizeof(buf));
    if (len < 0) {
      putlog(LOG_MISC, "*", "Failed to uncompress file `%s': gzread failed.",
             f_src);
      return COMPF_ERROR;
    }
    if (!len)
      break;
    if ((int) fwrite(buf, 1, (unsigned int) len, fout) != len) {
      putlog(LOG_MISC, "*", "Failed to uncompress file `%s': fwrite "
             "failed: %s.", f_src, strerror(errno));
      return COMPF_ERROR;
    }
  }
  if (fclose(fout)) {
    putlog(LOG_MISC, "*", "Failed to uncompress file `%s': fclose failed: %s.",
           f_src, strerror(errno));
    return COMPF_ERROR;
  }
  if (gzclose(fin) != Z_OK) {
    putlog(LOG_MISC, "*", "Failed to uncompress file `%s': gzclose failed.",
           f_src);
    return COMPF_ERROR;
  }
  return COMPF_SUCCESS;
#else
  return 0;
#endif
}

/* Enforce limits. */
inline static void adjust_mode_num(int *mode)
{
  if (*mode > 9)
    *mode = 9;
  else if (*mode < 0)
    *mode = 0;
}

#ifdef HAVE_MMAP
/* Attempt to compress in one go, by mmap'ing the file to memory. */
int compress_to_file_mmap(FILE *fout, FILE *fin)
{
#ifndef ZLIB_PROBLEM
  int len, ifd = fileno(fin);
  char *buf;
  struct stat st;

  if (fstat(ifd, &st) < 0)
    return COMPF_ERROR;
  if (st.st_size <= 0)
    return COMPF_ERROR;

  buf = mmap(0, st.st_size, PROT_READ, MAP_SHARED, ifd, 0);
  if (buf < 0)
    return COMPF_ERROR;

  len = gzwrite(fout, buf, st.st_size);
  if (len != (int) st.st_size)
    return COMPF_ERROR;

  munmap(buf, st.st_size);
  fclose(fin);
  if (gzclose(fout) != Z_OK)
    return COMPF_ERROR;
  return COMPF_SUCCESS;
#else
  return 0;
#endif
}
#endif /* HAVE_MMAP */

/* Compresses a file `f_src' and saves it as `f_target'. */
int compress_to_file(char *f_src, char *f_target, int mode_num)
{
#ifndef ZLIB_PROBLEM
  char buf[BUFLEN], mode[5];
  FILE *fin, *fout;
  int len;

  adjust_mode_num(&mode_num);
  EvangelineSnprintf(mode, sizeof mode, "wb%d", mode_num);

  if (!is_file(f_src)) {
    putlog(LOG_MISC, "*", "Failed to compress file `%s': not a file.", f_src);
    return COMPF_ERROR;
  }
  fin = fopen(f_src, "rb");
  if (!fin) {
    putlog(LOG_MISC, "*", "Failed to compress file `%s': open failed: %s.",
           f_src, strerror(errno));
    return COMPF_ERROR;
  }

  fout = gzopen(f_target, mode);
  if (!fout) {
    putlog(LOG_MISC, "*", "Failed to compress file `%s': gzopen failed.",
           f_src);
    return COMPF_ERROR;
  }

#ifdef HAVE_MMAP
  if (compress_to_file_mmap(fout, fin) == COMPF_SUCCESS) {
    return COMPF_SUCCESS;
  } else {
    gzclose(fout);
    fout = gzopen(f_target, mode);
  }
#endif /* HAVE_MMAP */

  while (1) {
    len = fread(buf, 1, sizeof(buf), fin);
    if (ferror(fin)) {
      putlog(LOG_MISC, "*", "Failed to compress file `%s': fread failed: %s",
             f_src, strerror(errno));
      return COMPF_ERROR;
    }
    if (!len)
      break;
    if (gzwrite(fout, buf, (unsigned int) len) != len) {
      putlog(LOG_MISC, "*", "Failed to compress file `%s': gzwrite failed.",
             f_src);
      return COMPF_ERROR;
    }
  }
  fclose(fin);
  if (gzclose(fout) != Z_OK) {
    putlog(LOG_MISC, "*", "Failed to compress file `%s': gzclose failed.",
           f_src);
    return COMPF_ERROR;
  }
  return COMPF_SUCCESS;
#else
  return 0;
#endif
}

/* Compresses a file `filename' and saves it as `filename'. */
int compress_file(char *filename, int mode_num)
{
#ifndef ZLIB_PROBLEM
  char *temp_fn, randstr[5];
  int ret;

  temp_fn = nmalloc(strlen(filename) + 5);
  make_rand_str(randstr, 4);
  strcpy(temp_fn, filename);
  strcat(temp_fn, randstr);

  ret = compress_to_file(filename, temp_fn, mode_num);

  if (ret == COMPF_SUCCESS)
    movefile(temp_fn, filename);

  nfree(temp_fn);
  return ret;
#else
  return 0;
#endif
}

/* Uncompresses a file `filename' and saves it as `filename'. */
int uncompress_file(char *filename)
{
#ifndef ZLIB_PROBLEM
  char *temp_fn, randstr[5];
  int ret;

  temp_fn = nmalloc(strlen(filename) + 5);
  make_rand_str(randstr, 4);
  strcpy(temp_fn, filename);
  strcat(temp_fn, randstr);

  ret = uncompress_to_file(filename, temp_fn);

  if (ret == COMPF_SUCCESS)
    movefile(temp_fn, filename);

  nfree(temp_fn);
  return ret;
#else
  return 0;
#endif
}
