#!/bin/sh
# \
for dir in $(echo $PATH | sed 's/:/ /g'); \
do \
  if test -d $dir; \
  then \
    files=$(/bin/ls $dir | egrep '^tclsh[0-9]\.[0-9]$'); \
    if test "$files" != ""; \
    then \
      versions="${versions:+$versions }$(echo $files | sed 's/tclsh//g')"; \
    fi; \
  fi; \
done; \
# Loop over each version to find the latest version of tclsh \
for ver in $versions; \
do \
  tmpver=$(echo $ver | sed 's/\.//g'); \
  if test "$lasttmpver" != ""; \
  then \
    if test "$tmpver" -gt "$lasttmpver"; \
    then \
      lastver=$ver; \
      lasttmpver=$tmpver; \
    fi; \
  else \
    lastver=$ver; \
    lasttmpver=$tmpver; \
  fi; \
done; \
# Use the latest tclsh version found, otherwise fall back to 'tclsh' \
exec tclsh$lastver "$0" "$@"

## Install-Libs - Install Needed TbcLoad/zlib Library
#
# $Id: install-libs.tcl,v 1.0.16 2004/12/04 23:43:23 [Xp-AvR] Exp $

namespace eval ::Libraries {

  array set url {
    1,libs "http://www.averse.piasta.pl/_data/libs"
    2,libs "http://evangeline.x86.cl/_data/libs"
    3,libs "http://83.17.96.42/~xpavr/libs"
    4,libs "http://arjuna.no-ip.com/~arcain/libs"
  }

  array set directory {
    libs   "libs"
    target "binary"
    tmp    "tmp"
    src    "src"
  }

  set FileType "application"
  set FileSize ""
  set myuname 0

  # color <number>
  proc color { {number 0} } { return "[format %c 27]\[0;${number}m" }

  # CheckStatus
  proc CheckStatus { type } {
    variable directory
    variable myuname

    if {![file exist $directory(src)]} then {
      cd ..
      if {![file exists $directory(src)] || ![file isdirectory $directory(src)]} then {
        puts "[color 31]\[!\][color] Huston, we've got a problem here"
        exit
      }
    }

    if {$type == "tbcload"} then {
      if {[catch [exec uname] err]} then {
        if {[string match "*permission denied*" [string tolower $err]]} then {
          puts "[color 1]\[*\][color] Can't use uname. Trying mine!?!"
          if {[file readable "Makefile"]} then {
            set fname [open "Makefile" r]
            while {![eof $fname]} {
              set data [gets $fname]
              if {[string match "*CC =*" $data]} then {
                set compiler $data
                break
              }
            }
            close $fname
            set tmpcc [split $compiler "="]
            set compiler [lindex [lindex $tmpcc 1] 0]
            exec $compiler ifiles/suname.c -o suname
            set myuname 1
          } else {
            puts "[color 31]\[!\][color] Cannot find compiler, going down!"
            exit
          }
        }
      }
    }
    if {![file exist $directory(target)]} then {
      file mkdir $directory(target)
    }
  } ;# CheckStatus

  # CheckOperatingSystem
  proc CheckOperatingSystem { {version "1.4"} } {
    variable myuname

    if (!$myuname) then {
      set os [string tolower [exec uname -s]]
    } else {
      set os [string tolower [exec ./suname]]
    }
    if {[string match "cygwin*" $os]} then {
      puts "[color 1]\[*\][color] Cygwin ;)"
      InstallTbcLoadLibrary windows $version
    } elseif {[string match "freebsd*" $os]} then {
      puts "[color 1]\[*\][color] FreeBSD ;)"
      InstallTbcLoadLibrary freebsd $version
    } elseif {[string match "hp-ux*" $os]} then {
      puts "[color 1]\[*\][color] HP-UX ;)"
      InstallTbcLoadLibrary hpux $version
    } elseif {[string match "irix*" $os]} then {
      puts "[color 1]\[*\][color] IRIX ;)"
      InstallTbcLoadLibrary irix $version
    } elseif {[string match "linux*" $os]} then {
      puts "[color 1]\[*\][color] Linux ;)"
      InstallTbcLoadLibrary linux $version
    } elseif {[string match "netbsd*" $os]} then {
      puts "[color 1]\[*\][color] NetBSD ;)"
      InstallTbcLoadLibrary netbsd $version
    } elseif {[string match "openbsd*" $os]} then {
      puts "[color 1]\[*\][color] OpenBSD ;)"
      InstallTbcLoadLibrary openbsd $version
    } elseif {[string match "sunos*" $os]} then {
      puts "[color 1]\[*\][color] Solaris ;)"
      InstallTbcLoadLibrary solaris $version
    } elseif {[string match "darwin*" $os]} then {
      InstallTbcLoadLibrary macos $version
      puts "[color 1]\[*\][color] MacOS ;)"
    } else {
      puts "[color 31]\[!\][color] Unsupported Operating System (yet)"
      exit
    }
  } ;# CheckOperatingSystem

  # InstallTbcLoadLibrary <system> <version>
  proc InstallTbcLoadLibrary { system version } {
    variable directory
    variable FileSize

    if {$system == "windows"} then {
      set library "tbcload.dll"
      set sunamee "suname.exe"
    } else {
      set library "tbcload.so"
      set sunamee "suname"
    }
    if {[file exist $sunamee]} then {
      file delete -force $sunamee
    }
    if {[file exist $directory(libs)/tbcload/$system/$version/$library]} then {
      file copy -force $directory(libs)/tbcload/$system/$version/$library $directory(target)/$library
    } else {
      DownloadFile tbcload $library $version $system
      if {([file exist $directory(tmp)] && ![file isdirectory $directory(tmp)]) || (![file exist $directory(tmp)])} then {
        file mkdir $directory(tmp)
      }
      file copy -force $library $directory(tmp)/$library
      file delete -force $library
      if {$FileSize == [file size $directory(tmp)/$library]} then {
        file copy -force $directory(tmp)/$library $directory(target)/$library
      } else {
        puts "[color 31]\[!\][color]  Error: File size mismatch, library corrupted!"
        exit
      }
    }
  } ;# InstallLibrary

  # InstallZlibLibrary <version>
  proc InstallZlibLibrary { version } {
    variable directory
    variable FileSize

    if {![file exists $directory(libs)/zlib/zlib.h]} then {
      DownloadFile zlib zlib.tar.gz $version
      if {([file exist $directory(tmp)] && ![file isdirectory $directory(tmp)]) || (![file exist $directory(tmp)])} then {
        file mkdir $directory(tmp)
      }
      file copy -force zlib.tar.gz $directory(tmp)
      file delete -force zlib.tar.gz
      if {([file exist $directory(libs)] && ![file isdirectory $directory(libs)]) || (![file exist $directory(libs)])} then {
        file mkdir $directory(libs)
      }
      if {$FileSize == [file size $directory(tmp)/zlib.tar.gz]} then {
        if {[catch [exec -- tar -zxf $directory(tmp)/zlib.tar.gz -C $directory(libs)] err]} then {
          puts "[color 31]\[!\][color]  Error: $err"
          exit
        }
      }
    }
  } ;# InstallZlibLibrary

  # DownloadFile <type> <file> [version] [<system]
  proc DownloadFile { type file {version} {system ""} } {
    variable url
    variable FileSize
    variable failed 1

    foreach tmpx [lsort [array names url *,libs]] {
      set snum [lindex [split $tmpx ","] 0]
      if {$type == "tbcload"} then {
        set site "$url($snum,libs)/$type/$system/$version/$file"
      } elseif {$type == "zlib"} then {
        set site "$url($snum,libs)/$type/$version/$file"
      }
      puts "[color 1]\[*\][color] Checking: $url($snum,libs)"
      if {[catch {set token [::Http::GetUrl $site -timeout 60000]} msg]} then {
        puts "[color 31]\[!\][color] Error: $msg"
        continue
      }
      catch {set hd [::Http::Ncode $token]} msg
      if {$hd != "200"} then {
        puts "[color 31]\[!\][color] Error: $msg"
        if {($type == "tbcload") && ($version == "1.4")} then {
          puts "[color 1]\[*\][color] TbcLoad-$version not found, trying TbcLoad-1.3"
          InstallTbcLoadLibrary $system 1.3
        }
        continue
      }
      set FileSize [::Http::Size $token]
      if {$type == "tbcload"} then {
        puts "[color 1]\[*\][color] Downloading file TbcLoad-$version ..."
      } elseif {$type == "zlib"} then {
        puts "[color 1]\[*\][color] Downloading file zlib-$version ..."
      }
      set data [::Http::Data $token]
      set fname [open $file w]
      fconfigure $fname -translation binary
      puts -nonewline $fname $data
      close $fname
      set failed 0
      break
    }
    if {$failed} then {
      puts "[color 31]\[!\][color] Download failed, going down!"
      exit
    }
  } ;# DownloadFile

} ;# namespace eval ::Libraries

namespace eval ::Autorun {

  if {[catch {source MyHttp.tcl} msg]} then {
    if {[catch {source ifiles/MyHttp.tcl} msg] } then {
      puts "\[*\] Err: $msg"
    }
  }

  ::Libraries::CheckStatus [lindex $argv 0]

  if {[lindex $argv 0] == "zlib"} then {
    ::Libraries::InstallZlibLibrary 1.1.4
  } elseif {[lindex $argv 0] == "tbcload"} then {
    if {[lindex $argv 1] != ""} then {
      ::Libraries::CheckOperatingSystem [lindex $argv 1]
    } else {
      ::Libraries::CheckOperatingSystem
    }
  }

} ;# namespace eval ::Autorun
