/*
 * dynbuff.h
 */

#ifndef __dynbuff_h
#  define __dynbuff_h

#ifdef _USE_SSL
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#endif

class db_parser;

class dynbuff 
{
private:
    char * contents;
    unsigned int  min,       /* minimum size to keep buffer */
                  max,
                  bytes_in,  /* how much data in */
                  alloced;   /* how much mem allocated */
      
public:
    dynbuff(unsigned int, unsigned int);
    ~dynbuff();

    /* SSL Support */
#ifdef _USE_SSL
    int add(SSL *, unsigned int = 0);
    int flush(SSL *, unsigned int = 0);
#endif

    /* Regular methods to add/retrieve data */
    int add(const char *, unsigned int = 0);
    int add(int fd, unsigned int = 0);
    int copy_line(unsigned int, char *, unsigned int, bool = 0, bool = 0);

    /* flush contents to a file descriptor */
    int     flush(int, unsigned int = 0);

    int     optimize(void);
    int     transfer(dynbuff *);

    unsigned int get_size(void) const    {     return bytes_in;  }
    unsigned int get_alloced(void) const {     return alloced;   }
    unsigned int flush(void)             {     return bytes_in = 0; }
    void   reset(void)             {     flush();    }

protected:
    int remove(unsigned int, unsigned int);
    int append(const char *, unsigned int);

    friend class db_parser;
};

/* Allows easy parsing of a dynbuff (line by line)
 *
 * note -- it's (very) important that you don't screw with the
 * original buffer while you're using one of these. It's not quite
 * able to handle the buffer changing from underneath it. In the
 * real world we'd have some locking mechanism or someother kind of
 * protection, but i'm too lazy to implement one right now.
 *
 * and call zap() when you've parsed all the lines u need to
 */
class db_parser
{
private:
    dynbuff * buff;
    char    * next_line_ptr;
public:
    db_parser(dynbuff *);
    char * get_next_line(bool);

    /* This effectively remove all buffer contents upto point
     *  of (next_line_ptr) */
    u_long zap(void)
    {
        return buff->remove(0, (u_long) (next_line_ptr - buff->contents));
    }
};

enum {
    DYNBUFF_ERR_READ  = -1,
    DYNBUFF_ERR_WRITE = -2,
    DYNBUFF_ERR_FULL  = -3,
    DYNBUFF_ERR_MEM   = -4,
    DYNBUFF_ERR_EMPTY = -5
};

#endif
