/*
 * ruleset.cpp
 *
 * Part of ezbounce
 *
 * (C) 1998-2002 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "autoconf.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "ruleset.h"
#include "general.h"
#include "config.h"
#include "linkedlist.h"
#include "debug.h"

bool smart_match(const char *hostname, const char *pattern); 
bool port_in_set(u_short port, const char *set);


const char ruleset::FROM      = 70;
const char ruleset::TO        = 69;
const int  ruleset::UNLIMITED = -1;


/* 
 *  Checks if port is in set 'set'. 
 */
bool port_in_set(u_short port, const char *set)
{
    int cur = 1;
    if (!set)
        return 0;
    if (strcasecmp(set, "all") == 0)
        return 1;
    /* 
     * Find tokens seperated by commas.
     * Will also get the first and only one if there are no commas.
     */
    do {
        char buff[15];
        if (!gettok(set, buff, 15, ',', cur++))
            break;
        if (strchr(buff, '-'))
        {
            char min[6], max[6];
            unsigned int _min, _max;
            /* x1-x2; get them both and compare */
            gettok(buff, min, sizeof(min), '-', 1);
            gettok(buff, max, sizeof(max), '-', 2);
            /* we are kind enough to trim down min/max if they 
               are too big */
            _min = (unsigned) atoi(min); _max = (unsigned) atoi(max);
            _min = (_min > 65536) ? 65536 : _min;
            _max = (_max > 65536) ? 65536 : _max;
            if (port >= (u_short)_min && port <= (u_short) _max)
                return 1;
        }
        else {
            if (port == u_short(atoi(buff)))
                return 1;
        }
    } while (1);
    return 0;
}

/* 
 * Checks if address works as an ip address..
 * Works if wildcards are in it too.
 */
static bool is_ip_address(const char *address)
{
    bool legal = 0;
    do {
        if (*address >= '0' && *address <= '9')
            legal = 1;
        else if (*address == '?' || *address == '*' || *address == '.')
            continue;
        else
            return 0;
    } while (*++address);
    return legal;
}

/*
 * smart_[and slow_]match
 * Hostname can be either an ip address or its resolved form.
 * Will try to convert hostname to type of pattern and then wild card match
 * them. It won't do that w/ the pattern however.
 */
bool smart_match(const char *hostname, const char *pattern)
{
    if (!strcmp(pattern, "*"))
        return 1;
    bool is_pattern_ip = is_ip_address(pattern);
    bool is_host_ip = is_ip_address(hostname);

    /* pattern and host are same type */
    if (is_pattern_ip == is_host_ip)
        return wild_match(pattern, hostname);
    else if (!is_pattern_ip && is_host_ip && !(pcfg.flags & NO_REVERSE_LOOKUPS) )
    {  
        /* host is an ip address, but pattern is not */
        char resolved[512];
        if (reverse_lookup(hostname, resolved, 512))
            return wild_match(pattern, resolved);
        return 0;
    }
    else if (is_pattern_ip && !is_host_ip && !(pcfg.flags & NO_REVERSE_LOOKUPS))
    {  
        /* host is not an ip address, but pattern is */
        struct sockaddr_in sin;
        if (resolve_address(hostname, &sin))
            return wild_match(pattern, inet_ntoa(sin.sin_addr));
    }
    return 0;
}

/* functions for ruleset class */
ruleset::ruleset(void)
{
    obsolete = 0;
    num_registered_from = num_registered_to = 0;
    DEBUG("ruleset::ruleset(): %p\n", this);

}

ruleset::~ruleset(void)
{
	destroy_list(&to_hosts, 0);
	destroy_list(&from_hosts, 0);
	DEBUG("ruleset::~ruleset(): %p\n", this);
}

/*
 *   Code for allowed_ruleset class:
 *   a config file block can be :
 *  allow {
 *     [num] from (address) [ports] : Num is optional.
 *                                     Max # of clients
 *                                     allowed to connect from that address.
 *                                     Defaults to -1, which means unlimited.
 *                              address:
 *                                    Address to permit clients from
 *                              ports: 
 *                                    What ports should it allow/deny to:
 *                                    "all"
 *                                    "6667"
 *                                    "6660-7000"
 *                                    "7000,4000,40355,29421,500-2403,6660-6666"
 *           
 *     [num] to (address) [ports]  : blah blah
 *   }
 *   There must be at least one `from' and one 'to' entry.
 *   `Reason' arguments to below functions are ignored.
 *              
 */
bool allowed_ruleset::add_host_to(const char *address_to, const char *ports,
                                   const char *, int max_num)
{
    rs_host * r = new rs_host(TO, address_to, ports, NULL, max_num);
    if (r)
        to_hosts.add(r);
    return (r != 0);
}

bool allowed_ruleset::add_host_from(const char *address_from, const char *ports,
                                     const char *, int max_num)
{
    rs_host * r = new rs_host(FROM, address_from, ports, NULL, max_num);
    if (r) 
        from_hosts.add(r);
    return (r != 0);
}

bool allowed_ruleset::does_match(const char *from, unsigned short port, char t)

{
    list_iterator<rs_host> i((t == FROM ? &from_hosts : &to_hosts));
    while (i.has_next())
    {
    	rs_host * p = i.next();
        if (port_in_set(port, p->ports) &&
            smart_match(from, p->address))
            return 1;
    }
    return 0;
}

/*
 *  Checks if this host is allowed entry by this rule list.
 *  return values:  
 *    1 - User is allowed
 *    0 - No matching hosts found
 *   -1 - User is banned - 'buff' is filled w/ the reason.. 
 *        for the case of the allowed_ruleset class, it will only be returned
 *        in case user limit for rule set was exceeded
 *                    
 *  buff = buffer to hold reason for banning, if any.
 *  
 */
int allowed_ruleset::is_allowed(const char *address, unsigned short port,
                                 char *buff, size_t bufflen)
{
    list_iterator<rs_host> i(&from_hosts);
	while (i.has_next())
	{
		rs_host * p = i.next();
        if (port_in_set(port, p->ports) && smart_match(address, p->address))
        {
            /* user is allowed, but are there anymore ppl permitted.. ? */
            if (p->num >= p->max && p->max != UNLIMITED)
            {
                safe_strcpy(buff, "No more connections allowed from your host.", bufflen);
                return -1;
            }
            else if (p->num < p->max || p->max == UNLIMITED)
                /* 
                  Enough room for this guy. 
                   Only one match is needed
                */
                return 1;
        }
    }
    return 0;
}

/* 
 * Like above but checks if it 'from' may connect to 'to' on port.
 * Even though there is a 'from' argument, we assume that the caller 
 * belongs to this rule set (for now)..
 */
int allowed_ruleset::is_allowed_to(const char *, const char *to,
                                    unsigned short port, char *buff, size_t bufflen)
{
    list_iterator<rs_host> i(&to_hosts);
    while (i.has_next())
    {
    	rs_host * p = i.next();
        if (smart_match(to, p->address) && port_in_set(port, p->ports))
        {
            /* again, check for max */
            if (p->num >= p->max && p->max != UNLIMITED)
            {
                safe_strcpy(buff, "No more users from your host are allowed to connect there.", 
                            bufflen);
                return -1;
            }
            else if (p->num < p->max || p->max == UNLIMITED)
                return 1;
        }
    }
    return 0; 
}   

/* 
 *  Return values by register_connection and friends:
 *  1 - Connection [un]registered successfully
 *  0 - ''          ''            unsuccessfully
 * -1 - full ?
 */
int allowed_ruleset::register_connection(char t, const char *from, const char *to,
                                          u_short port)
{
    /* first find the matching rs_hosts */
    list_iterator<rs_host> i((t == FROM ? &from_hosts : &to_hosts));
    while (i.has_next())
    {
    	rs_host * p = i.next();
        if (smart_match((t == FROM ? from : to),p->address) && port_in_set(port, p->ports))
        {
            /* check again if max has been exceeded */
            if (p->num >= p->max && p->max != UNLIMITED)
                return -1;
            else {
                p->num++;
                return ruleset::register_connection(t, from, to, port);
            }
        }
    }
    return 0;
}

int allowed_ruleset::unregister_connection(char t,
        const char *from, const char *to, u_short port)
{
    list_iterator<rs_host> i((t == FROM ? &from_hosts : &to_hosts));
    while (i.has_next())
    {
    	rs_host * p = i.next();  	
        if (smart_match((t == FROM ? from : to),p->address)
            && port_in_set(port, p->ports))
        {
            p->num--;
            ruleset::unregister_connection(t, from, to, port);
            return 1;
        }
    }
    return 0;
}

/*
 *  Stuff for denied_ruleset.
 *  Syntax is basically same but:
 *
 *     * A from w/o a to or vice versa is permitted
 *     * In a rule set w/ no from entries is_allowed will always
 *        return 1. is_allowed_to will check as usual. does_match() will
 *        also always return 1 in such case.
 *     * max # nubmers are ignored here
 *     * If a rule set has both from and tos, the tos are only
 *        applied to the from addresses. The addresses in the form fields
 *        will be granted passage
 */
bool denied_ruleset::add_host_to(const char *address, const char *ports,
                                  const char *reason4ban, int)          
{
    rs_host * r = new rs_host(TO, address, ports, reason4ban, 0);
    if (r)
        to_hosts.add(r);
    return (r  != 0);
}

bool denied_ruleset::add_host_from(const char *address, const char *ports,
                                    const char *reason4ban, int)
{
    rs_host * r = new rs_host(FROM, address, ports, reason4ban, 0);
    if (r)
        from_hosts.add(r);
    return (r != 0);
}

/* 
 *  NOTE: in the case that there are no entries of type 't',
 *        1 will be returned.
 */
bool denied_ruleset::does_match(const char *from, unsigned short port, char t)
{
    list_iterator<rs_host> i(&from_hosts);
    if (t == TO)
    {
        if (!to_hosts.size())
            return 1;
        i.attach(&to_hosts);
    } else {
        if (!from_hosts.size())
            return 1;
    }

	while (i.has_next())
    {
		rs_host * p = i.next();
		if (port_in_set(port, p->ports) && smart_match(from, p->address))
           return 1;
    }
    return 0;
}

/*
 *  Returns:
 *    -1 - if banned
 *     0 - not found in list (not banned)
 */
int denied_ruleset::is_allowed(const char *address, u_short port,
                                char *buff, size_t bufflen)
{
    list_iterator<rs_host> i(&from_hosts);
	while (i.has_next())
	{
	    rs_host *p = i.next();
        if (smart_match(address, p->address) && port_in_set(port, p->ports))
        {
            safe_strcpy(buff, p->reason, bufflen);
            return -1;
        }
    }

    return 0;
}

/*
 *  return:
 *     -1: not allowed to connect
 *      0: not in list, does not necesssarily mean he may connect there tho
 */
int denied_ruleset::is_allowed_to(const char *, const char *address_to, u_short port,
                                   char *buff, size_t bufflen)
{
    list_iterator<rs_host> i(&to_hosts);
	while (i.has_next())
	{
		rs_host * p = i.next();
        if (smart_match(address_to, p->address) && port_in_set(port, p->ports))
        {
	        safe_strcpy(buff, p->reason, bufflen);
            return (-1);
        }
    }
    return 0;
}

inline int denied_ruleset::register_connection(char t, const char*,const char *, u_short)
{
    return ruleset::register_connection(t, NULL, NULL, 0), 1;
}

inline int denied_ruleset::unregister_connection(char t, const char*,const char *, u_short)
{
    return ruleset::unregister_connection(t, NULL, NULL, 0), 1;
}

/*
 * Compare two rulesets
 * Ignores num_registered_xxx and obsolete and rs_host.num
 * Note! - Since this operates on the ruleset base class, there
 *     is no way of knowing for sure if the *type* of the object we are being compared
 *     to is the same type as us. (Actually each rs_host of an Allowed_ruleset has
 *     reason set to NULL, and each denined_ruleset should have it set to something
 *     other than that, because the parsing code will give 'No reason was given!'..)
 *     (ok, maybe you can make this virtual and do it that way somehow)
 *
 */
bool ruleset::operator == (ruleset& r2)
{
    list_iterator<rs_host> rs_it(&from_hosts);
    list_iterator<rs_host> rs2_it(&r2.from_hosts);
    rs_host * rs2, * rs;
    if (!(from_hosts.size() == r2.from_hosts.size()) ||
        !(to_hosts.size()   == r2.to_hosts.size()))
        return 0;
    /* Compare from hosts first, then check the 'to' hosts */
    for (int i = 0; i < 2; i++)
    {
        while (rs_it.has_next() && rs2_it.has_next())
        {
            rs = rs_it.next();
            rs2 = rs2_it.next();

            /* fixme - we check for reason first because it can be null for
             * allowed_ruleset types. However we assume the other pointers are all vaild */
            if ((rs->reason && !rs2->reason) || (!rs->reason && rs2->reason))
                return 0;
            if ((rs->max  != rs2->max)  ||
                /* (rs->num  != rs2->num)  || */
                strcasecmp(rs->address, rs2->address) ||
                strcasecmp(rs->ports, rs2->ports) ||
                (rs->reason && rs2->reason && strcasecmp(rs->reason, rs2->reason)))
                return 0;
        }
        rs_it.attach(&to_hosts);
        rs2_it.attach(&r2.to_hosts);
    }
    return 1;
}

ruleset::rs_host::~rs_host()
{
    delete[] reason;
    delete[] address;
    delete[] ports;
    DEBUG("\trs_host::~rs_host(): [%p] destructed\n", this);
}

ruleset::rs_host::rs_host(char _type, const char *_addr,
                           const char *_ports , const char *_reason, int _max)
{
    max = _max; 
    num = 0;
    type = _type;
    reason = my_strdup(_reason);
    address = my_strdup(_addr);
    ports   = my_strdup(_ports);
    DEBUG("\trs_host::rs_host(): [%p] created -- type %c, addr: %s, ports: %s, max: %d reason: %s\n",
    			this, type, address, ports, max, reason);
}

/* Searches src for rulesets matching host and port,
 * and puts them in target
 */
int ruleset::find_matching(const char *host, unsigned short port,
                                  list<ruleset> *src, list<ruleset> * target)
{
    int numfound = 0;
    ruleset * r = 0;
    list_iterator<ruleset> i(src);

	while (i.has_next())
	{
		r = i.next();
        if (!r->obsolete && r->does_match(host, port, ruleset::FROM))
        {
            if (target)
                target->add(r);
            numfound++;
        }
    }
    return numfound;
}

/*
 * Search a list
 * Return:
 *   -1:   Banned
 *    0:   Not found on list (so could be either)
 *    1:   Found on list (definately allowed)
 */
int ruleset::list_is_allowed_from(list<ruleset> * ls, const char * host, unsigned short port ,
                    char * buff , long size)
{
    list_iterator<ruleset> i(ls);
    bool permit = 0;

   	while (i.has_next())
   	{
   		ruleset * r = i.next();

        switch(r->is_allowed(host, port, buff, size))
        {
        case -1:
            return -1;
        case 1:
            permit = 1;
        }
    }
    return permit;
}

int ruleset::list_is_allowed_to(list<ruleset> * ls, const char * from, const char * to, unsigned short port ,
                    char * buff , long size)
{
    list_iterator<ruleset> i(ls);
    bool permit = 0;

   	while (i.has_next())
   	{
   		ruleset * r = i.next();
   		
        switch(r->is_allowed_to(from, to, port, buff, size))
        {
        case -1:
            return -1;
        case 1:
            permit = 1;
        }
    }
    return permit;
}

int ruleset::list_register_from(list<ruleset> * ls, const char * host, unsigned short port)
{
    list_iterator<ruleset> i(ls);
   	while (i.has_next())
   		i.next()->register_connection(ruleset::FROM, host, 0, port);
    return 1;
}

int ruleset::list_register_to(list<ruleset> * ls, const char * host,  unsigned short port)
{
    list_iterator<ruleset> i(ls);
   	while (i.has_next())
   		i.next()->register_connection(ruleset::TO, 0, host, port);
    return 1;
}

int ruleset::list_unregister_from(list<ruleset> * ls, const char * host,
                unsigned short port)
{
    list_iterator<ruleset> i(ls);
   	while (i.has_next())
   		i.next()->unregister_connection(ruleset::FROM, host, 0, port);
    return 1;
}

int ruleset::list_unregister_to(list<ruleset> * ls, const char * host,
                unsigned short port)
{
    list_iterator<ruleset> i(ls);
   	while (i.has_next())
   		i.next()->unregister_connection(ruleset::TO, 0, host, port);
    return 1;
}


/*
 * FIXME :-P
 */
list<ruleset> * ruleset::sync_lists(list<ruleset> * oldlist, list<ruleset> * newlist)
{
    list_iterator<ruleset> i(oldlist);
    list<ruleset> * list3 = new list<ruleset>;
    ruleset * r, * r2;

    while (i.has_next())
    {
		r = i.next();
        list_iterator<ruleset> i2(newlist);
        r2 = 0;

        while (i2.has_next())
        {
        	r2 = i2.next();
            /* They're the same. Delete new one, transfer old one
             * to new list */
           if (*r2 == *r)
           {
                DEBUG("         RULESET: Found match for %p == %p\n", r, r2);
                delete r2;
                i2.remove();
                list3->add(r);
                goto redo;
            }
        }

        /* Oops, didn't match anybody. If no users, destroy.
         * If there are, mark as obsolete */
        if (!r->num_registered_from && !r->num_registered_to)
        {
            DEBUG("         RULESET: %p didn't match: deleting\n", r);
            delete r;
            continue;
        }
        /* We have users.. mark as obsolete, add to list */
        DEBUG("         RULESET: %p didn't match: marking as obsolete and re-adding\n", r);
        r->obsolete = true;
        list3->add(r);
        redo:;
    }

    /* Add any new ones */
    i.attach(newlist);
    while (i.has_next())
	{
		r = i.next();		
        DEBUG("Adding new ruleset: %p\n", r);
        list3->add(r);
    }

    delete oldlist;
    delete newlist;
    return list3;
}

