
#ifndef __USER_H
#define __USER_H

class user_options;
class userdef;

#include "autoconf.h"
#include "linkedlist.h"
#include "ruleset.h"
#include "conn.h"


enum  {
    /* non-flag options */
    OPT_PASSWORD                     = 0x220000,
    OPT_AUTOSERVER,
    OPT_MAX_LOGFILE_SIZE,
    OPT_DEFAULT_LOG_OPTIONS,
    OPT_MAX_IDLE_TIME,

    /* flags */
    OPT_AUTO_FAKE_IDENT              = 0x100,
    OPT_ENABLE_OUTGOING_DCC_PROXYING = 0x200,
    OPT_ENABLE_INCOMING_DCC_PROXYING = 0x400,
    OPT_ENABLE_PUBLIC_LOGGING        = 0x800,
    OPT_ENABLE_PRIVATE_LOGGING       = 0x1000,
    OPT_ENABLE_SEPERATE_LOGGING      = 0x2000,
    OPT_DROP_ON_DISCONNECT           = 0x4000,
    OPT_ENABLE_DETACH_COMMAND        = 0x8000,
    OPT_ENABLE_AUTO_DETACH           = 0x10000,
    OPT_ENABLE_VHOST_COMMAND         = 0x20000,
    OPT_ENABLE_FAKE_IDENTS           = 0x40000,
    OPT_USER_IS_ADMIN                = 0x80000,
    OPT_DEFAULT_VHOST                = 0x100000,
    OPT_LOG_OPTIONS = OPT_ENABLE_PUBLIC_LOGGING | OPT_ENABLE_PRIVATE_LOGGING,

    /***************************
     * Flags for 'set' command *
     ***************************/

    PREF_DCC_IN                 = OPT_ENABLE_INCOMING_DCC_PROXYING,
    PREF_DCC_OUT                = OPT_ENABLE_OUTGOING_DCC_PROXYING,
    PREF_AUTO_DETACH            = OPT_ENABLE_AUTO_DETACH,
    PREF_LOG                    = OPT_LOG_OPTIONS,
    PREF_FAKE_IDENT             = OPT_ENABLE_FAKE_IDENTS,
    PREF_VHOST                  = OPT_ENABLE_VHOST_COMMAND,
    PREF_AUTO_PASS              = 0x10000000
};




extern list<userdef> * users;

/*
 * user_options
 * facilities for user preferences
 * set(int, int)            = for setting the various options
 * set(int, const char *)   =  [------------ " -------------]
 *
 */


class user_options
{
public:
    user_options(user_options *);
    user_options();
    ~user_options();

public:
    int             set(int, int);           /* numeric opts */
    int             set(int, const char *);  /* string opts */
    int             clear(int);              /* clear */
    int             get(int) const;
    char *          get(int, char *) const;
    int             decide(int) const;


    /********************************************
     * inline functions to access flags & prefs *
     ********************************************/
    int getf(void) const
    {
        return flags;
    }

    int getp(void) const
    {
        return prefs;
    }

    int setf(int v)
    {
        return (flags |= v);
    }
    int setp(int v)
    {
        return (prefs |= v);
    }

    int checkf(int v)  const
    {
        return (flags & v);
    }
    int checkp(int v) const
    {
        return (prefs & v);
    }

    int clearf(int v)
    {
        return (flags &= (~v));
    }
    int clearp(int v)
    {
        return (prefs &= (~v));
    }

    int load(const char *);
    int save(const char *, int);
    int copy(user_options *);
    int copy_config_only(const user_options *);

    static int      lookup_cmd(const char *);       /* command parsing utility */

private:
    int             flags;                           /* boolean variables */
    int             prefs;                           /* user preferences (override above options) */

    char *          password;                        /* user password */
    char *          autoserver;                      /* Where to automagically connect this user */

    char *          autopass;                        /* Auto-Detach pass */
    char *          fake_ident;                      /* The fake ident */

    unsigned        max_logsize;                     /* how big can the log files get? */
    time_t          max_idle;                        /* idle time limit */
    int             log_options;                     /* What log options does the user want? */

    struct in_addr  iface;                           /* def. interface to connect on */
};




class userdef
{
public:
    userdef(const char *);
    ~userdef(void);
    int add(conn *, list<ruleset> *,
            const char * password, const char * hostname,
            unsigned short port);
    int remove(conn *);
    bool is_obsolete(void) { return obsolete; }

    static userdef * find(list<userdef> *, const char *);
    static list<userdef> * sync_lists(list<userdef> *, list<userdef> *);

private:
    bool  obsolete;

public:
    char * name;
    list<ruleset> * rulesets;
    list<conn>    * conns;
    list<char>    * vhosts;
    user_options    cfg;

    friend class conn;
};


#endif
