// FGA example: travelling salesman problem
// by Alessandro Presta
// This source code is protected by the GNU Lesser General Public License
// (see license.txt)

#include "fga.hpp"
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <algorithm>
#include <fstream>
#include <iostream>

using namespace fga;
using namespace std;

#define T 2 // number of tribes (threads) in population
#define MAX_COST 10 // maximum cost for a road
#define PRINT_ALL() {	system("clear"); \
						cout << "Generations:             " << my_population.get_generations() << endl << endl; \
						cout << "Mutation rate:           " << mutation_rate << endl << endl; \
						cout << "Best cost:               " << MAX_COST * L - score << endl << endl; \
						cout << "Best path:" << endl; \
						for (int i = 0; i < L; ++i) \
							cout << best[i] << " "; \
						cout << endl << endl; \
		    		}

int N; // number of chromosomes in population
int L; // number of genes in a chromosome
int G; // maximum number of stable generations before stop
int **cost; // cost for the road between nodes [i][j]

// fitness evaluation function
float my_fitness(int *chromosome)
{
	int total_cost = 0;
	for (int i = 0; i < L - 1; ++i)
		total_cost += cost[chromosome[i]][chromosome[i + 1]];
	return (float)(MAX_COST * L - total_cost);
}

// random chromosome creation function
void my_random_chromosome(int *chromosome)
{
	int path[L];
	for (int i = 0; i < L; ++i)
		path[i] = i;
	random_shuffle(path, path + L);
	memcpy(chromosome, path, L * sizeof(int));
}

// chromosome mutation function
void my_mutate_chromosome(int *chromosome)
{
	int a = rand() % L;
	int b = rand() % L;
	int tmp = chromosome[a];
	chromosome[a] = chromosome[b];
	chromosome[b] = tmp;
}

// custom crossover function
void my_crossover(int *parent1, int *parent2)
{
	int child1[L], child2[L];
	bool mark[L];
	memset(mark, 0, L * sizeof(bool));
	// select a random sub-string from parent chromosome 1
	int a = rand() % L;
	int b = rand() % (L - a) + a;
	for (int i = 0; i <= b - a; ++i) {
		child1[i] = parent1[a + i];
		mark[child1[i]] = 1;	
	}
	// add the maximum compatible sub-string from parent chromosome 2
	int p = 1;	
	for (int i = 0; i < L - 1; ++i)
		if (parent2[i] == parent1[b]) {
			p = 1;
			while (i + p < L && !mark[parent2[i + p]]) {			
				mark[parent2[i + p]] = 1;			
				child1[b - a + p] = parent2[i + p];
				++p;			
			}	
			break;
		}
	// if needed, complete the path with missing nodes
	for (int i = 0; i < L; ++i)
		if (!mark[i])
			child1[b - a + p++] = i;
	// repeat each step with parents swapped
	memset(mark, 0, L * sizeof(bool));
	a = rand() % L;
	b = rand() % (L - a) + a;
	for (int i = 0; i <= b - a; ++i) {
		child2[i] = parent2[a + i];
		mark[child2[i]] = 1;	
	}	
	p = 1;
	for (int i = 0; i < L - 1; ++i)
		if (parent1[i] == parent2[b]) {
			p = 1;
			while (i + p < L && !mark[parent1[i + p]]) {				
				mark[parent1[i + p]] = 1;			
				child2[b - a + p] = parent1[i + p];	
				++p;			
			}	
			break;
		}
	for (int i = 0; i < L; ++i)
		if (!mark[i])
			child2[b - a + p++] = i;
	// apply changes to the population
	memcpy(parent1, child1, L * sizeof(int));
	memcpy(parent2, child2, L * sizeof(int));
}

int main()
{
	srand(time(NULL));
	ifstream in_file("graph.txt");
	in_file >> L;
	cost = new int*[L];
	for (int i = 0; i < L; ++i) {
		cost[i] = new int[L];
		for (int j = 0; j < L; ++j)
			cost[i][j] = 0;
	}
	for (int i = 0; i < L - 1; ++i)
		for (int j = i + 1; j < L; ++j) {
			in_file >> cost[i][j];
			cost[j][i] = cost[i][j];
		}
	in_file.close();
	N = L * 4;
	PopulationMT<int> my_population(T, N / T, L, my_fitness, NULL, NULL, my_crossover, my_mutate_chromosome, my_random_chromosome);
	float mutation_rate = 0.1;	
	my_population.set_mutation_rate(mutation_rate);		
	int best[L];
	float score = -1;	
	int count1 = 0;
	int count2 = 0;
	G = 300 * L;	
	while (1) {
		my_population.cycle();
		if (count1 % 10 == 0)
			PRINT_ALL();
		if (my_population.get_all_time_best_score() > score) {
			score = my_population.get_all_time_best_score();
			memcpy(best, my_population.get_all_time_best(), L * sizeof(int));
			count1 = 0;
			count2 = 0;
			PRINT_ALL();
		}
		++count1;
		++count2;
		if (mutation_rate < 0.8 && count2 * 6 >= G) {
			mutation_rate += 0.1;
			my_population.set_mutation_rate(mutation_rate);
			count2 = 0;
			PRINT_ALL();
		}		
		if (count1 >= G) {
			PRINT_ALL();
			cout << "Best path was stable for " << G << " generations. Algorithm halted." << endl;
			goto HALT;
		}
	}
HALT:
	for (int i = 0; i < L; ++i)
		delete[] cost[i];
	delete[] cost;	
	return 0;	
}
