/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2004 Mark D. Roth
**  All rights reserved.
**
**  connect.c - FTP connection code
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>
#include <pwd.h>
#include <time.h>

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif


static fget_hash_t *ftpcache_h = NULL;
static int num_sessions = 0;


struct ftpcache
{
	char host[MAXHOSTNAMELEN];
	char user[MAXUSERNAMELEN];
	FTP *ftp;
	time_t timestamp;
};
typedef struct ftpcache FTPCACHE;


static int
ftpcache_match(struct ftp_url *key, FTPCACHE *data)
{
#ifdef DEBUG
	printf("=== ftpcache_match(key={\"%s\",\"%s\"}, "
	       "data={\"%s\",\"%s\"})\n",
	       key->fu_hostname, key->fu_login, data->host, data->user);
#endif
	return (strcmp(key->fu_login, data->user) == 0
		&& strcmp(key->fu_hostname, data->host) == 0);
}


static int
ftp_match(FTP *key, FTPCACHE *data)
{
#ifdef DEBUG
	printf("=== ftpcache_match(key={\"%s\",\"%s\"}, "
	       "data={\"%s\",\"%s\"})\n",
	       ftp_get_host(key), ftp_whoami(key), data->host, data->user);
#endif
	return (strcmp(ftp_whoami(key), data->user) == 0
		&& strcmp(ftp_get_host(key), data->host) == 0);
}


static char *
domain_of(char *host)
{
	char *cp;
	int numdots = 0;

	cp = host + strlen(host) - 1;
	if (*cp == '.')
		cp--;

	for (; numdots < 2 && cp > host; cp--)
	{
		if (*cp == '.')
			numdots++;
	}

	if (cp != host)
		cp++;

	return cp;
}


static int
ftpcache_hash(FTPCACHE *key, int numbuckets)
{
#ifdef DEBUG
	printf("=== ftpcache_hash(key=0x%lx, numbuckets=%d)\n",
	       key, numbuckets);
#endif
	return (domain_of(key->host)[0] % numbuckets);
}


static int
fget_disconnect(fget_hashptr_t *hp, int fast)
{
	FTPCACHE *fcp;

#ifdef DEBUG
	printf("==> fget_disconnect(hp=0x%lx, fast=%d)\n", hp, fast);
#endif
	
	fcp = (FTPCACHE *)fget_hashptr_data(hp);

	if (verbose)
		printf("\nfget: disconnecting\n");

	if (ftp_quit(fcp->ftp, fast) == -1)
		fprintf(stderr, "fget: ftp_quit(): %s\n", strerror(errno));

	fget_hash_del(ftpcache_h, hp);
	free(fcp);
	num_sessions--;

	return 0;
}


int
fget_disconnect_handle(FTP *ftp)
{
	FTPCACHE *fcp;
	fget_hashptr_t hp;

#ifdef DEBUG
	printf("==> fget_disconnect_handle(ftp=(%s))\n", ftp_get_host(ftp));
#endif

	fget_hashptr_reset(&hp);
	if (fget_hash_search(ftpcache_h, &hp, ftp,
	    		     (fget_matchfunc_t)ftp_match) != 0)
		fget_disconnect(&hp, FTP_QUIT_FAST);

	return 0;
}


int
fget_disconnect_all(void)
{
	fget_hashptr_t hp;

	fget_hashptr_reset(&hp);
	while (fget_hash_next(ftpcache_h, &hp) == 1)
		fget_disconnect(&hp, 0);

	return 0;
}


static int
fget_conn_expire(void)
{
	fget_hashptr_t hp, oldest_hp;
	FTPCACHE *fcp1, *fcp2 = NULL;

	if (num_sessions <= max_sessions)
		return 0;

	fget_hashptr_reset(&hp);
	fget_hashptr_reset(&oldest_hp);

	while (fget_hash_next(ftpcache_h, &hp) == 1)
	{
		fcp1 = (FTPCACHE *)fget_hashptr_data(&hp);
		if (oldest_hp.node == NULL ||
		    fcp1->timestamp < fcp2->timestamp)
		{
			memcpy(&oldest_hp, &hp, sizeof(hp));
			fcp2 = (FTPCACHE *)fget_hashptr_data(&oldest_hp);
		}
	}

	if (oldest_hp.node == NULL)
		return 0;

	return fget_disconnect(&oldest_hp, 0);
}


FTP *
fget_connect(struct ftp_url *ftpurl)
{
	char buf[FTPBUFSIZE];
	FTPCACHE *fcp;
	fget_hashptr_t hp;
	struct passwd *pwd;
	int i, num_tries = 0;
	char *cp;

#ifdef DEBUG
	printf("host=\"%s\"\n", ftpurl->fu_hostname);
	printf("user=\"%s\"\n", ftpurl->fu_login);
	printf("pass=\"%s\"\n", ftpurl->fu_passwd);
	printf("path=\"%s\"\n", ftpurl->fu_path);
#endif

	fget_conn_expire();

	if (ftpurl->fu_login[0] == '\0')
	{
		strlcpy(ftpurl->fu_login, "ftp", sizeof(ftpurl->fu_login));
		if (ftpurl->fu_passwd[0] == '\0')
		{
			pwd = getpwuid(getuid());
			i = snprintf(ftpurl->fu_passwd,
				     sizeof(ftpurl->fu_passwd), "%s@",
				     pwd ? pwd->pw_name : "unknown");
			gethostname(ftpurl->fu_passwd + i,
				    sizeof(ftpurl->fu_passwd) - i);
			ftpurl->fu_passwd[sizeof(ftpurl->fu_passwd) - 1] = '\0';
		}
	}

	fget_hashptr_reset(&hp);
	if (ftpcache_h != NULL
	    && fget_hash_getkey(ftpcache_h, &hp, ftpurl,
	    			(fget_matchfunc_t)ftpcache_match) != 0)
	{
		fcp = (FTPCACHE *)fget_hashptr_data(&hp);
		if (verbose)
			printf("\nfget: resuming cached connection to %s...\n",
			       fcp->host);
		fcp->timestamp = time(NULL);
		return fcp->ftp;
	}

	if (ftpcache_h == NULL)
	{
#ifdef DEBUG
		puts("    fget_connect(): creating ftpcache_h");
#endif
		ftpcache_h = fget_hash_new(127, (fget_hashfunc_t)ftpcache_hash);
	}
	if (ftpcache_h == NULL)
	{
		fprintf(stderr, "fget: fget_hash_new(): %s\n", strerror(errno));
		return NULL;
	}

	fcp = (FTPCACHE *)calloc(1, sizeof(FTPCACHE));
	if (fcp == NULL)
	{
		fprintf(stderr, "fget: calloc(): %s\n", strerror(errno));
		return NULL;
	}

	for (num_tries = 0; tries == 0 || num_tries < tries; num_tries++)
	{
		if (verbose)
		{
			printf("fget: connecting to FTP server %s",
			       ftpurl->fu_hostname);
			if (num_tries > 0)
				printf(" (attempt #%d)", num_tries + 1);
			puts("...");
		}

		if (ftp_connect(&(fcp->ftp), ftpurl->fu_hostname,
				buf, sizeof(buf), FTP_CONNECT_DNS_RR,
				FTP_OPT_PASSIVE,
				(BIT_ISSET(options, OPT_PASSIVE) ? 1 : 0),
				FTP_OPT_IO_TIMEOUT,
				timeout,
				FTP_OPT_CACHE_MAXSIZE,
				max_cache_size,
				FTP_OPT_CACHE_EXPIRE,
				cache_expire_time,
				FTP_OPT_SEND_HOOK,
				(debug ? send_debug : NULL),
				FTP_OPT_RECV_HOOK,
				(debug ? recv_debug : NULL),
				0) == -1)
		{
			if ((errno == ETIMEDOUT || errno == ECONNRESET)
			    && (tries == 0 || num_tries + 1 < tries))
			{
				printf("fget: connection timed out "
				       "(%lu seconds)\n",
				       (unsigned long)timeout);
				continue;
			}
			fprintf(stderr, "fget: ftp_connect(): %s\n",
				strerror(errno));
			return NULL;
		}

		break;
	}

	if (verbose && !debug)
		printf("%s", buf);

	if (verbose)
		printf("\nfget: logging in as user %s\n", ftpurl->fu_login);
	if (ftpurl->fu_passwd[0] == '\0' && isatty(fileno(stdin)))
		cp = getpass("Password: ");
	else
		cp = ftpurl->fu_passwd;
	if (ftp_login(fcp->ftp, ftpurl->fu_login, cp) == -1)
	{
		fprintf(stderr, "fget: ftp_login(): %s\n", strerror(errno));
		ftp_quit(fcp->ftp, FTP_QUIT_FAST);
		free(fcp);
		return NULL;
	}

	strlcpy(fcp->host, ftpurl->fu_hostname, sizeof(fcp->host));
	strlcpy(fcp->user, ftpurl->fu_login, sizeof(fcp->user));
	time(&fcp->timestamp);

	fget_hash_add(ftpcache_h, fcp);
	num_sessions++;

	return fcp->ftp;
}


