/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2004 Mark D. Roth
**  All rights reserved.
**
**  delete.c - local file deletion code for fget
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <dirent.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <string.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif


int
recursive_remove(char *dirpath)
{
	char filepath[MAXPATHLEN];
	struct stat s;
	DIR *dirp;
	struct dirent *dent;
	int retval = 0;

	dirp = opendir(dirpath);
	if (dirp == NULL)
	{
		fprintf(stderr, "fget: opendir(\"%s\"): %s\n", dirpath,
			strerror(errno));
		return -1;
	}

	while ((dent = readdir(dirp)) != NULL)
	{
		if (strcmp(dent->d_name, ".") == 0 ||
		    strcmp(dent->d_name, "..") == 0)
			continue;

		snprintf(filepath, sizeof(filepath), "%s/%s", dirpath,
			 dent->d_name);

		if (lstat(filepath, &s) != 0)
		{
			fprintf(stderr, "fget: lstat(\"%s\"): %s\n", filepath,
				strerror(errno));
			retval = -1;
			break;
		}

		if (S_ISDIR(s.st_mode))
		{
			if (recursive_remove(filepath) != 0)
			{
				retval = -1;
				break;
			}
		}
		else if (remove(filepath) != 0)
		{
			fprintf(stderr, "fget: remove(\"%s\"): %s\n", filepath,
				strerror(errno));
			retval = -1;
			break;
		}
	}

	closedir(dirp);

	if (retval == 0)
	{
		if (rmdir(dirpath) != 0)
		{
			fprintf(stderr, "fget: remove(\"%s\"): %s\n", dirpath,
				strerror(errno));
			retval = -1;
		}
	}

	return retval;
}


/* delete local files which don't exist on the server */
int
fget_delete_local(FTP *ftp, char *remotepath, char *localpath)
{
	struct ftpstat fs;
	DIR *dirp;
	struct dirent *dent;
	char remotefile[MAXPATHLEN];
	char localfile[MAXPATHLEN];
	struct stat s;
	int retval = 0;

#ifdef DEBUG
	printf("==> fget_delete_local(\"%s\", \"%s\")\n", remotepath,
	       localpath);
#endif

	if (localpath == NULL || localpath[0] == '\0')
		localpath = ".";

	dirp = opendir(localpath);
	if (dirp == NULL)
	{
		fprintf(stderr, "fget: opendir(\"%s\"): %s\n", localpath,
			strerror(errno));
		return -1;
	}

	while ((dent = readdir(dirp)) != NULL)
	{
		if (strcmp(dent->d_name, ".") == 0 ||
		    strcmp(dent->d_name, "..") == 0)
			continue;

		snprintf(localfile, sizeof(localfile), "%s/%s", localpath,
			 dent->d_name);
		if (lstat(localfile, &s) != 0)
		{
			fprintf(stderr, "fget: lstat(\"%s\"): %s\n", localfile,
				strerror(errno));
			retval = -1;
			break;
		}

		/* don't nuke the local lost+found directory */
		if (strcmp(dent->d_name, "lost+found") == 0
		    && S_ISDIR(s.st_mode))
			continue;

		snprintf(remotefile, sizeof(remotefile), "%s/%s", remotepath,
			 dent->d_name);
		if (ftp_lstat(ftp, remotefile, &fs) == 0)
			continue;
		if (errno != ENOENT)
		{
			fprintf(stderr, "fget: ftp_stat(\"%s\"): %s\n",
				remotefile, strerror(errno));
			retval = -1;
			break;
		}

		if (verbose)
			printf("fget: removing local %s %s...\n",
			       (S_ISDIR(s.st_mode) ? "tree" : "file"),
			       localfile);

		if (S_ISDIR(s.st_mode))
		{
			if (recursive_remove(localfile) != 0)
			{
				retval = -1;
				break;
			}
		}
		else if (remove(localfile) != 0)
		{
			fprintf(stderr, "fget: remove(\"%s\"): %s\n", localfile,
				strerror(errno));
			retval = -1;
			break;
		}
	}

	closedir(dirp);
	return retval;
}


int
fget_unlink(char *path, struct stat *sp)
{
	if (S_ISDIR(sp->st_mode))
		return recursive_remove(path);

	if (unlink(path) == -1)
	{
		fprintf(stderr, "fget: unlink(\"%s\"): %s\n", path,
			strerror(errno));
		return -1;
	}

	return 0;
}


