/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  fget.c - main driver program for fget
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>
#include <sys/param.h>
#include <fcntl.h>
#include <pwd.h>

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#if defined(DEBUG) && defined(HAVE_GETRUSAGE)
# include <sys/time.h>
# include <sys/resource.h>
#endif


#define MODE_DOWNLOAD		1
#define MODE_LIST		2
#define MODE_ESTIMATE		3


unsigned short verbose = 1;
unsigned long options = DEFAULT_OPTIONS;
unsigned short debug = 0;
unsigned short max_sessions = 5;
long max_cache_size = 1024;
time_t cache_expire_time = (time_t)-1;
time_t timeout = -1;
unsigned short tries = 1;
char *extendfs_command = NULL;


static void
usage(void)
{
	fprintf(stderr, "Options: [-cdDFHNpPqrRsvV] [-C dir] "
			"[-E extendfs_command] [-I urlfile]\n");
	fprintf(stderr, "         [-m max_connections] [-t timeout] "
			"[-T num_retries]\n");
	fprintf(stderr, "         [-i cache_interval] [-M max_cache_size]\n");
	fprintf(stderr, "Usage:\n");
	fprintf(stderr, "   fget [options] URL ...\n");
	fprintf(stderr, "   fget [options] -l URL ...\n");
	fprintf(stderr, "   fget [options] -e URL ...\n");
	exit(1);
}


int
fget_main(char *url, char *changeroot, int mode)
{
	FTP *ftp = NULL;
	struct ftp_url ftpurl;
	ftp_glob_t pglob;
	int i, retval = 0;
	char buf[MAXPATHLEN];

	ftp_url_parse(url, &ftpurl);
	ftp = fget_connect(&ftpurl);
	if (ftp == NULL)
		return 1;

	if (ftpurl.fu_path[0] != '/')
	{
		snprintf(buf, sizeof(buf), "%s/%s", ftp_getcwd(ftp),
			 ftpurl.fu_path);
		fget_cleanpath(buf, ftpurl.fu_path, sizeof(ftpurl.fu_path));
	}

	i = ftp_glob(ftp, ftpurl.fu_path, FTPGLOB_BRACE, NULL,
		     &pglob);
	if (i < 0)
	{
		if (i == FTPGLOB_NOMATCH)
			errno = ENOENT;
		else if (i == FTPGLOB_NOSPACE)
			errno = ENOMEM;
		fprintf(stderr, "fget: ftp_glob(\"%s\"): %s\n",
			ftpurl.fu_path, strerror(errno));
		if (errno == ECONNRESET)
			fget_disconnect_handle(ftp);
		ftp_globfree(&pglob);
		return 1;
	}

	for (i = 0; i < pglob.gl_pathc; i++)
	{
#ifdef DEBUG
		printf("DEBUG: main: pglob.gl_pathv[%d] = \"%s\"\n",
		       i, pglob.gl_pathv[i]);
#endif
		if (init_visit_lists() == -1)
			return 1;

		switch (mode)
		{
		case MODE_DOWNLOAD:
			if (fget_download(ftp, pglob.gl_pathv[i], changeroot)
			    != R_FILEOK)
				retval++;
			break;
		case MODE_LIST:
			if (fget_list(ftp, pglob.gl_pathv[i]) != R_FILEOK)
				retval++;
			break;
		case MODE_ESTIMATE:
			if (fget_du(ftp, pglob.gl_pathv[i]) != R_FILEOK)
				retval++;
		}

		free_visit_lists();
	}

	ftp_globfree(&pglob);
	return retval;
}


int
main(int argc, char *argv[])
{
	int i;
	int mode = 0, retval = 0;
	char *urlfile = NULL;
	char *changeroot = NULL;
	FILE *f = NULL;
	char buf[MAXPATHLEN + MAXHOSTNAMELEN + 6];
#if defined(DEBUG) && defined(HAVE_GETRUSAGE)
	struct rusage rusage;
#endif

	while ((i = getopt(argc, argv,
			   "cC:dDeE:FhHi:I:lm:M:NpPqrRsSt:T:vV")) != -1)
	{
		switch (i)
		{
		case 'V':
			printf("fget %s by Mark D. Roth <roth@feep.net>\n",
			       PACKAGE_VERSION);
			break;
		case 'd':
			debug++;
			break;
		case 'v':
			verbose++;
			break;
		case 'q':
			verbose = 0;
			break;
		case 'l':
			if (mode != 0)
				usage();
			mode = MODE_LIST;
			BIT_CLEAR(options, OPT_RECURSIVE);
			BIT_CLEAR(options, OPT_CHASELINKS);
			break;
		case 'e':
			if (mode != 0)
				usage();
			mode = MODE_ESTIMATE;
			break;
		case 'C':
			if (changeroot != NULL)
				free(changeroot);
			changeroot = strdup(optarg);
			break;
		case 't':
			timeout = atoi(optarg);
			break;
		case 'T':
			tries = atoi(optarg);
			break;
		case 'm':
			max_sessions = atoi(optarg);
			break;
		case 'M':
			max_cache_size = atoi(optarg);
			break;
		case 'i':
			cache_expire_time = atoi(optarg);
			break;
		case 'I':
			if (urlfile != NULL)
				free(urlfile);
			urlfile = strdup(optarg);
			break;
		case 'E':
			if (extendfs_command != NULL)
				free(extendfs_command);
			extendfs_command = strdup(optarg);
			break;
		case 'c':
			BIT_TOGGLE(options, OPT_CHASELINKS);
			break;
		case 'D':
			BIT_TOGGLE(options, OPT_DEREFLINKS);
			break;
		case 'F':
			BIT_TOGGLE(options, OPT_FIXLINKS);
			break;
		case 'N':
			BIT_TOGGLE(options, OPT_NEWER);
			break;
		case 'R':
			BIT_TOGGLE(options, OPT_RECURSIVE);
			break;
		case 'p':
			BIT_TOGGLE(options, OPT_PASSIVE);
			break;
		case 'P':
			BIT_TOGGLE(options, OPT_SAVEPERMS);
			break;
		case 'r':
			BIT_TOGGLE(options, OPT_DELETELOCAL);
			break;
		case 's':
			BIT_TOGGLE(options, OPT_DIFFSIZE);
			break;
		case 'S':
			BIT_TOGGLE(options, OPT_STRIPDIR);
			break;
		case 'H':
			BIT_TOGGLE(options, OPT_HOSTDIRS);
			break;
		case 'h':
		default:
			usage();
		}
	}

	if (mode == 0)
		mode = MODE_DOWNLOAD;
	if (argc - optind == 0 && urlfile == NULL)
		usage();

	for (i = optind; i < argc; i++)
		retval += fget_main(argv[i], changeroot, mode);

	if (urlfile)
	{
		f = fopen(urlfile, "r");
		if (f == NULL)
		{
			fprintf(stderr, "fget: cannot open URL file %s\n",
				urlfile);
			retval++;
		}
		else
		{
			while (fgets(buf, sizeof(buf), f) != NULL)
			{
				i = strlen(buf) - 1;
				if (buf[i] == '\n')
					buf[i] = '\0';
				retval += fget_main(buf, changeroot, mode);
			}
			fclose(f);
		}
	}

	fget_disconnect_all();

#if defined(DEBUG) && defined(HAVE_GETRUSAGE)
	if (getrusage(RUSAGE_SELF, &rusage) == -1)
		perror("fget: getrusage()");
	else
		printf("DEBUG: rusage.ru_maxrss = %l pages (%l bytes)\n",
		       rusage.ru_maxrss,
		       rusage.ru_maxrss * sysconf(_SC_PAGESIZE));
#endif

	exit(retval);
}


