/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  list.c - FTP file listing code for fget
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <string.h>
#endif


/*
** list an individual file
** (will be called by fget_list() or fget_recursion())
*/
static int
fget_list_file(FTP *ftp, char *filename, struct ftpstat *fsp, void *dummy)
{
#ifdef DEBUG
	printf("==> fget_list_file(filename=\"%s\")\n", filename);
#endif

	/* display "ls -l"-style output */
	ftp_long_ls(ftp, filename, fsp);

	/*
	** Instead of recursing into the subdirectory from here, we add
	** it to the need-to-visit list and let fget_list() take care of
	** it later.  The reason for this is that we don't want to recurse
	** down into subdirectories until we finish listing the
	** directory we're already in the middle of.
	*/
	if (BIT_ISSET(options, OPT_RECURSIVE)
	    && S_ISDIR(fsp->fs_mode))
	{
#ifdef DEBUG
		printf("    fget_list_file(): adding %s to visit list\n",
		       filename);
#endif
		if (record_need_to_visit(filename) == -1)
			return R_SKIP;
	}

	return R_FILEOK;
}


/*
** entry point for list mode
*/
int
fget_list(FTP* ftp, char *filename)
{
	struct ftpstat fs;
	int retval;

#ifdef DEBUG
	printf("==> fget_list(\"%s\")\n", filename);
#endif

	if (fget_stat(ftp, filename, &fs) != R_FILEOK)
		return R_RETURN;

	/* if we've already visited it, skip it */
	if (S_ISLNK(fs.fs_mode)
	    && already_visited(filename))
		return R_SKIP;

	if (verbose)
		printf("\nfget: listing %s...\n", filename);

	if (S_ISDIR(fs.fs_mode))
	{
		/*
		** This will call fget_list_file() for each file in the
		** directory.
		*/
		retval = fget_recursion(ftp, filename, fget_list_file, NULL);
	}
	else
	{
		/*
		** This is a file, not a directory, so we just call
		** fget_list_file() ourselves.
		*/
		retval = fget_list_file(ftp, filename, &fs, NULL);
 	}
 
	/*
	** If we're recursing, fget_list_file() will have populated the
	** need-to-visit list with subdirectories.
	**
	** If we're chasing links, fget_recursion() will have populated
	** the need-to-visit list with link targets.
	*/
	if (BIT_ISSET(options, OPT_RECURSIVE)
	    || BIT_ISSET(options, OPT_CHASELINKS))
	{
		char path[MAXPATHLEN];

		while (next_path_to_visit(path, sizeof(path)))
		{
			/* recursively call ourself */
			retval = fget_list(ftp, path);
			if (retval == R_RETURN)
				break;
		}
	}

#ifdef DEBUG
	printf("<== fget_list()\n");
#endif

	return retval;
}


