/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  download.c - FTP downloading code for fget
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>
#include <utime.h>
#include <sys/stat.h>
#include <time.h>

#ifdef STDC_HEADERS
# include <string.h>
#endif


/*
** download an individual file
** (will be called by fget_download() or fget_recursion())
*/
static int
fget_download_file(FTP *ftp, char *remotepath, struct ftpstat *fsp,
		   void *localtree)
{
	int i;
	char localpath[MAXPATHLEN], linkbuf[MAXPATHLEN];
	struct stat s;
	struct utimbuf utb;

#ifdef DEBUG
	printf("==> fget_download_file(remotepath=\"%s\", localtree=\"%s\")\n",
	       remotepath, (char *)localtree);
#endif

	/*
	** determine the local filename
	*/
	strlcpy(localpath, (char *)localtree, sizeof(localpath));
	if (strcmp(remotepath, "/") != 0)
	{
		if (localpath[0] != '\0')
			strlcat(localpath, "/", sizeof(localpath));
		strlcat(localpath, basename(remotepath), sizeof(localpath));
	}

	/*
	** stat the local file
	*/
	memset(&s, 0, sizeof(s));
	if (lstat(localpath, &s) == -1
	    && errno != ENOENT)
	{
		fprintf(stderr, "fget: lstat(\"%s\"): %s\n", localpath,
			strerror(errno));
		return R_ERR;
	}

	if (S_ISDIR(fsp->fs_mode))
	{
		/*
		** it's a directory, so create it locally
		*/
		i = fget_download_dir(ftp, remotepath, fsp, localpath, &s);
		if (IS_ERROR(i))
			return i;

		/*
		** if we're recursing, do so
		*/
		if (BIT_ISSET(options, OPT_RECURSIVE))
		{
			if (verbose)
				printf("fget: checking %s...\n", remotepath);
			if (fget_recursion(ftp, remotepath,
					   fget_download_file,
					   localpath) == R_RETURN)
				return R_RETURN;

			/*
			** delete any local files that do not exist on
			** the server for the directory we just downloaded
			*/
			if (BIT_ISSET(options, OPT_DELETELOCAL))
				fget_delete_local(ftp, remotepath, localpath);
		}
	}
	else if (S_ISLNK(fsp->fs_mode))
	{
		/*
		** if it's a symlink, create it locally
		*/
		i = fget_download_symlink(ftp, remotepath, fsp, localpath, &s,
					  linkbuf, sizeof(linkbuf));
		if (IS_ERROR(i))
			return i;
	}
	else if (S_ISREG(fsp->fs_mode))
	{
		/*
		** if it's a regular file, download it
		*/
		i = fget_download_regfile(ftp, remotepath, fsp, localpath, &s);
		if (IS_ERROR(i))
			return i;
	}
	else
	{
		/*
		** we don't handle sockets, pipes, etc
		*/
		fprintf(stderr, "fget: unsupported file type (mode == 0%6o)\n",
			(unsigned int)fsp->fs_mode);
		return R_ERR;
	}

	/*
	** update timestamp
	**
	** Note: we do this here instead of in fget_download_*(),
	** because if it's a directory, its timestamp has been modified
	** after it was created because we downloaded files into the
	** directory.  so we set it here, after we're done writing to it.
	*/
	if (! S_ISLNK(fsp->fs_mode))
	{
		utb.actime = time(NULL);
		utb.modtime = fsp->fs_mtime;
		if (utime(localpath, &utb) == -1)
		{
			fprintf(stderr, "fget: utime(\"%s\"): %s\n",
				localpath, strerror(errno));
			return R_ERR;
		}
	}

	/*
	** attempt to save permissions, if appropriate
	*/
	if (BIT_ISSET(options, OPT_SAVEPERMS))
		fget_set_perms(localpath, fsp);

	return R_FILEOK;
}


/*
** Canonifies paths for hostdirs, including following symlinks.
** Returns new remote path in the buffer pointed to by cwd.
*/
static int
fget_hostdirs_canonify(FTP *ftp, char *remotepath, char *root,
		       char *cwd, size_t cwdsize)
{
	char rpbuf[MAXPATHLEN], localpath[MAXPATHLEN];
	char tmp[MAXPATHLEN], linkbuf[MAXPATHLEN];
	int i;
	char *thisp, *nextp;
	struct ftpstat fs;
	struct stat s;

#ifdef DEBUG
	printf("==> fget_hostdirs_canonify(ftp=(%s), remotepath=\"%s\", "
	       "root=\"%s\")\n", ftp_get_host(ftp), remotepath, root);
#endif

	/*
	** if we're in the root directory and the remote path is not the
	** root directory, initialize cwd to an empty string, since
	** we're going to append a "/" in the loop below, and we don't
	** want the path to start with a double slash. 
	**
	** otherwise, initialize cwd to the current working directory.
	*/
	if (strcmp(ftp_getcwd(ftp), "/") == 0
	    && strcmp(remotepath, "/") != 0)
		cwd[0] = '\0';
	else
		strlcpy(cwd, ftp_getcwd(ftp), cwdsize);

	/*
	** iterate through components of remotepath
	*/
	strlcpy(rpbuf, remotepath, sizeof(rpbuf));
	nextp = rpbuf;
	while ((thisp = strsep(&nextp, "/")) != NULL)
	{
		if (*thisp == '\0')
			continue;

		/*
		** append component to cwd
		*/
		strlcat(cwd, "/", cwdsize);
		strlcat(cwd, thisp, cwdsize);
#ifdef DEBUG
		printf("    fget_hostdirs_canonify(): cwd = \"%s\"\n", cwd);
		printf("    fget_hostdirs_canonify(): nextp = \"%s\"\n", nextp);
#endif

		/*
		** stat the directory so far
		*/
		if (fget_stat(ftp, cwd, &fs) != R_FILEOK)
			return R_RETURN;

		/*
		** add new cwd to localpath
		*/
		strlcpy(localpath, root, sizeof(localpath));
		strlcat(localpath, cwd, sizeof(localpath));
#ifdef DEBUG
		printf("    fget_hostdirs_canonify(): localpath = \"%s\"\n",
		       localpath);
#endif

		/*
		** stat the local file
		*/
		memset(&s, 0, sizeof(s));
		if (lstat(localpath, &s) == -1
		    && errno != ENOENT)
		{
			fprintf(stderr, "fget: lstat(\"%s\"): %s\n",
				localpath, strerror(errno));
			return R_RETURN;
		}

		if (S_ISDIR(fs.fs_mode))
		{
			/*
			** if it's a directory, create it locally
			*/
			i = fget_download_dir(ftp, cwd, &fs, localpath, &s);
			if (IS_ERROR(i))
				return i;
		}
		else if (S_ISLNK(fs.fs_mode))
		{
			/*
			** if it's a symlink, create it locally
			*/
			i = fget_download_symlink(ftp, cwd, &fs, localpath,
						  &s, linkbuf, sizeof(linkbuf));
			if (IS_ERROR(i))
				return i;

			/*
			** redirect to where the link points to and call
			** ourselves recursively with that path
			*/
			snprintf(tmp, sizeof(tmp), "%s/%s",
				 dirname(cwd), linkbuf);
			if (nextp != NULL && *nextp != '\0')
			{
				strlcat(tmp, "/", sizeof(tmp));
				strlcat(tmp, nextp, sizeof(tmp));
			}
			if (fget_cleanpath(tmp, linkbuf, sizeof(linkbuf)) == -1)
			{
				fprintf(stderr,
					"fget: fget_cleanpath(\"%s\"): %s\n",
					tmp, strerror(errno));
				return R_ERR;
			}
			return fget_hostdirs_canonify(ftp, linkbuf, root, cwd,
						      cwdsize);
		}
		else if (nextp != NULL && *nextp != '\0')
		{
			/*
			** regular file - it's an error if there are any
			** path components left
			*/
			fprintf(stderr, "fget: %s: %s\n", cwd,
				strerror(ENOTDIR));
			return R_RETURN;
		}
	}

	return R_FILEOK;
}


/*
** entry point for download mode
*/
int
fget_download(FTP *ftp, char *remotepath, char *changeroot)
{
	struct ftpstat fs;
	char localroot[MAXPATHLEN] = ".", newpath[MAXPATHLEN];
	int retval = 0;
	char *cp;

#ifdef DEBUG
	printf("==> fget_download(ftp=(%s), remotepath=\"%s\", "
	       "changeroot=\"%s\")\n",
	       ftp_get_host(ftp), remotepath,
	       (changeroot ? changeroot : "NULL"));
#endif

	/*
	** update localroot
	*/

	/* if changeroot is given, use that instead of "." */
	if (changeroot)
		strlcpy(localroot, changeroot, sizeof(localroot));
	/* if we're using host dirs, add the hostname */
	if (BIT_ISSET(options, OPT_HOSTDIRS))
	{
		strlcat(localroot, "/", sizeof(localroot));
		strlcat(localroot, ftp_get_host(ftp), sizeof(localroot));
	}

	/*
	** create the local directory tree
	*/
	if (fget_mkdirhier(localroot, NULL, NULL) == -1)
	{
		fprintf(stderr, "fget: fget_mkdirhier(\"%s\"): %s\n",
			localroot, strerror(errno));
		return R_RETURN;
	}

	/*
	** set newpath to be the correct remote path name
	*/
	if (BIT_ISSET(options, OPT_HOSTDIRS))
	{
		/*
		** call fget_hostdirs_canonify() to determine the correct
		** remote path
		*/
		if (fget_hostdirs_canonify(ftp, remotepath,
					   localroot, newpath,
					   sizeof(newpath)) != R_FILEOK)
			return R_RETURN;

		/*
		** if we're using host dirs, append the remote path to
		** localroot
		** (don't append if the remote directory is "/", since
		** that would leave a trailing slash on localroot)
		*/
		if (strcmp(newpath, "/") != 0)
		{
			if (newpath[0] != '/')
				strlcat(localroot, "/", sizeof(localroot));
			strlcat(localroot, newpath, sizeof(localroot));
		}
	}
	else
	{
		/*
		** not using host dirs, so just find the canonified path
		*/
		if (ftp_realpath(ftp, remotepath, newpath,
				 sizeof(newpath)) == -1)
		{
			fprintf(stderr, "fget: ftp_realpath(\"%s\"): %s\n",
				remotepath, strerror(errno));
			if (errno == ECONNRESET)
				fget_disconnect_handle(ftp);
			return R_RETURN;
		}
	}

#ifdef DEBUG
	printf("    fget_download(): newpath = \"%s\"\n", newpath);
	printf("    fget_download(): localroot = \"%s\"\n", localroot);
#endif

	/*
	** stat newpath
	*/
	if (fget_stat(ftp, newpath, &fs) != R_FILEOK)
		return R_RETURN;

	if (S_ISDIR(fs.fs_mode))
	{
		/*
		** it's a directory
		*/
		if (! BIT_ISSET(options, OPT_HOSTDIRS))
		{
			/*
			** for host dirs, append remote path
			*/
			if (! BIT_ISSET(options, OPT_STRIPDIR)
			    && strcmp(basename(remotepath), "/") != 0)
			{
				strlcat(localroot, "/", sizeof(localroot));
				strlcat(localroot, basename(remotepath),
					sizeof(localroot));
			}

			/*
			** create local directory tree
			*/
			if (fget_mkdirhier(localroot, NULL, NULL) == -1)
			{
				fprintf(stderr,
					"fget: fget_mkdirhier(\"%s\"): %s\n",
					localroot, strerror(errno));
				return R_RETURN;
			}
		}

		/*
		** process directory
		*/
		retval = fget_recursion(ftp, newpath, fget_download_file,
					localroot);

		/*
		** delete local files that don't exist on the server
		*/
		if (retval != R_RETURN
		    && BIT_ISSET(options, OPT_DELETELOCAL))
			fget_delete_local(ftp, newpath, localroot);
	}
	else
	{
		/*
		** not a directory - strip off last local path component
		*/
		if (BIT_ISSET(options, OPT_HOSTDIRS)
		    && (cp = strrchr(localroot, '/')) != NULL)
			*cp = '\0';

		/*
		** download the file
		*/
		retval = fget_download_file(ftp, newpath, &fs, localroot);
	}

	/*
	** if we're chasing links, fget_recursion() will have populated
	** the need-to-visit list with link targets.
	*/
	if (BIT_ISSET(options, OPT_CHASELINKS)
	    && retval != R_RETURN)
	{
		char path[MAXPATHLEN];

		while (next_path_to_visit(path, sizeof(path)))
		{
			if ((BIT_ISSET(options, OPT_HOSTDIRS)
			     || strncmp(path, newpath,
					strlen(newpath)) == 0)
			    && ! already_visited(path)
			    && fget_download(ftp, path,
					     changeroot) == R_RETURN)
				break;
		}
	}

	return retval;
}


