/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  du.c - FTP disk usage code
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>

#ifdef STDC_HEADERS
# include <string.h>
#endif

#ifdef HAVE_INTTYPES_H
# include <inttypes.h>
#endif


/*
** get size of specified file
** (will be called by fget_recursion())
*/
static int
fget_du_file(FTP *ftp, char *filename, struct ftpstat *fsp, void *total)
{
	uint64_t my_total = 0;

#ifdef DEBUG
	printf("==> fget_du_file(filename=\"%s\", total=%lu)\n",
	       filename, *(unsigned long *)total);
#endif

#ifdef DEBUG
	ftp_long_ls(ftp, filename, fsp);
#endif

	/*
	** add the file's size to our total
	*/
	*((uint64_t *)total) += fsp->fs_size;

	/*
	** handle recursion
	*/
	if (BIT_ISSET(options, OPT_RECURSIVE) && S_ISDIR(fsp->fs_mode))
	{
		/*
		** this will call us for every file in the subdirectory
		*/
		if (fget_recursion(ftp, filename, fget_du_file,
				   &my_total) == R_RETURN)
			return R_RETURN;

		/*
		** print the total for the subdir we just finished,
		** and then add it to the total for the current dir
		*/
		printf("\t%10lld  %s\n", (my_total / 1024), filename);
		*((uint64_t *)total) += my_total;
	}

	return R_FILEOK;
}


/*
** main entry point for du mode
*/
int
fget_du(FTP *ftp, char *filename)
{
	uint64_t total = 0;
	struct ftpstat fs;
	int retval;
	char path[MAXPATHLEN];

	putchar('\n');

	/*
	** start recursion
	*/
	if (fget_stat(ftp, filename, &fs) != R_FILEOK)
		return R_RETURN;
	retval = fget_du_file(ftp, filename, &fs, &total);

	/*
	** if we're chasing links, fget_recursion() will have populated
	** the need-to-visit list with link targets.
	*/
	if (BIT_ISSET(options, OPT_CHASELINKS))
	{
		while (next_path_to_visit(path, sizeof(path)))
		{
			if (! already_visited(path))
			{
				/*
				** start recursion
				*/
				retval = fget_stat(ftp, path, &fs);
				if (retval == R_FILEOK)
					retval = fget_du_file(ftp, path,
							      &fs, &total);

				if (retval == R_RETURN)
					break;
			}
		}
	}

	/*
	** print total usage
	*/
	if (retval == R_FILEOK)
	{
		printf("\t%10lld  %s\n\n", (total / 1024), filename);
		printf("fget: total is %lld Kbytes\n", (total / 1024));
	}

	return retval;
}


