/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2004 Mark D. Roth
**  All rights reserved.
**
**  local.c - local file manipulation code for fget
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <errno.h>
#include <utime.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <pwd.h>
#include <grp.h>

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif


/*
** attempt to preserve owner, group, and mode
*/
void
fget_set_perms(char *localpath, struct ftpstat *fsp)
{
	static fget_list_t *warned_users_l = NULL, *warned_groups_l = NULL;
	fget_listptr_t lp;
	struct passwd *pwd = NULL;
	struct group *grp = NULL;

	/*
	** warn once for each user or group that does not exist locally
	*/
	if (warned_users_l == NULL)
		warned_users_l = fget_list_new(LIST_QUEUE, NULL);
	if (warned_groups_l == NULL)
		warned_groups_l = fget_list_new(LIST_QUEUE, NULL);

	/*
	** if it's not a symlink, change permissions
	*/
	if (! S_ISLNK(fsp->fs_mode))
	{
		if (chmod(localpath, fsp->fs_mode) == -1)
			fprintf(stderr,
				"fget: %s: cannot change permissions: %s\n",
				localpath, strerror(errno));
	}

	/*
	** warn if user doesn't exist
	*/
	fget_listptr_reset(&lp);
	if (fget_list_search(warned_users_l, &lp, fsp->fs_username, NULL) == 0)
	{
		pwd = getpwnam(fsp->fs_username);
		if (pwd == NULL && verbose > 1)
		{
			fprintf(stderr, "fget: warning: no local account "
				"for user \"%s\"\n", fsp->fs_username);
			fget_list_add(warned_users_l,
				      strdup(fsp->fs_username));
		}
	}

	/*
	** warn if group doesn't exist
	*/
	fget_listptr_reset(&lp);
	if (fget_list_search(warned_groups_l, &lp,
			     fsp->fs_groupname, NULL) == 0)
	{
		grp = getgrnam(fsp->fs_groupname);
		if (grp == NULL && verbose > 1)
		{
			fprintf(stderr, "fget: warning: no local account "
				"for group \"%s\"\n", fsp->fs_groupname);
			fget_list_add(warned_groups_l,
				      strdup(fsp->fs_groupname));
		}
	}

	/*
	** update user and group
	*/
	if (pwd != NULL || grp != NULL)
	{
		if (
#ifdef HAVE_LCHOWN
			   lchown
#else
			   chown
#endif
			   (localpath, (pwd != NULL ? pwd->pw_uid : -1),
			    (grp != NULL ? grp->gr_gid : -1)) == -1)
			fprintf(stderr,
				"fget: %s: cannot change owner or group: %s\n",
				localpath, strerror(errno));
	}
}


/*
** run extendfs command, if applicable
**
** returns 0 on success, -1 on error
*/
static int
fget_extendfs(void)
{
	int status;

	/* no extendfs command given */
	if (extendfs_command == NULL)
		return -1;

	printf("\nfget: executing command: \"%s\"\n",
	       extendfs_command);
	status = system(extendfs_command);

	if (WIFEXITED(status))
	{
		/* success */
		if (WEXITSTATUS(status) == 0)
			return 0;

		fprintf(stderr,
			"fget: system(): child exited with status %d\n",
			WEXITSTATUS(status));
	}
	else /* if (WIFSIGNALED(status)) */
	{
		fprintf(stderr,
			"fget: system(): child died with signal %d\n",
			WTERMSIG(status));
	}

	return -1;
}


/*
** create a local directory corresponding to a directory on the server
*/
int
fget_download_dir(FTP* ftp, char *remotepath, struct ftpstat *fsp,
		  char *localpath, struct stat *sp)
{
	int try_extendfs;

#ifdef DEBUG
	printf("==> fget_download_dir(remotepath=\"%s\", localpath=\"%s\")\n",
	       remotepath, localpath);
#endif

	/*
	** directory already exists - no action needed
	*/
	if (S_ISDIR(sp->st_mode))
	{
		if (verbose > 1)
			printf("fget: directory already exists: %s\n",
			       localpath);
		return R_SKIP;
	}

	/*
	** if it exists but it's not a directory, nuke it
	*/
	if (sp->st_mode != 0)
	{
		if (fget_unlink(localpath, sp) == -1)
			return R_ERR;
	}

	/*
	** it doesn't exist, so we need to create it
	*/
	if (verbose)
		printf("fget: creating directory: %s\n", localpath);

	/*
	** If we get ENOSPC, try to run the extendfs command.  If we get
	** ENOSPC again, the command obviously didn't help, so trying to
	** run it again will just get us into an infinite loop.
	** Instead, just return an error.
	*/
	for (try_extendfs = 1; try_extendfs >= 0; try_extendfs--)
	{
		if (mkdir(localpath, 0755) == -1)
		{
			if (try_extendfs && (errno == ENOSPC))
			{
				/* run extendfs command and retry */
				if (fget_extendfs() == 0)
					continue;
			}

			/*
			** either it's not ENOSPC, or we've already
			** tried running the extendfs command and it
			** didn't help
			*/
			fprintf(stderr, "fget: mkdir(\"%s\"): %s\n",
				localpath, strerror(errno));

			/*
			** no point in trying any further operations if
			** we're out of local disk space
			*/
			if (errno == ENOSPC)
				return R_RETURN;

			return R_ERR;
		}

		/*
		** successfully created directory
		*/
		break;
	}

	return R_FILEOK;
}


/*
** create a local symlink corresponding to a symlink on the server
*/
int
fget_download_symlink(FTP* ftp, char *remotepath, struct ftpstat *fsp,
		      char *localpath, struct stat *sp,
		      char *linkto, size_t ltlen)
{
	int try_extendfs;
	char linkbuf[MAXPATHLEN];

	/*
	** see what the link on the server points to
	*/
	if (ftp_readlink(ftp, remotepath, linkbuf, sizeof(linkbuf)) == -1)
	{
		fprintf(stderr, "fget: ftp_readlink(\"%s\"): %s\n",
			remotepath, strerror(errno));
		return R_ERR;
	}

	/*
	** if we're fixing links, adjust absolute paths accordingly;
	** otherwise, copy the link target as-is
	*/
	if (BIT_ISSET(options, OPT_FIXLINKS) && linkbuf[0] == '/')
	{
		if (fget_relativepath(dirname(remotepath), linkbuf,
				      linkto, ltlen) == -1)
		{
			fprintf(stderr, "fget: fget_relativepath(): %s\n",
				strerror(errno));
			return R_ERR;
		}
	}
	else
		strcpy(linkto, linkbuf);

	/*
	** check if the link already exists
	*/
	if (S_ISLNK(sp->st_mode))
	{
		/*
		** see what the local link points to
		*/
		if (readlink(localpath, linkbuf, sizeof(linkbuf)) == -1)
		{
			fprintf(stderr, "fget: readlink(\"%s\"): %s\n",
				localpath, strerror(errno));
			return R_ERR;
		}
		linkbuf[sizeof(linkbuf) - 1] = '\0';

		/*
		** if it already points to the right place, we're done
		*/
		if (strcmp(linkto, linkbuf) == 0)
		{
			if (verbose > 1)
				printf("fget: symlink already exists: %s\n",
				       localpath);
			return R_SKIP;
		}
	}

	/*
	** local file is not a link, or link points to the wrong place.
	** nuke it and recreate it.
	*/
	if (sp->st_mode != 0)
	{
		if (fget_unlink(localpath, sp) == -1)
			return R_ERR;
	}

	if (verbose)
		printf("fget: creating symlink: %s -> %s\n", localpath,
		       linkto);

	/*
	** If we get ENOSPC, try to run the extendfs command.  If we get
	** ENOSPC again, the command obviously didn't help, so trying to
	** run it again will just get us into an infinite loop.
	** Instead, just return an error.
	*/
	for (try_extendfs = 1; try_extendfs >= 0; try_extendfs--)
	{
		if (symlink(linkto, localpath) == -1)
		{
			if (try_extendfs && (errno == ENOSPC))
			{
				/* run extendfs command and retry */
				if (fget_extendfs() == 0)
					continue;
			}

			/*
			** either it's not ENOSPC, or we've already
			** tried running the extendfs command and it
			** didn't help
			*/
			fprintf(stderr, "fget: symlink(\"%s\", \"%s\"): %s\n",
				linkto, localpath, strerror(errno));

			/*
			** no point in trying any further operations if
			** we're out of local disk space
			*/
			if (errno == ENOSPC)
				return R_RETURN;

			return R_ERR;
		}

		/*
		** successfully created symlink
		*/
		break;
	}

	return R_FILEOK;
}


/*
** download a regular file from the server
*/
int
fget_download_regfile(FTP *ftp, char *remotepath, struct ftpstat *fsp,
		      char *localpath, struct stat *sp)
{
	char readbuf[FTPBUFSIZE];
	struct timeval before, after;
	float elapsed;
	int i, ofd = -1, try_extendfs;
	size_t sz;
	off_t bytecount = 0, nextreport = 0, interval = 10240;
	FTPFILE *ftpfile = NULL;

	/*
	** If the file exists, and it's older than the file on the
	** server, and it's the same size as the file on the server,
	** then we don't have anything more to do.
	*/
	if (sp->st_mode != 0
	    && (! BIT_ISSET(options, OPT_NEWER)
	        || sp->st_mtime >= fsp->fs_mtime)
	    && (! BIT_ISSET(options, OPT_DIFFSIZE)
		|| sp->st_size == fsp->fs_size))
	{
		if (verbose > 1)
			printf("fget: not updating file %s\n", localpath);
		return R_SKIP;
	}

	/*
	** otherwise, we need to download the file
	*/
	if (verbose)
		printf("fget: downloading: %s => %s...\n", remotepath,
		       localpath);

	/*
	** open the file on the FTP server
	*/
	if (ftp_open(&ftpfile, ftp, remotepath, O_RDONLY) == -1)
	{
		fprintf(stderr, "fget: ftp_open(\"%s\"): %s\n", remotepath,
			strerror(errno));
		return R_ERR;
	}

	/*
	** If we get ENOSPC, try to run the extendfs command.  If we get
	** ENOSPC again, the command obviously didn't help, so trying to
	** run it again will just get us into an infinite loop.
	** Instead, just return an error.
	*/
	for (try_extendfs = 1; try_extendfs >= 0; try_extendfs--)
	{
		/*
		** open the local file
		*/
		ofd = open(localpath, O_WRONLY | O_CREAT | O_TRUNC, 0644);
		if (ofd == -1)
		{
			if (try_extendfs && errno == ENOSPC)
			{
				/* run extendfs command and retry */
				if (fget_extendfs() == 0)
					continue;
			}

			/*
			** either it's not ENOSPC, or we've already
			** tried running the extendfs command and it
			** didn't help
			*/
			fprintf(stderr, "fget: open(\"%s\"): %s\n", localpath,
				strerror(errno));

			/* close the file on the FTP server */
			i = errno;
			if (ftp_close(ftpfile) == -1)
				fprintf(stderr, "fget: ftp_close(): %s\n",
					strerror(errno));
			errno = i;

			/*
			** no point in trying any further operations if
			** we're out of local disk space
			*/
			if (errno == ENOSPC)
				return R_RETURN;

			return R_ERR;
		}

		/*
		** successfully opened file
		*/
		break;
	}

	elapsed = 0;
	gettimeofday(&before, NULL);
	while ((sz = ftp_read(ftpfile, readbuf, sizeof(readbuf))) > 0)
	{
		/*
		** If we get ENOSPC, try to run the extendfs command.  If we
		** get ENOSPC again, the command obviously didn't help, so
		** trying to run it again will just get us into an infinite
		** loop.  Instead, just return an error.
		*/
		for (try_extendfs = 1; try_extendfs >= 0; try_extendfs--)
		{
			/*
			** write to the local file
			*/
			if (write(ofd, readbuf, sz) == -1)
			{
				if (try_extendfs && errno == ENOSPC)
				{
					/*
					** run extendfs command and retry.
					**
					** the time spent running the
					** extendfs command should not
					** count towards the download,
					** so we update elapsed time
					** beforehand and reset the
					** start time afterwards.
					*/
					gettimeofday(&after, NULL);
					elapsed +=
					    (after.tv_sec - before.tv_sec) +
					    ((after.tv_usec -
					      before.tv_usec) / 1000000.0);
					if (fget_extendfs() == 0)
					{
						gettimeofday(&before, NULL);
						continue;
					}
				}

				/*
				** either it's not ENOSPC, or we've already
				** tried running the extendfs command and it
				** didn't help
				*/
				fprintf(stderr, "fget: write(): %s\n",
					strerror(errno));

				/* close the files */
				i = errno;
				if (close(ofd) == -1)
					fprintf(stderr,
						"fget: close(): %s\n",
						strerror(errno));
				if (ftp_close(ftpfile) == -1)
					fprintf(stderr,
						"fget: ftp_close(): %s\n",
						strerror(errno));
				errno = i;

				/*
				** no point in trying any further operations
				** if we're out of local disk space
				*/
				if (errno == ENOSPC)
					return R_RETURN;

				return R_ERR;
			}

			/*
			** successfully wrote to file
			*/
			break;
		}

		/* print progress indicator */
		bytecount += sz;
		if (verbose
		    && isatty(fileno(stdout))
		    && bytecount > nextreport)
		{
			printf("    received %lu/%lu bytes\r",
			       (unsigned long)bytecount,
			       (unsigned long)fsp->fs_size);
			fflush(stdout);
			nextreport += interval;
		}
	}

	/* print download stats */
	gettimeofday(&after, NULL);
	elapsed += (after.tv_sec - before.tv_sec) +
	    ((after.tv_usec - before.tv_usec) / 1000000.0);
	if (verbose)
		printf("    received %lu bytes in %.2f seconds (%.2f K/s)\n",
		       (unsigned long)bytecount, elapsed,
		       (((float)bytecount / 1024.0) / elapsed));

	/* close FTP file */
	if (ftp_close(ftpfile) == -1)
	{
		fprintf(stderr, "fget: ftp_close(): %s\n", strerror(errno));
		return R_ERR;
	}

	/* close local file */
	if (close(ofd) == -1)
	{
		fprintf(stderr, "fget: close(): %s\n", strerror(errno));
		return R_ERR;
	}

	/* success */
	return R_FILEOK;
}


