/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  recursion.c - FTP remote directory recursion code
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <sys/param.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif


/* 
** stat an FTP file.
*/
int
fget_stat(FTP *ftp, char *filename, struct ftpstat *fsp)
{
#ifdef DEBUG
	printf("==> fget_stat(ftp=(%s), filename=\"%s\", fsp=0x%lx)\n",
	       ftp_get_host(ftp), filename, fsp);
#endif

	/*
	** if we're dereferencing links, just use ftp_stat() and be done
	*/
	if (BIT_ISSET(options, OPT_DEREFLINKS))
	{
		if (ftp_stat(ftp, filename, fsp) == -1)
		{
			fprintf(stderr, "fget: ftp_stat(\"%s\"): %s\n",
				filename, strerror(errno));
			if (errno == ECONNRESET)
			{
#if 0
				fget_disconnect_handle(ftp);
#endif
				return R_RETURN;
			}
			return R_SKIP;
		}
		return R_FILEOK;
	}

	/*
	** not deferencing links, so we use ftp_lstat()
	*/
	if (ftp_lstat(ftp, filename, fsp) == -1)
	{
		fprintf(stderr, "fget: ftp_lstat(\"%s\"): %s\n",
			filename, strerror(errno));
		if (errno == ECONNRESET)
		{
#if 0
			fget_disconnect_handle(ftp);
#endif
			return R_RETURN;
		}
		return R_SKIP;
	}

	/*
	** chase symlinks if requested
	*/
	if (S_ISLNK(fsp->fs_mode)
	    && BIT_ISSET(options, OPT_CHASELINKS))
	{
		char link_target[MAXPATHLEN];
		fget_listptr_t lp;

		if (ftp_readlink(ftp, filename, link_target,
				 sizeof(link_target)) == -1)
		{
			fprintf(stderr, "fget: ftp_readlink(\"%s\"): %s\n",
				filename, strerror(errno));
			return R_SKIP;
		}

		/* canonify relative paths */
		if (link_target[0] != '/')
		{
			char buf[MAXPATHLEN];

			snprintf(buf, sizeof(buf), "%s/%s",
				 dirname(filename), link_target);
			if (fget_cleanpath(buf, link_target,
					   sizeof(link_target)) == -1)
			{
				fprintf(stderr,
					"fget: fget_cleanpath(\"%s\"): %s\n",
					buf, strerror(errno));
				return R_SKIP;
			}
		}

		/* see if we've already visited the file */
		if (! already_visited(link_target)
		    && record_need_to_visit(link_target) == -1)
			return R_SKIP;

#ifdef DEBUG
		print_need_to_visit_list();
#endif
	}

	return R_FILEOK;
}


/*
** core directory recursion.
*/
int
fget_recursion(FTP *ftp, char *dir, fget_plugin file_func, void *ptr)
{
	FTPDIR *ftpdir;
	struct ftpdirent fde;
	char filepath[MAXPATHLEN];
	struct ftpstat fs;

#ifdef DEBUG
	printf("==> fget_recursion(\"%s\", 0x%lx)\n", dir, file_func);
#endif

	if (ftp_opendir(&ftpdir, ftp, dir) == -1)
	{
		fprintf(stderr, "fget: ftp_opendir(\"%s\"): %s\n", dir,
			strerror(errno));
		if (errno == ECONNRESET)
		{
			fget_disconnect_handle(ftp);
			return R_RETURN;
		}
		return R_SKIP;
	}

	/* iterate through each entry in the directory */
	while (ftp_readdir(ftpdir, &fde) == 1)
	{
		/* skip "." and ".." entries */
		if (strcmp(fde.fd_name, ".") == 0
		    || strcmp(fde.fd_name, "..") == 0)
			continue;

		/* generate file path */
		snprintf(filepath, sizeof(filepath), "%s%s%s",
			 dir, (strcmp(dir, "/") == 0 ? "" : "/"),
			 fde.fd_name);

		switch (fget_stat(ftp, filepath, &fs))
		{
		case R_SKIP:
			continue;
		case R_RETURN:
			return R_RETURN;
		default:
			break;
		}

		/* call file_func */
		if ((*file_func)(ftp, filepath, &fs, ptr) == R_RETURN)
		{
			ftp_closedir(ftpdir);
			return R_RETURN;
		}

		record_visit(dir);
	}

	/* don't bother with error checking */
	ftp_closedir(ftpdir);
#ifdef DEBUG
	printf("<== fget_recursion(): R_FILEOK\n");
#endif

	return R_FILEOK;
}


