/*
**  Copyright 2000-2005 Mark D. Roth
**  All rights reserved.
**
**  visit.c - FTP remote directory visit lists
**
**  Mark D. Roth <roth@feep.net>
*/

#include <fget.h>

#include <stdio.h>
#include <sys/param.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif


/*
** TODO(roth):
** Globals... yuck.  If this were library code, I'd shoot myself.
*/

/* a list of paths that we still need to visit */
static fget_list_t *need_to_visit_l = NULL;

/* a list of paths we've already visited */
static fget_list_t *already_visited_l = NULL;


/*
** initialize lists
** returns 0 on success, -1 on error
*/
int
init_visit_lists(void)
{
	need_to_visit_l = fget_list_new(LIST_QUEUE, NULL);
	if (need_to_visit_l == NULL)
		return -1;
	already_visited_l = fget_list_new(LIST_QUEUE, NULL);
	if (already_visited_l == NULL)
		return -1;
	return 0;
}


/* free lists */
void
free_visit_lists(void)
{
	fget_list_free(need_to_visit_l, free);
	need_to_visit_l = NULL;
	fget_list_free(already_visited_l, free);
	already_visited_l = NULL;
}


/*
** matching function for files in the same tree
**
** if donefile matches the leading path components of checkfile,
** it's a match.
*/
static int
subdir_match(char *checkfile, char *donefile)
{
	int retval = (strncmp(checkfile, donefile, strlen(donefile)) == 0 &&
		      (checkfile[strlen(donefile)] == '/' ||
		       checkfile[strlen(donefile)] == '\0'));
#ifdef DEBUG
	printf("=== subdir_match(\"%s\", \"%s\") == %s\n",
	       checkfile, donefile, retval ? "TRUE" : "FALSE");
#endif
	return retval;
}


/* returns 1 if we've already visited the path, 0 otherwise */
int
already_visited(char *path)
{
	fget_listptr_t lp;
	fget_matchfunc_t match_func = NULL;

	if (BIT_ISSET(options, OPT_RECURSIVE))
		match_func = (fget_matchfunc_t)subdir_match;

	fget_listptr_reset(&lp);
	return fget_list_search(already_visited_l, &lp, path, match_func);
}


/* mark a path as visited */
int
record_visit(char *path)
{
	if (fget_list_add(already_visited_l, strdup(path)) == -1)
	{
		fprintf(stderr, "fget: fget_list_add(\"%s\"): %s\n",
			path, strerror(errno));
		return -1;
	}

	return 0;
}


/* record the fact that we need to visit a path */
int
record_need_to_visit(char *path)
{
	fget_listptr_t lp;

	/* already in list - don't re-add it */
	fget_listptr_reset(&lp);
	if (fget_list_search(need_to_visit_l, &lp, path, NULL) == 1)
		return 0;

	if (fget_list_add(need_to_visit_l, strdup(path)) == -1)
	{
		fprintf(stderr, "fget: fget_list_add(\"%s\"): %s\n",
			path, strerror(errno));
		return -1;
	}

	return 0;
}


/* return the next path we need to visit */
int
next_path_to_visit(char *buf, size_t buflen)
{
	char *path;
	fget_listptr_t lp;

	fget_listptr_reset(&lp);
	if (fget_list_next(need_to_visit_l, &lp) == 0)
		return 0;

	path = (char *)fget_listptr_data(&lp);
	strlcpy(buf, path, sizeof(buf));
	free(path);
	fget_list_del(need_to_visit_l, &lp);

	return 1;
}


#ifdef DEBUG
void
print_need_to_visit_list(void)
{
	fget_listptr_t lp;

	printf("##### printing chase list:\n");
	fget_listptr_reset(&lp);
	while (fget_list_next(need_to_visit_l, &lp) != 0)
		printf("#####\t%s\n",
		       (char *)fget_listptr_data(&lp));
}
#endif


