/*
**  Copyright 2000-2004 University of Illinois Board of Trustees
**  Copyright 2000-2004 Mark D. Roth
**  All rights reserved.
**
**  libfget.h - header file for libfget
**
**  Mark D. Roth <roth@feep.net>
*/

#ifndef LIBFGET_H
#define LIBFGET_H

#include <netdb.h>
#include <sys/types.h>
#include <sys/param.h>
#include <fcntl.h>


#ifndef MAXUSERNAMELEN
# define MAXUSERNAMELEN		16	/* max length of user name */
#endif

#ifndef MAXGROUPNAMELEN
# define MAXGROUPNAMELEN	8	/* max length of group name */
#endif


extern const char libfget_version[];

extern const int libfget_is_thread_safe;

typedef struct ftp FTP;


/***** ftpstat.c *************************************************************/

struct ftpstat
{
	mode_t fs_mode;				/* file mode */
	nlink_t fs_nlink;			/* number of links */
	char fs_username[MAXUSERNAMELEN];	/* user name */
	char fs_groupname[MAXGROUPNAMELEN];	/* group name */
	off_t fs_size;				/* size */
	time_t fs_mtime;			/* date */
	dev_t fs_rdev;				/* device ID
						   (for char or block devs) */
};

/* stat a symlink */
int ftp_lstat(FTP *, char *, struct ftpstat *);

/* stat a file */
int ftp_stat(FTP *, char *, struct ftpstat *);

/* read the contents of a symbolic link */
int ftp_readlink(FTP *, char *, char *, size_t);

/* resolve symlinks */
int ftp_realpath(FTP *, char *, char *, size_t);


/***** options.c *************************************************************/

/* options to pass ftp_connect(), ftp_set_options(), and ftp_get_options() */
enum ftp_options {
	FTP_OPT_PASSIVE		= 1,	/* flag for using PASV mode */
	FTP_OPT_USE_MLST,		/* flag for using MLST/MLSD
					   (if supported by server) */
	FTP_OPT_USE_ABOR,		/* use ABOR command */
	FTP_OPT_IO_TIMEOUT,		/* I/O timeout */
	FTP_OPT_CACHE_MAXSIZE,		/* max size for directory cache */
	FTP_OPT_CACHE_EXPIRE,		/* expire time for directory cache */
	FTP_OPT_SEND_HOOK,		/* send hook */
	FTP_OPT_RECV_HOOK,		/* receive hook */
	FTP_OPT_HOOK_DATA		/* send & receive hook data ptr */
};

/*
** ftp_get_option() - get an option from the FTP handle
*/
void ftp_get_options(FTP *, ...);

/*
** ftp_set_option() - set an option in the FTP handle
*/
void ftp_set_options(FTP *, ...);


/***** handle.c **************************************************************/

typedef void (*ftp_hookfunc_t)(char *, FTP *, void *);

/*
** ftp_connect() - open a control connection to the FTP server
** returns:
**	0				success
**	-1 (and sets errno)		failure
*/
int ftp_connect(FTP **, char *, char *, size_t, unsigned short, ...);

/* values for ftp_connect() flags bitmask */
#define FTP_CONNECT_DNS_RR	1	/* grok round-robin DNS entries */

/*
** ftp_login() - login to the FTP server
** returns:
**	0				success
**	-1 (and sets errno)		failure
*/
int ftp_login(FTP *, char *, char *);

/*
** ftp_quit() - disconnect from FTP server
** returns:
**	0				success
**	-1 (and sets errno)		failure
*/
int ftp_quit(FTP *, unsigned short);

/* flags for ftp_quit() */
#define FTP_QUIT_FAST		1	/* close without sending "QUIT" */

/*
** ftp_fd() - return file descriptor associated with FTP control connection
** returns:
**	file descriptor of ftp control connection
*/
int ftp_fd(FTP *);

/*
** ftp_data_fd() - return file descriptor associated with FTP data connection
** returns:
**	file descriptor of ftp data connection
*/
int ftp_data_fd(FTP *);

/*
** ftp_get_host() - get the hostname from the FTP handle
*/
char *ftp_get_host(FTP *);

/*
** ftp_whoami() - get the name of the logged-in user from the FTP handle
*/
char *ftp_whoami(FTP *);


/***** ftpdata.c *************************************************************/

/* values for ftp_type() argument */
#define FTP_TYPE_ASCII			"A"	/* ASCII mode */
#define FTP_TYPE_BINARY			"I"	/* binary (image) mode */

/* set TYPE for data connection */
int ftp_type(FTP *, char *);


/***** ftpdir.c **************************************************************/

typedef struct ftpdir FTPDIR;

struct ftpdirent
{
	char *fd_name;		/* file name */
};


/* open directory */
int ftp_opendir(FTPDIR **, FTP *, char *);

/* read directory */
int ftp_readdir(FTPDIR *, struct ftpdirent *);

/* rewind directory */
void ftp_rewinddir(FTPDIR *);

/* tell where we are in the directory */
off_t ftp_telldir(FTPDIR *);

/* seek to a given point in the directory */
void ftp_seekdir(FTPDIR *, off_t);

/* close directory */
void ftp_closedir(FTPDIR *);


/***** ftpfile.c *************************************************************/

typedef struct ftpfile FTPFILE;


/* open a file via FTP */
int ftp_open(FTPFILE **, FTP *, char *, int);

/* read a file via FTP */
ssize_t ftp_read(FTPFILE *, char *, size_t);

/* write a file via FTP */
ssize_t ftp_write(FTPFILE *, char *, size_t);

/* seek to a given offset in an FTP file */
off_t ftp_lseek(FTPFILE *, off_t, int);

/* close an FTP file */
int ftp_close(FTPFILE *);


/***** ftpglob.c *************************************************************/

typedef struct
{
	int gl_pathc;		/* Count of total paths so far. */
	int gl_matchc;		/* Count of paths matching pattern. */
	int gl_offs;		/* Reserved at beginning of gl_pathv. */
	int gl_flags;		/* Copy of flags parameter to glob. */
	char **gl_pathv;	/* List of paths matching pattern. */
	int (*gl_errfunc)(const char *, int);
				/* Copy of errfunc parameter to glob. */
}
ftp_glob_t;

/* Flags */
# define FTPGLOB_APPEND	0x0001	/* Append to output from previous call. */
# define FTPGLOB_DOOFFS	0x0002	/* Use gl_offs. */
# define FTPGLOB_ERR	0x0004	/* Return on error. */
# define FTPGLOB_MARK	0x0008	/* Append / to matching directories. */
# define FTPGLOB_NOCHECK 0x0010	/* Return pattern itself if nothing matches. */
# define FTPGLOB_NOSORT	0x0020	/* Don't sort. */
# define FTPGLOB_BRACE	0x0080	/* Expand braces ala csh. */
# define FTPGLOB_MAGCHAR 0x0100	/* Pattern had globbing characters. */
# define FTPGLOB_NOMAGIC 0x0200	/* FTPGLOB_NOCHECK without magic chars (csh). */
# define FTPGLOB_NOESCAPE 0x1000	/* Disable backslash escaping. */

/* Error values returned by glob(3) */
# define FTPGLOB_NOSPACE (-1)	/* Malloc call failed. */
# define FTPGLOB_ABORTED (-2)	/* Unignored error. */
# define FTPGLOB_NOMATCH (-3)	/* No match and FTPGLOB_NOCHECK not set. */


/*
** ftp_glob() - generate pathnames of remote files matching a pattern
** returns:
**	0			success
**	FTPGLOB_NOSPACE		malloc() error
**	FTPGLOB_ABORTED		error (and sets errno)
**	FTPGLOB_NOMATCH		no match and FTPGLOB_NOCHECK not set
*/
int ftp_glob(FTP *, const char *, int,
	     int (*)(const char *, int), ftp_glob_t *);

/*
** ftp_globfree() - free memory allocated by ftp_glob()
*/
void ftp_globfree(ftp_glob_t *);


/***** ftpops.c **************************************************************/

/* rename a file */
int ftp_rename(FTP *, char *, char *);

/* remove a file */
int ftp_unlink(FTP *, char *);

/* remove a file or directory */
int ftp_remove(FTP *, char *);

/* determine the remote system type */
char *ftp_systype(FTP *);

/* get server status */
int ftp_status(FTP *, char *, size_t);

/* change directory */
int ftp_chdir(FTP *, char *);

/* make a directory */
int ftp_mkdir(FTP *, char *);

/* remove a directory */
int ftp_rmdir(FTP *, char *);

/* get current directory */
char *ftp_getcwd(FTP *);


/***** ftpsite.c *************************************************************/

/* execute a site command */
int ftp_site(FTP *, char *, ...);

/* execute a site command that opens a (pseudo)file */
int ftp_site_open(FTPFILE **, FTP *, int, char *, ...);


/***** url.c *****************************************************************/

#define MAXPASSLEN              64
#define MAXURLLEN               (MAXHOSTNAMELEN + MAXUSERNAMELEN + \
                                 MAXPASSLEN + MAXPATHLEN + 9)

struct ftp_url
{
	char fu_hostname[MAXHOSTNAMELEN];
	char fu_login[MAXUSERNAMELEN];
	char fu_passwd[MAXPASSLEN];
	char fu_path[MAXPATHLEN];
	char fu_type;
};


/* parse RFC-1738 style URL and fill in caller-supplied fields */
int ftp_url_parse(char *, struct ftp_url *);


#endif /* ! LIBFGET_H */
